/////////////////////////////////////////////////////////////////////
////                                                             ////
////  FMUL                                                       ////
////  Floating Point Multiply Unit (Single precision)            ////
////                                                             ////
////  Author: Rudolf Usselmann                                   ////
////          russelmann@hotmail.com                             ////
////                                                             ////
/////////////////////////////////////////////////////////////////////
////                                                             ////
//// Copyright (C) 2000 Rudolf Usselmann                         ////
////                    russelmann@hotmail.com                   ////
////                                                             ////
//// This source file may be used and distributed without        ////
//// restriction provided that this copyright statement is not   ////
//// removed from the file and that any derivative work contains ////
//// the original copyright notice and the associated disclaimer.////
////                                                             ////
//// THIS SOURCE FILE IS PROVIDED "AS IS" AND WITHOUT ANY        ////
//// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT           ////
//// LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND   ////
//// FITNESS FOR A PARTICULAR PURPOSE.                           ////
////                                                             ////
/////////////////////////////////////////////////////////////////////


`timescale 1ns / 10ps

module fmul(clk, opa, opb, out);
input		clk;
input	[31:0]	opa, opb;
output	[31:0]	out;

////////////////////////////////////////////////////////////////////////
//
// Local Wires
//

reg	[31:0]	opa_r, opb_r;		// Input operand registers
reg	[31:0]	opa_r1, opb_r1;		// Input operand registers

reg	[31:0]	out;			// Output
wire		signa, signb;		// alias to opX sign
wire	[7:0]	expa, expb;		// alias to opX exponenet
wire	[23:0]	fracta, fractb;		// alias to opX fraction
reg	[7:0]	exp_out;
reg		sign_out;
wire	[47:0]	prod_dn;

////////////////////////////////////////////////////////////////////////
//
// Input Registers & Aliases
//

always @(posedge clk)
	opa_r1 <= #1 opa;

always @(posedge clk)
	opb_r1 <= #1 opb;

always @(posedge clk)
	opa_r <= #1 opa_r1;

always @(posedge clk)
	opb_r <= #1 opb_r1;

assign  signa = opa_r[31];
assign  signb = opb_r[31];
assign   expa = opa_r[30:23];
assign   expb = opb_r[30:23];
assign fracta = {1'b1, opa_r[22:0]};
assign fractb = {1'b1, opb_r[22:0]};

////////////////////////////////////////////////////////////////////////
//
// Calculate Exponenet
//

wire	[8:0]	exp_tmp1;
wire	[7:0]	exp_tmp2;

assign exp_tmp1 = expa + expb;		// bias is now added twice
assign exp_tmp2 = exp_tmp1 - 8'd127;	// subtract one bias

always @(posedge clk)
	exp_out <= #1 exp_tmp2;

////////////////////////////////////////////////////////////////////////
//
// Determine sign for the output
//

// sign: 0=Posetive Number; 1=Negative Number
always @(signa or signb)
   case({signa, signb})		// synopsys full_case parallel_case
	2'b0_0: sign_out = 0;
	2'b0_1: sign_out = 1;
	2'b1_0: sign_out = 1;
	2'b1_1: sign_out = 0;
   endcase

////////////////////////////////////////////////////////////////////////
//
// Perform actual MUL
//

mul24_r u3(
	.clk(clk),				// System Clock
	.opa(fracta),				// Fraction A input
	.opb(fractb),				// Fraction B Input
	.prod(prod_dn) );			// Product output denormalized(registered)

////////////////////////////////////////////////////////////////////////
//
// Normalize Result
//

wire	[30:0]	out_d;

norm_mul u4(
	.clk(clk),			// System Clock
	.fract_in(prod_dn),		// Fraction Input
	.exp_in(exp_out),		// Exponnent Input
	.out(out_d) );			// Normalized output (un-registered)

////////////////////////////////////////////////////////////////////////
//
// FMUL Outputs
//

reg		sign_r1;	// Sign output delay line

always @(posedge clk)
	sign_r1 <= #1 sign_out;

always @(posedge clk)
	out <= #1 {sign_r1, out_d};

endmodule





