-- ------------------------------------------------------------------------
-- ------------------------------------------------------------------------
-- Title: mii_rx_drv (Bus functional model for MII RX i/f)
--
-- Author: Kostas Pramataris <kpram@telecom.ntua.gr> 
--
-- Date: 30/Nov/00
--
-- Description:
-- The mii_rx_drv component reads packet-data from a file and
-- drives these data on the MII RX i/f. The component needs to
-- be triggered by a high-to-low transition of its reset signal.
--
-- The supported file format is:
--
--     # this is a comment line
--     IDLE 60 
--     PACKET_START
--         0123_ABC4_22
--         FFXX_0000
--     PACKET_END
--
--     IDLE 100 
--
--     # another packet
--     PACKET_START
--         0123ABC4
--     PACKET_END
--
-- Through "generics" you can specify:
--     - the name of the packet file
--     - the period of the mii_rx_clk
--
-- Dependencies: string_package.vhd
--
-- Bugs/Known Issues: The component could be enhanced in order to be
-- able to drive the mii_rx_err signal according to a specified pattern.
--

library STD;
use STD.textio.all;

library WORK;
use WORK.string_package.all;

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;


entity mii_rx_drv is
    generic (
        CLK_PERIOD      : time := 40 ns;
        INPUT_FILE      : string := "test.pkt"
    );
    port (
        mii_rxd     : out   std_logic_vector(3 downto 0);
        mii_rx_dv   : out   std_logic;
        mii_rx_err  : out   std_logic;
        mii_rx_clk  : out   std_logic;

        reset       : in    std_logic
    );
end mii_rx_drv;


architecture behavioral of mii_rx_drv is

-- constant declaration
    constant tpd    : time := 10 ns; -- propagation delay
 
begin


    MII_RX_DRIVER_P: process
        file in_file            : text is in INPUT_FILE;
        variable data           : std_logic_vector(3 downto 0);
        variable value          : integer;
        variable L              : line;
        variable ch             : character;
        variable str            : line;
        variable command        : line;
        variable packet_done    : boolean;

    begin

        mii_rxd     <= "0000";
        mii_rx_dv   <= '0';
        mii_rx_err  <= '0';
        mii_rx_clk  <= '0';

        wait until reset'event and reset = '0';

        READ_FILE_LOOP: loop 

            if not(endfile(in_file)) then

                readline(in_file, L);

                if (L'length /= 0) then
                   
                    getString(L, command);     

                    if (strcmp(command.all, "#") = 0) then
                        null;
                    elsif (strcmp(command.all, "IDLE") = 0) then
                        read(L, value);
                        wait for (value * CLK_PERIOD);
                    elsif (strcmp(command.all, "PACKET_START") = 0) then
                        packet_done := FALSE ;
                        SEND_PACKET_LOOP: loop 
                            if endfile(in_file) then 
                                packet_done := TRUE ;
                            else
                                readline(in_file, L);

                                getString(L, str);
                            
                                if (strcmp(str.all, "PACKET_END") = 0) then
                                    packet_done := TRUE;
                                    mii_rx_clk  <= '1';
                                    mii_rxd     <= "0000" after tpd; 
                                    mii_rx_dv   <= '0' after tpd;
                                    wait for CLK_PERIOD/2;
                                    mii_rx_clk  <= '0';
                                else
                                    for i in str'left to str'right loop
                                        ch := str(i);
                                        if ((ch = '_') OR (ch = ' ')) then
                                            null;
                                        else
                                            case ch is
                                                when '0'       => data := "0000";
                                                when '1'       => data := "0001"; 
                                                when '2'       => data := "0010";
                                                when '3'       => data := "0111";
                                                when '4'       => data := "0100";
                                                when '5'       => data := "0101";
                                                when '6'       => data := "0110";
                                                when '7'       => data := "0111";
                                                when '8'       => data := "1000";
                                                when '9'       => data := "1001";
                                                when 'a' | 'A' => data := "1010";
                                                when 'b' | 'B' => data := "1011";
                                                when 'c' | 'C' => data := "1100";
                                                when 'd' | 'D' => data := "1101";
                                                when 'e' | 'E' => data := "1110";
                                                when 'f' | 'F' => data := "1111";
                                                when others    => data := "XXXX";
                                            end case;
                                            mii_rx_clk  <= '1';
                                            mii_rxd     <= data after tpd; 
                                            mii_rx_dv   <= '1' after tpd;
                                            wait for CLK_PERIOD/2;
                                            mii_rx_clk  <= '0';
                                            wait for CLK_PERIOD/2;
                                        end if;
                                    end loop;
                                end if; -- if PACKET_END
                            end if;
                            exit when packet_done;
                        end loop SEND_PACKET_LOOP;
                    else 
                        assert FALSE -- always
                            report "wrong command in file"
                            severity ERROR;
                    end if;  -- if command
                end if; -- if line not empty 
            end if; -- if not endfile
            exit when endfile(in_file);
        end loop;

    end process;

end behavioral;
            
                    
                    


    








