--==========================================================================--
-- Design units : Testbench for RxMAC
--
-- File name    : rxmac_testbench.vhd
--
-- Purpose      : Testing!
--
-- Note         : None
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-10-07  Initial revision
--          MB      2000-11-06  Testbench now programmable via input by file
--==========================================================================--

entity RxMAC_Testbench is
end RxMAC_Testbench;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;
use IEEE.Std_Logic_Arith.all;
use IEEE.Std_Logic_TextIO.all;
use Std.TextIO.all;

library EthMAC_Lib;
use EthMAC_Lib.Eth_CRC32_4.all;

architecture Behavioral of RxMAC_Testbench is

  ----------------------------------------------------------------------------
  -- Module Under Test
  ----------------------------------------------------------------------------
  component RxMAC
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:   in  Std_ULogic;
      RxClk:       in  Std_ULogic;
      -- Media Independend Interface
      RxD:         in  Std_ULogic_Vector(3 downto 0);
      RxDV:        in  Std_ULogic;
      RxEr:        in  Std_ULogic;
      GatedCol:    in  Std_ULogic;
      -- MAC-to-Host interface
      RxValid:     out Std_ULogic;
      RxData:      out Std_ULogic_Vector(3 downto 0);
      RxLength:    out Std_ULogic_Vector(11 downto 0);
      RxCrcError:  out Std_ULogic;
      RxPhyError:  out Std_ULogic;
      RxCollision: out Std_ULogic;
      RxLateCol:   out Std_ULogic
    );
  end component;

  ----------------------------------------------------------------------------
  -- Testbench constants
  ----------------------------------------------------------------------------
  constant tClk_period: Time := 40 ns;

  ----------------------------------------------------------------------------
  -- Testbench signals
  ----------------------------------------------------------------------------
  signal Tb_RxReset_N:   Std_ULogic;
  signal Tb_RxClk:       Std_ULogic;
  signal Tb_RxD:         Std_ULogic_Vector(3 downto 0);
  signal Tb_RxDV:        Std_ULogic;
  signal Tb_RxEr:        Std_ULogic;
  signal Tb_GatedCol:    Std_ULogic;
  signal Tb_RxValid:     Std_ULogic;
  signal Tb_RxData:      Std_ULogic_Vector(3 downto 0);
  signal Tb_RxLength:    Std_ULogic_Vector(11 downto 0);
  signal Tb_RxCrcError:  Std_ULogic;
  signal Tb_RxPhyError:  Std_ULogic;
  signal Tb_RxCollision: Std_ULogic;
  signal Tb_RxLateCol:   Std_ULogic;

  ----------------------------------------------------------------------------
  -- Testbench signals (delayed by module under test)
  ----------------------------------------------------------------------------
  type RxD_delayed is array (1 to 2) of Std_ULogic_Vector(3 downto 0);
  signal Tb_RxD_delayed: RxD_delayed;

  ----------------------------------------------------------------------------
  -- Testvector files
  ----------------------------------------------------------------------------
  file File_In:  Text open Read_Mode  is "rxmac_tb_in.txt";
  file File_Pin: Text open Write_Mode is "rxmac_tb_pin.txt";
  file File_Rpt: Text open Write_Mode is "rxmac_tb_rpt.txt";

  
begin  -- Architecture Behavioral of RxMAC_Testbench

  ----------------------------------------------------------------------------
  -- Connect module under test
  ----------------------------------------------------------------------------
  Tb_RxMAC: RxMAC port map(
      RxReset_N    => Tb_RxReset_N,
      RxClk        => Tb_RxClk,
      RxD          => Tb_RxD,
      RxDV         => Tb_RxDV,
      RxEr         => Tb_RxEr,
      GatedCol     => Tb_GatedCol,
      RxValid      => Tb_RxValid,
      RxData       => Tb_RxData,
      RxLength     => Tb_RxLength,
      RxCrcError   => Tb_RxCrcError,
      RxPhyError   => Tb_RxPhyError,
      RxCollision  => Tb_RxCollision,
      RxLateCol    => Tb_RxLateCol
  );

  ----------------------------------------------------------------------------
  -- Testbench pin observer (saves input/output-pins to file)
  ----------------------------------------------------------------------------
  ObservePins: process (Tb_RxClk)
    variable OutputLine: Line;
  begin
    if Rising_Edge(Tb_RxClk) then
      Write(OutputLine, Bit'(To_Bit(Tb_RxDV)));
      Write(OutputLine, String'("-"));
      for N in 3 downto 0 loop
        Write(OutputLine, Bit'(To_Bit(Tb_RxD(N))));
      end loop;
      Write(OutputLine, String'("-"));
      Write(OutputLine, Bit'(To_Bit(Tb_RxEr)));
      Write(OutputLine, Bit'(To_Bit(Tb_GatedCol)));
      Write(OutputLine, String'(" : "));
      Write(OutputLine, Bit'(To_Bit(Tb_RxValid)));
      Write(OutputLine, String'("-"));
      for N in 3 downto 0 loop
        Write(OutputLine, Bit'(To_Bit(Tb_RxData(N))));
      end loop;
      Write(OutputLine, String'("-"));
      for N in 11 downto 0 loop
        Write(OutputLine, Bit'(To_Bit(Tb_RxLength(N))));
      end loop;
      Write(OutputLine, String'("-"));
      Write(OutputLine, Bit'(To_Bit(Tb_RxCrcError)));
      Write(OutputLine, Bit'(To_Bit(Tb_RxPhyError)));
      Write(OutputLine, String'("-"));
      Write(OutputLine, Bit'(To_Bit(Tb_RxCollision)));
      Write(OutputLine, Bit'(To_Bit(Tb_RxLateCol)));
      WriteLine(File_Pin, OutputLine);
    end if;
  end process ObservePins;

  ----------------------------------------------------------------------------
  -- Testbench signal delaying process
  ----------------------------------------------------------------------------
  DelaySignals: process (Tb_RxClk)
  begin
    if Rising_Edge(Tb_RxClk) then
      Tb_RxD_delayed(1) <= Tb_RxD;
      Tb_RxD_delayed(2) <= Tb_RxD_delayed(1);
    end if;
  end process DelaySignals;

  ----------------------------------------------------------------------------
  -- Testbench
  ----------------------------------------------------------------------------
  Testbench: process
    variable InputLine:       Line;
    variable Nibble:          Bit_Vector(3 downto 0);
    variable OutputLine:      Line;
    variable Frame:           Boolean;
    variable Offset:          Integer;
    variable FrameOffset:     Integer;
    variable FrameOffset_PHY: Integer;
    variable FrameOffset_COL: Integer;
    variable FrameOffset_Crc: Integer;
    variable FrameCrc:        Std_ULogic_Vector(31 downto 0);
    variable FrameCrcOut:     Std_ULogic_Vector(31 downto 0);
    variable Error:           Boolean;
  begin
    Write(OutputLine, String'("--- EthMAC 10/100 - RxMAC Testbench ------"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'(" Maik Boden (boden@ite.inf.tu-dresden.de)"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'(" Dresden University of Technology"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'(" Department of Information Science"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'(" Institute of Computer Engineering"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'("------------------------------------------"));
    WriteLine(File_Rpt, OutputLine);
    Write(OutputLine, String'(""));
    WriteLine(File_Rpt, OutputLine);

    --------------------------------------------------------------------------
    -- 
    --------------------------------------------------------------------------
    Frame := false;
    Error := false;

    Tb_RxReset_N <= '0';
    Tb_RxDV      <= '0';
    Tb_RxD       <= (others => '0');
    Tb_RxEr      <= '0';
    Tb_GatedCol  <= '0';
    for N in 1 to 3 loop
      Tb_RxClk <= '0';
      wait for (tClk_period / 2);
      Tb_RxClk <= '1';
      wait for (tClk_period / 2);
    end loop;
    Tb_RxReset_N <= '1';

    --------------------------------------------------------------------------
    -- 
    --------------------------------------------------------------------------
    while not EndFile(File_In) loop
      ReadLine(File_In, InputLine);
      
      ------------------------------------------------------------------------
      -- 
      ------------------------------------------------------------------------
      if InputLine'Length > 0 and InputLine(1) = ':' then
        ----------------------------------------------------------------------
        -- 
        ----------------------------------------------------------------------
        if InputLine'Length = 6 and InputLine(2 to 6) = "frame" then
          Write(OutputLine, String'("--- New frame ---"));
          WriteLine(File_Rpt, OutputLine);

          Offset          := 0;
          FrameOffset     := -1;
          FrameOffset_PHY := -1;
          FrameOffset_COL := -1;
          FrameOffset_Crc := -1;
          FrameCrc        := (others => '1');

          Tb_RxDV <= '1';
        end if;

        ----------------------------------------------------------------------
        -- 
        ----------------------------------------------------------------------
        if InputLine'Length = 5 and InputLine(2 to 5) = "stop" then
          Write(OutputLine, String'(""));
          WriteLine(File_Rpt, OutputLine);
          Write(OutputLine, String'("--- Simulation stopped by user ---"));
          WriteLine(File_Rpt, OutputLine);

          wait;
        end if;

        ----------------------------------------------------------------------
        -- 
        ----------------------------------------------------------------------
        if Frame = true then
          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "pre" then
            Write(OutputLine, String'(" - Add preamble."));
            WriteLine(File_Rpt, OutputLine);
            
            Tb_RxD   <= "0101";
            for N in 1 to 15 loop
              Tb_RxClk <= '0';
              wait for (tClk_period / 2);
              Tb_RxClk <= '1';
              wait for (tClk_period / 2);
            end loop;
          end if;

          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "sfd" then
            Write(OutputLine, String'(" - Add SFD."));
            WriteLine(File_Rpt, OutputLine);
            
            Tb_RxD   <= "1101";
            Tb_RxClk <= '0';
            wait for (tClk_period / 2);
            Tb_RxClk <= '1';
            wait for (tClk_period / 2);

            FrameOffset     := 0;
            FrameOffset_Crc := FrameOffset;
          end if;

          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "crc" then
            Write(OutputLine, String'(" - Add FCS at offset "));
            Write(OutputLine, Integer'(FrameOffset));
            Write(OutputLine, String'("."));
            WriteLine(File_Rpt, OutputLine);

            FrameCrcOut := not FrameCrc;
            
            for N in 3 downto 0 loop
              for M in 3 downto 0 loop
                Tb_RxD(M) <= FrameCrcOut((8 * N) + (7 - M));
              end loop;
              Tb_RxClk <= '0';
              wait for (tClk_period / 2);
              Tb_RxClk <= '1';
              wait for (tClk_period / 2);

              for M in 3 downto 0 loop
                Tb_RxD(M) <= FrameCrcOut((8 * N) + (3 - M));
              end loop;
              Tb_RxClk <= '0';
              wait for (tClk_period / 2);
              Tb_RxClk <= '1';
              wait for (tClk_period / 2);
            end loop;

            FrameOffset_Crc := FrameOffset + 7;
            FrameOffset     := FrameOffset + 8;
          end if;

          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "eof" then
            Write(OutputLine, String'("# End-Of-Frame:"));
            WriteLine(File_Rpt, OutputLine);

            Frame := false;

            Tb_RxDV <= '0';
            Tb_RxD  <= (others => '0');
            while Tb_RxValid = '1' loop
              Tb_RxClk <= '0';
              wait for (tClk_period / 2);
              Tb_RxClk <= '1';
              wait for (tClk_period / 2);
            end loop;

            Write(OutputLine, String'(" RxCrcError  : "));
            Write(OutputLine, Bit'(To_Bit(Tb_RxCrcError)));
            if (Tb_RxCrcError = '0' and FrameOffset_Crc = FrameOffset - 1)
               or (Tb_RxCrcError = '1'
                   and FrameOffset_Crc /= FrameOffset - 1) then
              Write(OutputLine, String'("            (ok)"));
            else
              Write(OutputLine, String'("            (ERR)"));
              Error := true;
            end if;
            WriteLine(File_Rpt, OutputLine);

            Write(OutputLine, String'(" RxPhyError  : "));
            Write(OutputLine, Bit'(To_Bit(Tb_RxPhyError)));
            if (Tb_RxPhyError = '0' and FrameOffset_PHY < 0)
               or (Tb_RxPhyError = '1' and FrameOffset_PHY >= 0) then
              Write(OutputLine, String'("            (ok)"));
            else
              Write(OutputLine, String'("            (ERR)"));
              Error := true;
            end if;
            WriteLine(File_Rpt, OutputLine);

            Write(OutputLine, String'(" RxCollision : "));
            Write(OutputLine, Bit'(To_Bit(Tb_RxCollision)));
            if (Tb_RxCollision = '0' and FrameOffset_COL < 0)
               or (Tb_RxCollision = '1' and FrameOffset_COL >= 0) then
              Write(OutputLine, String'("            (ok)"));
            else
              Write(OutputLine, String'("            (ERR)"));
              Error := true;
            end if;
            WriteLine(File_Rpt, OutputLine);

            Write(OutputLine, String'(" RxLateCol   : "));
            Write(OutputLine, Bit'(To_Bit(Tb_RxLateCol)));
            if (Tb_RxLateCol = '1' and FrameOffset_COL > 127)
               or (Tb_RxLateCol = '0' and FrameOffset_COL < 128) then
              Write(OutputLine, String'("            (ok)"));
            else
              Write(OutputLine, String'("            (ERR)"));
              Error := true;
            end if;
            WriteLine(File_Rpt, OutputLine);

            Write(OutputLine, String'(" RxLength    : "));
            for N in 11 downto 0 loop
              Write(OutputLine, Bit'(To_Bit(Tb_RxLength(N))));
            end loop;
            if Tb_RxLength = To_StdULogicVector(Conv_Std_Logic_Vector(
                             (FrameOffset), 12)) then
              Write(OutputLine, String'(" (ok)"));
            else
              Write(OutputLine, String'(" (ERR: expected "));
              Write(OutputLine, Integer'(FrameOffset));
              Write(OutputLine, String'(")"));
              Error := true;
            end if;
            WriteLine(File_Rpt, OutputLine);

            Write(OutputLine, String'(""));
            WriteLine(File_Rpt, OutputLine);

            Tb_RxClk <= '0';
            wait for (tClk_period / 2);
            Tb_RxClk <= '1';
            wait for (tClk_period / 2);
          end if;
        else
          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "sof" then
            Write(OutputLine, String'("# Start-Of-Frame:"));
            WriteLine(File_Rpt, OutputLine);
            
            if FrameOffset_PHY < 0 then
              Write(OutputLine, String'(" - PhyError disabled."));
            else
              Write(OutputLine, String'(" - PhyError set to offset "));
              Write(OutputLine, Integer'(FrameOffset_PHY));
              Write(OutputLine, String'("."));
            end if;
            WriteLine(File_Rpt, OutputLine);
            
            if FrameOffset_COL < 0 then
              Write(OutputLine, String'(" - Collision disabled."));
            else
              Write(OutputLine, String'(" - Collision set to offset "));
              Write(OutputLine, Integer'(FrameOffset_COL));
              if FrameOffset_COL > 127 then
                Write(OutputLine, String'(" (late)"));
              end if;
              Write(OutputLine, String'("."));
            end if;
            WriteLine(File_Rpt, OutputLine);

            Frame := true;
          end if;

          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "phy" then
            ReadLine(File_In, InputLine);
            Read(InputLine, FrameOffset_PHY);
          end if;

          --------------------------------------------------------------------
          -- 
          --------------------------------------------------------------------
          if InputLine'Length = 4 and InputLine(2 to 4) = "col" then
            ReadLine(File_In, InputLine);
            Read(InputLine, FrameOffset_COL);
          end if;
        end if;
      ------------------------------------------------------------------------
      -- 
      ------------------------------------------------------------------------
      elsif InputLine'Length > 0 and InputLine(1) /= '#' and Frame = true then
        Read(InputLine, Nibble);

        if FrameOffset >= 0 then
          FrameCrc := NextCRC32_4(FrameCrc, To_StdULogicVector(Nibble));
          if FrameCrc = "11000111000001001101110101111011" then
            Write(OutputLine, String'(" - Found correct CRC at offset "));
            Write(OutputLine, Integer'(FrameOffset));
            Write(OutputLine, String'("."));
            WriteLine(File_Rpt, OutputLine);

            FrameOffset_Crc := FrameOffset;
          end if;

          if FrameOffset = FrameOffset_PHY then
            Tb_RxEr <= '1';
          else
            Tb_RxEr <= '0';
          end if;

          if FrameOffset = FrameOffset_COL then
            Tb_GatedCol <= '1';
          else
            Tb_GatedCol <= '0';
          end if;

          FrameOffset := FrameOffset + 1;
        end if;

        if Nibble = "1101" and FrameOffset < 0 then
          Write(OutputLine, String'(" - Found SFD "));
          if Offset = 0 then
            Write(OutputLine, String'("at the beginning."));
          else
            Write(OutputLine, String'("after "));
            Write(OutputLine, Integer'(Offset + 1));
            Write(OutputLine, String'(" nibbles."));
          end if;
          WriteLine(File_Rpt, OutputLine);

          FrameOffset := 0;
        end if;

        Tb_RxD   <= To_StdULogicVector(Nibble);
        Tb_RxClk <= '0';
        wait for (tClk_period / 2);
        Tb_RxClk <= '1';
        wait for (tClk_period / 2);

        if Tb_RxValid = '1' then
          if Tb_RxData /= Tb_RxD_delayed(2) then
            Write(OutputLine, String'(" - Found invalid data on RxData"));
            WriteLine(File_Rpt, OutputLine);
            Error := true;
          end if;
        end if;

        Offset := Offset + 1;
      end if;

      ------------------------------------------------------------------------
      -- 
      ------------------------------------------------------------------------
      if Error = true then
        Write(OutputLine, String'(""));
        WriteLine(File_Rpt, OutputLine);
        Write(OutputLine, String'("--- ERROR found, simulation stopped ---"));
        WriteLine(File_Rpt, OutputLine);

        wait;
      end if;
    end loop;

    --------------------------------------------------------------------------
    -- 
    --------------------------------------------------------------------------
    if Frame = true then
      Write(OutputLine, String'("--- No EOF found, simulation stopped ---"));
      WriteLine(File_Rpt, OutputLine);
    else
      Write(OutputLine, String'("--- Simulation successfull ---"));
      WriteLine(File_Rpt, OutputLine);
    end if;
    
    wait;    
  end process Testbench;

end Behavioral;  -- End of RxMAC_Testbench (Behavioral)
