--==========================================================================--
-- Design units : 10/100 Ethernet MAC - RxMAC
--
-- File name    : rxmac.vhd
--
-- Purpose      : The RxMAC (Receiver Part of Medium Access Controller)
--                controls the whole part of frame reception including error
--                detection and propagation. The Media Independend Interface
--                is used to connect the MAC to any IEEE-compliant PHY. For
--                a maximum of flexibility the MAC-to-Host interface is
--                independent from any memory access method like DMA or FIFO
--                control.
--
-- Note         : None
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-09-19  Initial revision
--          MB      2000-11-26  RxValid added as input of RxErrorDetection
--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

entity RxMAC is
  port(
    -- Receive clock and asynchronous reset
    RxReset_N:   in  Std_ULogic;
    RxClk:       in  Std_ULogic;
    -- Media Independend Interface
    RxD:         in  Std_ULogic_Vector(3 downto 0);
    RxDV:        in  Std_ULogic;
    RxEr:        in  Std_ULogic;
    GatedCol:    in  Std_ULogic;
    -- MAC-to-Host interface
    RxValid:     out Std_ULogic;
    RxData:      out Std_ULogic_Vector(3 downto 0);
    RxLength:    out Std_ULogic_Vector(11 downto 0);
    RxCrcError:  out Std_ULogic;
    RxPhyError:  out Std_ULogic;
    RxCollision: out Std_ULogic;
    RxLateCol:   out Std_ULogic
  );
end RxMAC;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

architecture RTL of RxMAC is

  ----------------------------------------------------------------------------
  -- List of sub modules
  ----------------------------------------------------------------------------
  component RxSyncFIFO
    port(
      -- Receive clock and asynchronous reset
      RxReset_N: in  Std_ULogic;
      RxClk:     in  Std_ULogic;
      -- Receive signals from MII
      RxD:       in  Std_ULogic_Vector(3 downto 0);
      RxDV:      in  Std_ULogic;
      RxEr:      in  Std_ULogic;
      GatedCol:  in  Std_ULogic;
      -- Receive signals stable and synchronous to RxClk
      RxFifoD:   out Std_ULogic_Vector(3 downto 0);
      RxFifoDV:  out Std_ULogic;
      RxFifoEr:  out Std_ULogic;
      RxFifoCol: out Std_ULogic
    );
  end component;

  component RxDataRegister
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:   in  Std_ULogic;
      RxClk:       in  Std_ULogic;
      -- Receive signals from RxSyncFIFO
      RxFifoD:     in  Std_ULogic_Vector(3 downto 0);
      RxFifoDV:    in  Std_ULogic;
      -- Receive data for host system
      RxData:      out Std_ULogic_Vector(3 downto 0);
      RxDataValid: out Std_ULogic
    );
  end component;

  component RxErrorDetection
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:   in  Std_ULogic;
      RxClk:       in  Std_ULogic;
      -- Receive signals from RxSyncFIFO
      RxFifoDV:    in  Std_ULogic;
      RxFifoEr:    in  Std_ULogic;
      RxFifoCol:   in  Std_ULogic;
      -- Host interface control signals and error indicators
      RxValid:     in  Std_ULogic;
      RxPhyError:  out Std_ULogic;
      RxCollision: out Std_ULogic;
      -- Collision management
      RxLate:      in Std_ULogic;
      RxLateCol:   out Std_ULogic
    );
  end component;

  component RxFrameControl
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:   in  Std_ULogic;
      RxClk:       in  Std_ULogic;
      -- Receive signals from RxSyncFIFO and RxDataRegister
      RxData:      in  Std_ULogic_Vector(3 downto 0);
      RxDataValid: in  Std_ULogic;
      RxFifoDV:    in  Std_ULogic;
      -- Receive frame data valid indicator
      RxValid:     out Std_ULogic
    );
  end component;

  component RxLengthCounter
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:   in  Std_ULogic;
      RxClk:       in  Std_ULogic;
      -- Valid frame data indicator
      RxValid:     in  Std_ULogic;
      -- Offset of the current nibble within the frame
      RxLate:      out Std_ULogic;
      RxLength:    out Std_ULogic_Vector(11 downto 0)
    );
  end component;

  component RxCRCCheck
    port(
      -- Receive clock and asynchronous reset
      RxReset_N:  in  Std_ULogic;
      RxClk:      in  Std_ULogic;
      -- Valid frame data indicator
      RxData:     in  Std_ULogic_Vector(3 downto 0);
      RxValid:    in  Std_ULogic;
      -- Offset of the current nibble within the frame
      RxCrcError: out Std_ULogic
    );
  end component;

  ----------------------------------------------------------------------------
  -- List of external available module outputs (pins)
  ----------------------------------------------------------------------------
  signal PinRxValid:     Std_ULogic;
  signal PinRxData:      Std_ULogic_Vector(3 downto 0);
  signal PinRxLength:    Std_ULogic_Vector(11 downto 0);
  signal PinRxCrcError:  Std_ULogic;
  signal PinRxPhyError:  Std_ULogic;
  signal PinRxCollision: Std_ULogic;
  signal PinRxLateCol:   Std_ULogic;

  ----------------------------------------------------------------------------
  -- List of not external available module outputs (wires)
  ----------------------------------------------------------------------------
  signal WireRxFifoD:     Std_ULogic_Vector(3 downto 0);
  signal WireRxFifoDV:    Std_ULogic;
  signal WireRxFifoEr:    Std_ULogic;
  signal WireRxFifoCol:   Std_ULogic;
  signal WireRxLate:      Std_ULogic;
  signal WireRxDataValid: Std_ULogic;

begin  -- Architecture RTL of RxMAC

  ModuleRxSyncFIFO: RxSyncFIFO
    port map(
      RxReset_N => RxReset_N,
      RxClk     => RxClk,
      RxD       => RxD,
      RxDV      => RxDV,
      RxEr      => RxEr,
      GatedCol  => GatedCol,
      RxFifoD   => WireRxFifoD,
      RxFifoDV  => WireRxFifoDV,
      RxFifoEr  => WireRxFifoEr,
      RxFifoCol => WireRxFifoCol
    );

  ModuleRxDataRegister: RxDataRegister
    port map(
      RxReset_N   => RxReset_N,
      RxClk       => RxClk,
      RxFifoD     => WireRxFifoD,
      RxFifoDV    => WireRxFifoDV,
      RxData      => PinRxData,
      RxDataValid => WireRxDataValid
    );

  ModuleRxErrorDetection: RxErrorDetection
    port map(
      RxReset_N   => RxReset_N,
      RxClk       => RxClk,
      RxFifoDV    => WireRxFifoDV,
      RxFifoEr    => WireRxFifoEr,
      RxFifoCol   => WireRxFifoCol,
      RxValid    => PinRxValid,
      RxPhyError  => PinRxPhyError,
      RxCollision => PinRxCollision,
      RxLate      => WireRxLate,
      RxLateCol   => PinRxLateCol
    );

  ModuleRxFrameControl: RxFrameControl
    port map(
      RxReset_N   => RxReset_N,
      RxClk       => RxClk,
      RxData      => PinRxData,
      RxDataValid => WireRxDataValid,
      RxFifoDV    => WireRxFifoDV,
      RxValid     => PinRxValid
    );

  ModuleRxLengthCounter: RxLengthCounter
    port map(
      RxReset_N => RxReset_N,
      RxClk     => RxClk,
      RxValid   => PinRxValid,
      RxLate    => WireRxLate,
      RxLength  => PinRxLength
    );

  ModuleRxCRCCheck: RxCRCCheck
    port map(
      RxReset_N  => RxReset_N,
      RxClk      => RxClk,
      RxData     => PinRxData,
      RxValid    => PinRxValid,
      RxCrcError => PinRxCrcError
    );

  ----------------------------------------------------------------------------
  -- Propagate module outputs to design outputs
  ----------------------------------------------------------------------------
  RxValid     <= PinRxValid;
  RxData      <= PinRxData;
  RxLength    <= PinRxLength;
  RxCrcError  <= PinRxCrcError;
  RxPhyError  <= PinRxPhyError;
  RxCollision <= PinRxCollision;
  RxLateCol   <= PinRxLateCol;

end RTL;  -- End of RxMAC (RTL)
