--==========================================================================--
-- Design units : 10/100 Ethernet MAC - RxMAC - RxCRCCheck
--
-- File name    : rxcrccheck.vhd
--
-- Purpose      : The CRC check module computes the CRC-32 of the received
--                frame including the FCS field. In all cases of a succesfully
--                transmission the value of the internal register is equal to
--                the so called CRC remainder (0xC704DD7B). Otherwise the
--                recepted frame data was corrupt.
--
-- Note         : This checker computes the CRC-32 algorithm on 4-bit-wide
--                input data (nibble-wide). Not byte per byte!
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-09-04  Initial revision
--          MB      2000-11-01  CRC-32 function extracted into a package
--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

entity RxCRCCheck is
  port(
    -- Receive clock and asynchronous reset
    RxReset_N:  in  Std_ULogic;
    RxClk:      in  Std_ULogic;
    -- Valid frame data indicator
    RxData:     in  Std_ULogic_Vector(3 downto 0);
    RxValid:    in  Std_ULogic;
    -- Offset of the current nibble within the frame
    RxCrcError: out Std_ULogic
  );
end RxCRCCheck;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;
library EthMAC_Lib;
use EthMAC_Lib.Eth_CRC32_4.all;

architecture RTL of RxCRCCheck is

  ----------------------------------------------------------------------------
  -- Internal 32-bit-wide register for CRC computation.
  ----------------------------------------------------------------------------
  signal CrcReg: Std_ULogic_Vector(31 downto 0);

begin  -- Architecture RTL of RxCRCCheck

  ----------------------------------------------------------------------------
  -- The computation of the CRC is done while RxValid is one. At the beginning
  -- of a new frame (detected by watching RxValid) the CrcReg is reset
  -- to the initial value.
  ----------------------------------------------------------------------------
  ComputeCRC: process (RxReset_N, RxClk)
  begin
    if RxReset_N = '0' then
      CrcReg <= (others => '1');
    elsif Rising_Edge(RxClk) then
      if RxValid = '1' then
        CrcReg <= nextCrc32_4(CrcReg, RxData);
      else
        CrcReg <= (others => '1');
      end if;
    end if;
  end process ComputeCRC;
    
  ------------------------------------------------------------------------------
  -- The presence of any error can simply checked by comparing the final state
  -- of CrcReg to the CRC-32 remainder (0xC704DD7B).
  ------------------------------------------------------------------------------
  RxCrcError <= '0' when CrcReg = "11000111000001001101110101111011" else '1';

end RTL;  -- End of RxCRCCheck (RTL)
