/////////////////////////////////////////////////////////////////////
////                                                             ////
////  FASU                                                       ////
////  Floating point Add/Subtract Unit (Single precision)        ////
////                                                             ////
////  Author: Rudolf Usselmann                                   ////
////          russelmann@hotmail.com                             ////
////                                                             ////
/////////////////////////////////////////////////////////////////////
////                                                             ////
//// Copyright (C) 2000 Rudolf Usselmann                         ////
////                    russelmann@hotmail.com                   ////
////                                                             ////
//// This source file may be used and distributed without        ////
//// restriction provided that this copyright statement is not   ////
//// removed from the file and that any derivative work contains ////
//// the original copyright notice and the associated disclaimer.////
////                                                             ////
//// THIS SOURCE FILE IS PROVIDED "AS IS" AND WITHOUT ANY        ////
//// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT           ////
//// LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND   ////
//// FITNESS FOR A PARTICULAR PURPOSE.                           ////
////                                                             ////
/////////////////////////////////////////////////////////////////////


`timescale 1ns / 10ps

module fasu( clk, add, opa, opb, out);
input		clk;
input		add;
input	[31:0]	opa, opb;
output	[31:0]	out;

////////////////////////////////////////////////////////////////////////
//
// Local Wires
//
reg	[31:0]	opa_r, opb_r;		// Input operand registers
reg		add_r;			// Input register
reg	[31:0]	out;			// Output register
wire		signa, signb;		// alias to opX sign
wire		sign_out;		// sign output
reg		sign_r1;		// Sign output delay line
wire	[23:0]	fracta, fractb;		// Fraction Outputs from EQU block
wire	[7:0]	exp_dn;			// Exponent output from EQU block
reg	[7:0]	exp_dn_r;		// Exponnent output one cycle delayed
wire	[24:0]	fract_out_d;		// fraction output (denormalized)

////////////////////////////////////////////////////////////////////////
//
// Input Registers
//

always @(posedge clk)
	opa_r <= #1 opa;

always @(posedge clk)
	opb_r <= #1 opb;

always @(posedge clk)
	add_r <= #1 add;

////////////////////////////////////////////////////////////////////////
//
// Equalize block
// - Adjusts the numbers to equal exponents and sorts them
// - determine result sign
// - determine actual operation to perform (add or sub)
//

equ u0(
	.clk(clk),			// System Clock
	.add(add_r),			// Add/Sub Input
	.opa(opa_r),  .opb(opb_r),	// Registered Inputs
	.fracta_out(fracta),		// Equalized and sorted fraction
	.fractb_out(fractb),		// outputs (Registered)
	.exp_dn_out(exp_dn),		// Selected exponent output (registered);
	.sign(sign_out),		// Encoded output Sign (regisred)
	.op_out(op_out) );		// Actual operation output (registered)

////////////////////////////////////////////////////////////////////////
//
// Add/Sub
//

add_sub24_r u3(
	.clk(clk),				// System Clock
	.add(op_out),				// Add/Sub
	.opa(fracta),				// Fraction A input
	.opb(fractb),				// Fraction B Input
	.sum(fract_out_d) );			// SUM output (with carry, registered)

	
////////////////////////////////////////////////////////////////////////
//
// Normalize Result
//

wire	[30:0]	out_d;

always @(posedge clk)			// Exponent must be once cycle delayed
	exp_dn_r <= #1 exp_dn;

norm u4(
	.clk(clk),			// System Clock
	.fract_in(fract_out_d),		// Fraction Input
	.exp_in(exp_dn_r),		// Exponnent Input
	.out(out_d) );			// Normalized output (un-registered)


////////////////////////////////////////////////////////////////////////
//
// FASU Outputs
//

always @(posedge clk)
	sign_r1 <= #1 sign_out;

always @(posedge clk)
	out <= #1 {sign_r1, out_d};

endmodule
