/*************************************************************************\
*                                                                         *
*    File    : tx_state_machine.v                                         *
*    Purpose : Transmit State Machine Module                              *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to control transmit process.          *
*                                                                         *
*    When host has packet of data to be transmitted, tx_sof will be       *
*    activated. Then, the State Machine will give transmit_new_p signal   *
*    to restart Defer Counter and Collision Counter and wait until        *
*    transmit_available_p from IFG Timer is detected to get opportunity   *
*    to transmit Preamble and SFD. When waiting transmit_available_p, the *
*    State Machine monitors excess_deferral to limit waiting time for     *
*    transmit operation.                                                  *
*                                                                         *
*    After IFG has passed and transmit operation has got opportunity,     *
*    the State Machine will give appropriate data_select signal to Data   *
*    Multiplexer and assert transmit_enable to FIFO Synchronization,      *
*    Collision Counter, Frame Length Counter, and IFG Timer. The State    *
*    Machine will also give transmit_preamble to Collision Counter        *
*    followed by transmit_sfd.                                            *
*    The Preamble Field will be transmitted if the transmit_available_p   *
*    is detected until the next 15 clock cycles. The SFD Field will be    *
*    transmitted while the value of counter is 16. The Data Field will be *
*    transmitted if the value of count_length is greater then 16 until    *
*    tx_eof is detected. The PAD pattern will be transmitted if tx_eof is *
*    detected when the value of count_length is less than 136 and will be *
*    deactivated if the value of count_length reaches 136. The FCS field  *
*    will be transmitted if tx_eof is detected and the value of           *
*    count_length is greater than or equal to 136. The transmit_fcs will  *
*    be activated while count_fcs is less than or equal to 8.             *
*                                                                         *
*    If Data Field starts to be transmitted, tx_data_used and compute_crc *
*    are asserted. Then, the State Machine will monitor tx_eof that       *
*    indicates the last nibble of data is placed in tx_data.              *
*    If tx_eof is detected active and Tx Eth MAC doesn't to send PAD      *
*    pattern, the State Machine will give transmit_data_end to Frame      *
*    Length Counter and deassert compute_crc signal.                      *
*                                                                         *
*    While transmit operation is running, the State Machine continuously  * 
*    monitors tx_underrun, coll_event_p, and excessive_length. If         *
*    detected, transmit operation will be aborted or dropped. If          *
*    coll_event_p is detected, the State Machine will give tx_retransmit  *
*    pulse signal and deassert transmit_enable, compute_crc, and          *
*    tx_data_used. The following operation is waiting backoff_p and then  *
*    asserts transmit_enable again to restart transmit operation from     *
*    the beginning of Preamble Field or abort transmission because late   *
*    collision and or excessive collision is detected.                    *
*                                                                         *
*    If transmit operation terminates in normal condition or because any  *
*    error conditions, the State Machine will deassert transmit_enable    *
*    and followed by status signal.                                       *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

`define WIDTH_STATE   4
`define WIDTH_STATUS  7

module tx_state_machine (
  tx_sof,
  tx_eof,
  tx_underrun,
  clk,
  reset_n,
  transmit_available_p,
  excess_deferral,
  coll_event_p,
  late_coll,
  excessive_coll,
  backoff_p,
  count_length,
  count_fcs,
  excessive_length,
  count_jam,
  tx_data_used,
  tx_done,
  tx_abort,
  tx_retransmit,
  tx_status,
  tx_status_valid_n,
  transmit_new_p,
  transmit_enable,
  transmit_preamble,
  transmit_sfd,
  transmit_data_end,
  transmit_error,
  start_backoff,
  compute_crc,
  data_select
  );

input tx_sof;
input tx_eof;
input tx_underrun;
input clk;
input reset_n;
input transmit_available_p;
input excess_deferral;
input coll_event_p;
input late_coll;
input excessive_coll;
input backoff_p;
input [11:0] count_length;
input [3:0] count_fcs;
input excessive_length;
input [3:0] count_jam;
output tx_data_used;
output tx_done;
output tx_abort;
output tx_retransmit;
output [`WIDTH_STATUS-1:0] tx_status;
output tx_status_valid_n;
output transmit_new_p;
output transmit_enable;
output transmit_preamble;
output transmit_sfd;
output transmit_data_end;
output transmit_error; 
output start_backoff;
output compute_crc;
output [1:0] data_select;

reg tx_data_used;
reg tx_done;
reg tx_abort;
reg tx_retransmit;
reg [`WIDTH_STATUS-1:0] tx_status;
reg tx_status_valid_n;
reg transmit_new_p;
reg transmit_enable; 
reg transmit_preamble;
reg transmit_sfd;
reg transmit_data_end;
reg start_backoff;
reg compute_crc;
reg [1:0] data_select;
reg [`WIDTH_STATE-1:0] current_tx_state;
reg [`WIDTH_STATE-1:0] next_tx_state;
reg transmit_error;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    current_tx_state <= #`UD `TRANSMIT_IDLE;
  else
    current_tx_state <= #`UD next_tx_state;

always @(current_tx_state or tx_sof or tx_eof or tx_underrun or
         transmit_available_p or excess_deferral or excessive_length or 
         excessive_coll or late_coll or coll_event_p or backoff_p or
         count_length or count_fcs or count_jam)
  case (current_tx_state)
    `TRANSMIT_IDLE     : if (tx_sof == 1'b1)       //idle, no transmit request
                           next_tx_state = `NEW_TRANSMIT;
    `NEW_TRANSMIT      : if (tx_sof == 1'b0)  //new packet to transmit
                           next_tx_state = `TRANSMIT_IDLE;
                         else
                           next_tx_state = `WAIT_TRANSMIT;
    `WAIT_TRANSMIT     : if (tx_sof == 1'b0)
                           next_tx_state = `TRANSMIT_IDLE;
                         else
                         if (excess_deferral == 1'b1)
                           next_tx_state = `TRANSMIT_ABORT;
                         else
                         if (transmit_available_p == 1'b1)  //wait to transmit
                           next_tx_state = `TRANSMIT_PREAMBLE;
    `TRANSMIT_PREAMBLE : if (tx_sof == 1'b0)
                           next_tx_state = `TRANSMIT_IDLE;
                         else
                         if (count_length == `PREAMBLE_LENGTH-1)
                           next_tx_state = `TRANSMIT_SFD;
    `TRANSMIT_SFD      : if (tx_sof == 1'b0)
                           next_tx_state = `TRANSMIT_IDLE;
                         else
                           next_tx_state = `TRANSMIT_DATA;
    `TRANSMIT_DATA     : if (coll_event_p == 1'b1)
                           next_tx_state = `TRANSMIT_JAM;
                         else
                         if ( (tx_underrun == 1'b1) || 
                              (excessive_length == 1'b1) )
                           next_tx_state = `TRANSMIT_ABORT;
                         else
                         if ( (tx_eof == 1'b1) && (count_length <
                              (`PREAMBLE_LENGTH+`SFD_LENGTH+`DATA_LENGTH-1)) )
                           next_tx_state = `TRANSMIT_DATAEND2;
                         else
                         if ( (tx_eof == 1'b1) && (count_length >=
                              (`PREAMBLE_LENGTH+`SFD_LENGTH+`DATA_LENGTH-1)) )
                           next_tx_state = `TRANSMIT_DATAEND1;
    `TRANSMIT_DATAEND1 : if (coll_event_p == 1'b1)
                           next_tx_state = `TRANSMIT_JAM;
                         else
                         if (excessive_length == 1'b1)
                           next_tx_state = `TRANSMIT_ABORT;
                         else
                           next_tx_state = `TRANSMIT_FCS;
    `TRANSMIT_FCS      : if (coll_event_p == 1'b1 )
                           next_tx_state = `TRANSMIT_JAM;
                         else
                         if (excessive_length == 1'b1)
                           next_tx_state = `TRANSMIT_ABORT;
                         else
                         if (count_fcs == `FCS_LENGTH-1)
                           next_tx_state = `WAIT_DONE;
    `TRANSMIT_DATAEND2 : if (coll_event_p == 1'b1)
                           next_tx_state = `TRANSMIT_JAM;
                         else
                           next_tx_state = `TRANSMIT_PAD;
    `TRANSMIT_PAD      : if (coll_event_p == 1'b1)
                           next_tx_state = `TRANSMIT_JAM;
                         else
                         if ( count_length ==
                           (`PREAMBLE_LENGTH+`SFD_LENGTH+`DATA_LENGTH-1) )
                           next_tx_state = `TRANSMIT_DATAEND1;
    `TRANSMIT_JAM      : if (count_jam == `JAM_LENGTH)
                         begin
                           if ( (late_coll == 1'b1) || 
                                (excessive_coll == 1'b1) )
                             next_tx_state = `TRANSMIT_ABORT;
                           else  
                             next_tx_state = `WAIT_BACKOFF;
                         end
    `WAIT_BACKOFF      : if (backoff_p == 1'b1)
                           next_tx_state = `TRANSMIT_PREAMBLE;
    `WAIT_DONE         : if (coll_event_p == 1'b1 )
                           next_tx_state = `TRANSMIT_JAM;
                         else
                         if (excessive_length == 1'b1)
                           next_tx_state = `TRANSMIT_ABORT;
                         else
                           next_tx_state = `TRANSMIT_DONE;
    `TRANSMIT_DONE     : next_tx_state = `WRITE_STATUS;
    `TRANSMIT_ABORT    : next_tx_state = `WRITE_STATUS;
    `WRITE_STATUS      : next_tx_state = `TRANSMIT_IDLE;
    default            : next_tx_state = `TRANSMIT_IDLE;
  endcase

always @(current_tx_state)
  case (current_tx_state)
    `TRANSMIT_IDLE :
      begin
        tx_data_used = 1'b0;
        tx_done = 1'b0;
        tx_abort = 1'b0;
        tx_retransmit = 1'b0;
        tx_status = 7'b000_0000;
        tx_status_valid_n = 1'b1;
        transmit_new_p = 1'b0;
        transmit_enable = 1'b0;
        transmit_preamble = 1'b0;
        transmit_sfd = 1'b0;
        transmit_data_end = 1'b0;
        start_backoff = 1'b0;
        data_select = 2'b00;
        compute_crc = 1'b0;
      end
    `NEW_TRANSMIT :
      begin
        transmit_new_p = 1'b1;
      end
    `WAIT_TRANSMIT :
      begin
        transmit_new_p = 1'b0;
      end
    `TRANSMIT_PREAMBLE :
      begin
        start_backoff = 1'b0;
        tx_retransmit = 1'b0;
        transmit_preamble = 1'b1;
        transmit_enable = 1'b1;
        data_select = 2'b01;
      end
    `TRANSMIT_SFD :
      begin
        transmit_preamble = 1'b0;
        transmit_sfd = 1'b1;
        data_select = 2'b10;
      end
    `TRANSMIT_DATA :
      begin
        transmit_sfd = 1'b0;
        data_select = 2'b00;
        compute_crc = 1'b1;
        tx_data_used = 1'b1;
      end
    `TRANSMIT_DATAEND1 :
      begin
        transmit_data_end = 1'b1;
        compute_crc = 1'b0;
      end
    `TRANSMIT_FCS :
      begin
        transmit_data_end = 1'b0;
        data_select = 2'b11;
        tx_data_used = 1'b0;
      end
    `TRANSMIT_DATAEND2 :      ;
    `TRANSMIT_PAD :
      begin
        data_select = 2'b01;
        tx_data_used = 1'b0;
      end
    `TRANSMIT_JAM :
      begin
        tx_data_used = 1'b0;
        data_select = 2'b01;
        compute_crc = 1'b0;
      end
    `WAIT_BACKOFF :
      begin
        start_backoff = 1'b1;
        tx_retransmit = 1'b1;
        transmit_enable = 1'b0;
        data_select = 2'b00;
      end
    `WAIT_DONE     :
      begin
        transmit_enable = 1'b0;
        transmit_data_end = 1'b0;
        data_select = 2'b00;
      end
    `TRANSMIT_DONE :
      begin
        tx_done = 1'b1;
      end
    `TRANSMIT_ABORT :
      begin
        transmit_data_end = 1'b0;
        transmit_new_p = 1'b0;
        transmit_enable = 1'b0;
        data_select = 2'b00;
        compute_crc = 1'b0;
        tx_abort = 1'b1;
      end
    `WRITE_STATUS :
      begin
        tx_done = 1'b0;
        tx_abort = 1'b0;
        tx_status [0] = excess_deferral;
        tx_status [1] = late_coll;
        tx_status [2] = excessive_coll;
        tx_status [3] = tx_underrun;
        tx_status [4] = excessive_length;
        tx_status [5] = (excess_deferral | late_coll | excessive_coll | 
                        tx_underrun | excessive_length);
        tx_status [6] = ~(excess_deferral & late_coll & excessive_coll & 
                        tx_underrun & excessive_length);
        tx_status_valid_n = 1'b0;
      end
    default :
      begin
        tx_data_used = 1'b0;
        tx_done = 1'b0;
        tx_abort = 1'b0;
        tx_retransmit = 1'b0;
        tx_status = 7'b000_0000;
        tx_status_valid_n = 1'b1;
        transmit_new_p = 1'b0;
        transmit_enable = 1'b0;
        transmit_preamble = 1'b0;
        transmit_sfd = 1'b0;
        transmit_data_end = 1'b0;
        start_backoff = 1'b0;
        data_select = 2'b00;
        compute_crc = 1'b0;
      end
  endcase

always @(posedge clk)
 transmit_error = tx_underrun; 

endmodule   