/*************************************************************************\
*                                                                         *
*    File    : tx_ethmac.v                                                *
*    Purpose : Tx Eth MAC Top Level Module                                *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    This module implements CSMA/CD algorithm for transmitting packets    *
*    of data from DMA to MII. Tx Eth MAC consists of eleven sub modules : *
*     - FIFO Synchronization (fifo_synch.v)                               *
*     - IFG Timer (ifg_timer.v)                                           *
*     - Defer Counter (defer_counter.v)                                   *
*     - Frame Length Counter (frame_length_counter.v)                     *
*     - Collision Counter (coll_counter.v)                                *
*     - Random Number Generator (random_number_gen.v)                     *
*     - Backoff Timer (backoff_timer.v)                                   *
*     - Jam Timer (jam_timer.v)                                           *
*     - CRC Generator (crc_gen.v)                                         *
*     - Data Multiplexer (data_mux.v)                                     *
*     - Tx State Machine (tx_state_machine.v)                             *
*                                                                         *
\*************************************************************************/

module tx_ethmac (
  tx_data,
  tx_sof,
  tx_eof,
  tx_underrun,
  full_duplex,
  reset_n,
  crs,
  coll,
  tx_clk,
  tx_data_used,
  tx_done,
  tx_abort,
  tx_retransmit,
  tx_status,
  tx_status_valid_n,
  txd,
  tx_en,
  tx_er
  );

// from DMA
input  [3:0] tx_data;
input  tx_sof;
input  tx_eof;
input  tx_underrun;
input  full_duplex; 
input  reset_n;

// from MII
input  crs;
input  coll;
input  tx_clk;

// to MII
output  [3:0] txd;
output  tx_en;
output  tx_er;

// to DMA
output  tx_data_used;
output  tx_done;
output  tx_abort;
output  tx_retransmit;
output  [6:0] tx_status;
output  tx_status_valid_n;

wire [3:0] data;
wire transmit_error;
wire transmit_enable;
wire compute_crc;
wire [3:0] crc;
wire [1:0] data_select;
wire transmit_available_p;
wire transmit_new_p;
wire excess_deferral;
wire transmit_preamble;
wire transmit_sfd;
wire [3:0] coll_attempt;
wire late_coll;
wire excessive_coll;
wire coll_event_p;
wire transmit_64byte;
wire transmit_data_end;
wire [11:0] count_length;
wire [3:0] count_fcs;
wire excessive_length;
wire [3:0] count_jam;
wire [9:0] random;
wire backoff_p;
wire start_backoff;

//instantiates modules
fifo_synch tx_1 (
  .data(data),
  .transmit_enable(transmit_enable),
  .transmit_error(transmit_error),
  .clk(tx_clk),
  .reset_n(reset_n),
  .txd(txd),
  .tx_en(tx_en),
  .tx_er(tx_er)
  );

ifg_timer tx_2 (
  .crs(crs),
  .full_duplex(full_duplex),
  .transmit_enable(transmit_enable),
  .clk(tx_clk),
  .reset_n(reset_n),
  .transmit_available_p(transmit_available_p)
  );

defer_counter tx_3 (
  .transmit_available_p(transmit_available_p),
  .transmit_new_p(transmit_new_p),
  .clk(tx_clk),
  .reset_n(reset_n),
  .excess_deferral(excess_deferral)
  );

frame_length_counter tx_4 (
  .transmit_enable(transmit_enable),
  .transmit_data_end(transmit_data_end),
  .clk(tx_clk),
  .reset_n(reset_n),
  .count_length(count_length),
  .count_fcs(count_fcs),
  .excessive_length(excessive_length),
  .transmit_64byte(transmit_64byte)
  );

coll_counter tx_5 (
  .transmit_new_p(transmit_new_p),
  .transmit_enable(transmit_enable),
  .transmit_preamble(transmit_preamble),
  .transmit_sfd(transmit_sfd),
  .transmit_64byte(transmit_64byte),
  .clk(tx_clk),
  .reset_n(reset_n),
  .coll(coll),
  .full_duplex(full_duplex),
  .coll_event_p(coll_event_p),
  .late_coll(late_coll),
  .excessive_coll(excessive_coll),
  .coll_attempt(coll_attempt)
  );

random_number_gen tx_6 (
  .coll_attempt(coll_attempt),
  .clk(tx_clk),
  .reset_n(reset_n),
  .random(random)
  );

backoff_timer tx_7 (
  .start_backoff(start_backoff),
  .random(random),
  .clk(tx_clk),
  .reset_n(reset_n),
  .backoff_p(backoff_p)
  );

jam_timer tx_8 (
  .coll_event_p(coll_event_p),
  .clk(tx_clk),
  .reset_n(reset_n),
  .count_jam(count_jam)
  );

crc_gen tx_9 (
  .compute_crc(compute_crc),
  .data(data),
  .clk(tx_clk),
  .reset_n(reset_n),
  .crc(crc)
  );

data_mux tx_10(
  .data_select(data_select),
  .tx_data(tx_data),
  .crc(crc),
  .data(data)
  );

tx_state_machine tx_11 (
  .tx_sof(tx_sof),
  .tx_eof(tx_eof),
  .tx_underrun(tx_underrun),
  .clk(tx_clk),
  .reset_n(reset_n),
  .transmit_available_p(transmit_available_p),
  .excess_deferral(excess_deferral),
  .coll_event_p(coll_event_p),
  .late_coll(late_coll),
  .excessive_coll(excessive_coll),
  .backoff_p(backoff_p),
  .count_length(count_length),
  .count_fcs(count_fcs),
  .excessive_length(excessive_length),
  .count_jam(count_jam),
  .tx_data_used(tx_data_used),
  .tx_done(tx_done),
  .tx_abort(tx_abort),
  .tx_retransmit(tx_retransmit),
  .tx_status(tx_status),
  .tx_status_valid_n(tx_status_valid_n),
  .transmit_new_p(transmit_new_p),
  .transmit_enable(transmit_enable),
  .transmit_preamble(transmit_preamble),
  .transmit_sfd(transmit_sfd),
  .transmit_data_end(transmit_data_end),
  .transmit_error(transmit_error),
  .start_backoff(start_backoff),
  .compute_crc(compute_crc),
  .data_select(data_select)
  );

endmodule