/*************************************************************************\
*                                                                         *
*    File    : random_number_gen.v                                        *
*    Purpose : Random Number Generator Module                             *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    This module generates random number using Linear Feedback Shift      *
*    Register which contains 10 bit register with polinomial feedback is  *
*    feedback = x[2] XNOR x[9].                                           *
*    This Random Number Generator will select random number from range    *
*    between 0 to (2^k)-1 with k is the smaller number between number of  *
*    collision and 10. The number of collision is equal to coll_attempt.  *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module random_number_gen (
  coll_attempt,
  clk,
  reset_n,
  random
  );

input [3:0] coll_attempt;
input clk;
input reset_n;
output [9:0] random;

reg [9:0] random;
wire [9:0] x;
wire feedback;

dff_tx u0 ( .d(feedback), .clk(clk), .reset_n(reset_n), .q(x[0]) );
dff_tx u1 ( .d(x[0]), .clk(clk), .reset_n(reset_n), .q(x[1]) );
dff_tx u2 ( .d(x[1]), .clk(clk), .reset_n(reset_n), .q(x[2]) );
dff_tx u3 ( .d(x[2]), .clk(clk), .reset_n(reset_n), .q(x[3]) );
dff_tx u4 ( .d(x[3]), .clk(clk), .reset_n(reset_n), .q(x[4]) );
dff_tx u5 ( .d(x[4]), .clk(clk), .reset_n(reset_n), .q(x[5]) );
dff_tx u6 ( .d(x[5]), .clk(clk), .reset_n(reset_n), .q(x[6]) );
dff_tx u7 ( .d(x[6]), .clk(clk), .reset_n(reset_n), .q(x[7]) );
dff_tx u8 ( .d(x[7]), .clk(clk), .reset_n(reset_n), .q(x[8]) );
dff_tx u9 ( .d(x[8]), .clk(clk), .reset_n(reset_n), .q(x[9]) );

assign feedback = x[2] ~^ x[9];

always @(posedge clk)
  if (coll_attempt >= 4'd10)
    random <= #`UD x;
  else
    case (coll_attempt)
      4'd1 :
        begin 
          random [0] <= #`UD x [0];
          random [9:1] <= #`UD 9'd0;
        end
      4'd2 : 
        begin 
          random [1:0] <= #`UD x [1:0];
          random [9:2] <= #`UD 8'd0;
        end
      4'd3 : 
        begin 
          random [2:0] <= #`UD x [2:0];
          random [9:3] <= #`UD 7'd0;
        end
      4'd4 :
        begin 
          random [3:0] <= #`UD x [3:0];
          random [9:4] <= #`UD 6'd0;
        end
      4'd5 : 
        begin 
          random [4:0] <= #`UD x [4:0];
          random [9:5] <= #`UD 5'd0;
        end
      4'd6 : 
        begin 
          random [5:0] <= #`UD x [5:0];
          random [9:6] <= #`UD 4'd0;
        end
      4'd7 :
        begin 
          random [6:0] <= #`UD x [6:0];
          random [9:7] <= #`UD 3'd0;
        end
      4'd8 : 
        begin 
          random [7:0] <= #`UD x [7:0];
          random [9:8] <= #`UD 2'd0;
        end
      4'd9 : 
        begin 
          random [8:0] <= #`UD x [8:0];
          random [9] <= #`UD 1'd0;
        end
      default : random <= #`UD 10'd0;
    endcase

endmodule