/*************************************************************************\
*                                                                         *
*    File    : ifg_timer.v                                                *
*    Purpose : Inter Frame Gap Timer Module                               *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to assure time interval between two   *
*    packets of data or Inter Frame Gap (IFG) is 96 bit time or 24 clock  *
*    cycles. In half duplex mode, if counter can pass IFG without any     *
*    crs signal, transmit_available_p will be asserted. After asserting   *
*    transmit_available_p pulse, this module will monitor carrier signal  *
*    again. When operates in full duplex mode, this module will ignore    *
*    any crs signals and will count IFG after transmit_enable has been    *
*    disappeared.                                                         *
*    The IFG has two intervals time, IFG_1 for the first 60 bit times     *
*    and IFG_2 for the following 36 bit times of IFG. If crs signal       *
*    appears in the IFG_1 then the value of counter is reseted. But, if   *
*    the crs signal appears in the IFG_2 then the counter will keep       *
*    running until 96 bit times has been passed. This operation makes     *
*    every station has an equal opportunity to access the medium.         *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module ifg_timer (
  crs,
  full_duplex,
  transmit_enable,
  clk,
  reset_n,
  transmit_available_p
  );

input crs;
input full_duplex;
input transmit_enable;
input clk;
input reset_n;
output transmit_available_p;

reg transmit_available_p;
reg [4:0] count_ifg;
reg [1:0] state_ifg;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    state_ifg <= #`UD `IFG_IDLE;
  else
  if (full_duplex == 1'b1)    // full duplex mode
    case (state_ifg)
      `IFG_IDLE : if ( transmit_enable == 1'b0 )
                  begin
                    state_ifg <= #`UD `IFG_RUN;
                    count_ifg <= #`UD 5'd1;
                  end
                  else
                    count_ifg <= #`UD 5'd0;
      `IFG_RUN  : if ( transmit_enable == 1'b1 )
                    state_ifg <= #`UD `IFG_IDLE;
                  else
                  if ( count_ifg == `IFG_LENGTH_2-1 )
                    state_ifg <= #`UD `IFG_MAX;
                  else
                    count_ifg <= #`UD (count_ifg + 1'b1);
      `IFG_MAX  : state_ifg <= #`UD `IFG_IDLE;
    endcase
  else                        // half duplex mode
    case (state_ifg)
      `IFG_IDLE : if ( crs == 1'b0 )
                  begin
                    state_ifg <= #`UD `IFG_RUN;
                    count_ifg <= #`UD 5'd1;
                  end
                  else
                    count_ifg <= #`UD 5'd0;
      `IFG_RUN  : if ( (crs == 1'b1) && (count_ifg <= `IFG_LENGTH_1-1) )
                    state_ifg <= #`UD `IFG_IDLE;
                  else
                  if ( count_ifg == `IFG_LENGTH_2-1 )
                    state_ifg <= #`UD `IFG_MAX;
                  else
                    count_ifg <= #`UD (count_ifg + 1'b1);
      `IFG_MAX  : state_ifg <= #`UD `IFG_IDLE;
    endcase

always @(state_ifg)
  case (state_ifg)
    `IFG_IDLE : transmit_available_p = 1'b0;
    `IFG_RUN  : transmit_available_p = 1'b0;
    `IFG_MAX  : transmit_available_p = 1'b1;
    default   : transmit_available_p = 1'b0;
  endcase

endmodule