/*************************************************************************\
*                                                                         *
*    File    : frame_length_counter.v                                     *
*    Purpose : Frame Length Counter Module                                *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to count the length of frame has been *
*    transmitted to MII.                                                  *
*    The length counter counts the length of frame while transmit_enable  *
*    is detected and will be reseted if transmit_enable is not asserted.  *
*    If transmit_data_end is detected, the FCS counter will start         *
*    counting.                                                            *
*    This module also detects excessive length and condition when data    *
*    has been transmitted is 64 byte. Note, that the length counter       *
*    counts Preamble and SFD at the beginning of operation, which makes   *
*    the maximum length of frame and 64 byte indicator increased by 16.   *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module frame_length_counter (
  transmit_enable,
  transmit_data_end,
  clk,
  reset_n,
  count_length,
  count_fcs,
  excessive_length,
  transmit_64byte
  );

input transmit_enable;
input transmit_data_end;
input clk;
input reset_n;
output [11:0] count_length;
output [3:0] count_fcs;
output excessive_length;
output transmit_64byte;

reg [11:0] count_length;
reg [3:0] count_fcs;
reg [1:0] state_length;
reg excessive_length;
reg transmit_64byte;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    state_length <= #`UD `COUNT_LENGTH_IDLE;
  else
  case (state_length)
    `COUNT_LENGTH_IDLE : if (transmit_enable == 1'b1)
                         begin
                           transmit_64byte <= #`UD 1'b0;
                           excessive_length <= #`UD 1'b0;
                           count_length <= #`UD 12'd1;
                           state_length <= #`UD `COUNT_LENGTH_RUN;
                         end
                         else
                         begin
                           count_length <= #`UD 12'd0;
                           count_fcs <= #`UD 4'd0;
                         end
    `COUNT_LENGTH_RUN  : if (transmit_enable == 1'b0)
                           state_length <= #`UD `COUNT_LENGTH_IDLE;
                         else
                         if (transmit_data_end == 1'b1)
                         begin
                           transmit_64byte <= #`UD 
                             (count_length > `MIN_LENGTH-1) ? 1'b1 : 1'b0;
                           excessive_length <= #`UD 
                             (count_length > `MAX_LENGTH-1) ? 1'b1 : 1'b0;
                           state_length <= #`UD `COUNT_LENGTH_FCS;
                           count_length <= #`UD (count_length + 1'b1);
                         end
                         else
                           count_length <= #`UD (count_length + 1'b1);
    `COUNT_LENGTH_FCS  : if (transmit_enable == 1'b0)
                           state_length <= #`UD `COUNT_LENGTH_IDLE;
                         else
                         begin
                           transmit_64byte <= #`UD 
                             (count_length > `MIN_LENGTH-1) ? 1'b1 : 1'b0;
                           excessive_length <= #`UD 
                             (count_length > `MAX_LENGTH-1) ? 1'b1 : 1'b0;
                           count_length <= #`UD (count_length + 1'b1);
                           count_fcs <= #`UD (count_fcs + 1'b1);
                         end
    endcase

endmodule