/*************************************************************************\
*                                                                         *
*    File    : defer_counter.v                                            *
*    Purpose : Defer Counter Module                                       *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to limit defer time for waiting       *
*    opportunity to transmit a packet of data. If transmit_new_p pulse    *
*    is asserted, that indicates a new packet is ready to be transmitted, *
*    then counter start counting. When the counter has reached the        *
*    maximum value of defer time, excess_deferral will be asserted until  *
*    transmit_new_p appears again. Thus, the defer counter will be        *
*    reseted when transmit_new_p is asserted and will stop counting when  *
*    transmit_available_p is asserted. The maximum value of defer time is *
*    2 times of the maximum length of Ethernet frame, that is 3036 byte   *
*    or 6072 clock cycles.                                                *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module defer_counter (
  transmit_available_p, 
  transmit_new_p,
  clk, 
  reset_n, 
  excess_deferral
  );

input transmit_available_p;
input transmit_new_p;
input clk;
input reset_n;
output excess_deferral;

reg excess_deferral;

reg [12:0] count_defer;
reg [1:0] state_defer;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
  begin
    state_defer <= #`UD `DEFER_IDLE;
    count_defer <= #`UD 13'd0;
  end
  else
  case (state_defer)
    `DEFER_IDLE : if (transmit_new_p == 1'b1)
                  begin
                    state_defer <= #`UD `DEFER_RUN;
                    state_defer <= #`UD 1'b1;
                  end
    `DEFER_RUN  : if (count_defer == `MAX_DEFER)
                    state_defer <= #`UD `DEFER_MAX;
                  else
                  if (transmit_available_p == 1'b1)
                    state_defer <= #`UD `DEFER_IDLE;
                  else
                    count_defer <= #`UD (count_defer + 1'b1);
    `DEFER_MAX  : state_defer <= #`UD `DEFER_IDLE;
  endcase    

always @(state_defer)
  case (state_defer)
    `DEFER_IDLE : excess_deferral = excess_deferral;
    `DEFER_RUN  : excess_deferral = 1'b0;
    `DEFER_MAX  : excess_deferral = 1'b1;
    default     : excess_deferral = 1'b0;
  endcase

endmodule