/*************************************************************************\
*                                                                         *
*    File    : crc_gen.v                                                  *
*    Purpose : CRC Generator Module                                       *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to generate CRC number, which will    *
*    be added to Ethernet frame as FCS field.                             *
*    The CRC number is calculated from Destination Address field till     *
*    Data field, includes PAD bits if exists, and will be computed while  *
*    compute_crc is enabled. The Paralel CRC Algorithm is used to         *
*    calculate CRC number, and CRC is calculated from 4 bits of data      *
*    every one clock cycle.                                               *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module crc_gen (
  compute_crc,
  data,
  clk,
  reset_n,
  crc
  );

input compute_crc;
input [3:0] data;
input clk;
input reset_n;
output [3:0] crc;

reg [3:0] crc;
reg [1:0] state_crc;
reg [2:0] crc_stop;
reg crc_count;
reg [3:0] in_buffer;
reg [31:0] crc_buffer;

function [31:0] count_crc;  //parallel crc function
input [3:0] d;
input [31:0] c;
reg [3:0] d;          //input data
reg [31:0] c;         //previous crc
reg [31:0] new_crc;   //new crc
begin
  new_crc[0] = d[0] ^ c[28];
  new_crc[1] = d[1] ^ d[0] ^ c[28] ^ c[29];
  new_crc[2] = d[2] ^ d[1] ^ d[0] ^ c[28] ^ c[29] ^ c[30];
  new_crc[3] = d[3] ^ d[2] ^ d[1] ^ c[29] ^ c[30] ^ c[31];
  new_crc[4] = d[3] ^ d[2] ^ d[0] ^ c[0] ^ c[28] ^ c[30] ^ c[31];
  new_crc[5] = d[3] ^ d[1] ^ d[0] ^ c[1] ^ c[28] ^ c[29] ^ c[31];
  new_crc[6] = d[2] ^ d[1] ^ c[2] ^ c[29] ^ c[30];
  new_crc[7] = d[3] ^ d[2] ^ d[0] ^ c[3] ^ c[28] ^ c[30] ^ c[31];
  new_crc[8] = d[3] ^ d[1] ^ d[0] ^ c[4] ^ c[28] ^ c[29] ^ c[31];
  new_crc[9] = d[2] ^ d[1] ^ c[5] ^ c[29] ^ c[30];
  new_crc[10] = d[3] ^ d[2] ^ d[0] ^ c[6] ^ c[28] ^ c[30] ^ c[31];
  new_crc[11] = d[3] ^ d[1] ^ d[0] ^ c[7] ^ c[28] ^ c[29] ^ c[31];
  new_crc[12] = d[2] ^ d[1] ^ d[0] ^ c[8] ^ c[28] ^ c[29] ^ c[30];
  new_crc[13] = d[3] ^ d[2] ^ d[1] ^ c[9] ^ c[29] ^ c[30] ^ c[31];
  new_crc[14] = d[3] ^ d[2] ^ c[10] ^ c[30] ^ c[31];
  new_crc[15] = d[3] ^ c[11] ^ c[31];
  new_crc[16] = d[0] ^ c[12] ^ c[28];
  new_crc[17] = d[1] ^ c[13] ^ c[29];
  new_crc[18] = d[2] ^ c[14] ^ c[30];
  new_crc[19] = d[3] ^ c[15] ^ c[31];
  new_crc[20] = c[16];
  new_crc[21] = c[17];
  new_crc[22] = d[0] ^ c[18] ^ c[28];
  new_crc[23] = d[1] ^ d[0] ^ c[19] ^ c[28] ^ c[29];
  new_crc[24] = d[2] ^ d[1] ^ c[20] ^ c[29] ^ c[30];
  new_crc[25] = d[3] ^ d[2] ^ c[21] ^ c[30] ^ c[31];
  new_crc[26] = d[3] ^ d[0] ^ c[22] ^ c[28] ^ c[31];
  new_crc[27] = d[1] ^ c[23] ^ c[29];
  new_crc[28] = d[2] ^ c[24] ^ c[30];
  new_crc[29] = d[3] ^ c[25] ^ c[31];
  new_crc[30] = c[26];
  new_crc[31] = c[27];
  count_crc = new_crc;
end
endfunction

always @(posedge clk)
  in_buffer = data;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    state_crc <= #`UD `CRC_IDLE;
  else 
    case (state_crc)
      `CRC_IDLE  : if (compute_crc == 1'b1)
                     state_crc <= #`UD `CRC_FIRST;
                   else
                   begin
                     crc_stop <= #`UD 3'd0;
                     crc_count <= #`UD 1'b0;
                   end
      `CRC_FIRST : state_crc <= #`UD `CRC_RUN;
      `CRC_RUN   : if (compute_crc == 1'b0)
                     state_crc <= #`UD `CRC_STOP;
                   else
                     crc_count <= #`UD (crc_count + 1'b1);
      `CRC_STOP  : if (crc_stop == 3'd7)
                     state_crc <= #`UD `CRC_IDLE;
                   else
                     crc_stop <= #`UD (crc_stop + 1'b1);
    endcase

always @(state_crc or crc_stop or crc_count)
  case (state_crc)
    `CRC_IDLE  : begin           //reset buffer
                   crc_buffer = `CRC_RESET_VALUE;
                   crc = 4'h0;
                 end
    `CRC_FIRST : crc_buffer =
                 count_crc (in_buffer, `CRC_INIT_VALUE);   //load init value
    `CRC_RUN   : crc_buffer = count_crc (in_buffer, crc_buffer);
    `CRC_STOP  : case (crc_stop)
                   3'd0 : 
                     begin
                       crc_buffer = count_crc (in_buffer, crc_buffer);
                       crc = crc_buffer [31:28];
                     end
                   3'd1 : crc = crc_buffer [27:24];
                   3'd2 : crc = crc_buffer [23:20];
                   3'd3 : crc = crc_buffer [19:16];
                   3'd4 : crc = crc_buffer [15:12];   //store CRC to output
                   3'd5 : crc = crc_buffer [11:8];
                   3'd6 : crc = crc_buffer [7:4];
                   3'd7 : crc = crc_buffer [3:0];
                 endcase
    default    : crc = 4'h0;
  endcase

endmodule
