/*************************************************************************\
*                                                                         *
*    File    : coll_counter.v                                             *
*    Purpose : Collision Counter Module                                   *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The functions of this module are to count collision events, inspect  *
*    number of consecutive collisions, and detect late collision. If      *
*    collision appears while transmit_enable is asserted and coll_event_p *
*    is asserted. Then, the value of counter is increamented.             *
*    But, if collision is detected while Tx Eth MAC transmits Preamble or *
*    SFD, coll_event_p will be activated after all Preamble and SFD       *
*    patterns have been transmitted. If collision is detected when the    *
*    value of counter is 15, then excessive_coll is asserted and will be  *
*    deasserted when a new packet comes.                                  *
*    If collision is detected when transmit_64byte has been asserted by   *
*    Frame Length Counter, late collision occurs.                         *
*    When operates in full duplex mode, this module ignores any collision *
*    signals.                                                             *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module coll_counter (
  transmit_new_p,
  transmit_enable,
  transmit_preamble,
  transmit_sfd,
  transmit_64byte,
  clk,
  reset_n,
  coll,
  full_duplex,
  coll_event_p,
  late_coll,
  excessive_coll,
  coll_attempt
  );

input transmit_new_p;
input transmit_enable;
input transmit_preamble;
input transmit_sfd;
input transmit_64byte;
input clk;
input reset_n;
input coll;
input full_duplex;
output coll_event_p;
output late_coll;
output excessive_coll;
output [3:0] coll_attempt;

reg  coll_event_p;
reg  late_coll;
reg  excessive_coll;
reg  [3:0] coll_attempt;
reg  [1:0] state_coll;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    state_coll <= #`UD `COLL_RESTART;
  else
  if (full_duplex == 1'b1)  // full duplex mode, collision ignores
    case (state_coll)
      `COLL_RESTART : state_coll <= #`UD `COLL_RESTART;
    endcase
  else                      // half duplex mode
    case (state_coll)
      `COLL_IDLE    : if (transmit_new_p == 1'b1)
                        state_coll <= #`UD `COLL_RESTART;
                      else     // collision is detected
                      if ( (coll == 1'b1) &&
                           (transmit_preamble == 1'b0) &&
                           (transmit_sfd == 1'b0) &&
                           (transmit_enable == 1'b1) )
                        state_coll <= #`UD `COLL_EVENT;
      `COLL_RESTART : state_coll <= #`UD `COLL_IDLE;
      `COLL_EVENT   : state_coll <= #`UD `COLL_JAM;
      `COLL_JAM     : if (transmit_enable == 1'b1)
                        state_coll <= #`UD `COLL_JAM;
                      else
                        state_coll <= #`UD `COLL_IDLE;
    endcase

always @(state_coll)
  case (state_coll)
    `COLL_IDLE    : ;
    `COLL_RESTART : begin
                      coll_event_p = 1'b0;
                      late_coll = 1'b0;
                      excessive_coll = 1'b0;
                      coll_attempt = 4'd0;
                    end
    `COLL_EVENT   : begin
                      coll_event_p = 1'b1;
                      late_coll = transmit_64byte;
                      excessive_coll = (coll_attempt == `MAX_COLLISION) ?
                                       1'b1 : 1'b0;
                      coll_attempt = (coll_attempt == `MAX_COLLISION) ?
                                     coll_attempt : (coll_attempt + 1'b1);
                    end
    `COLL_JAM     : coll_event_p = 1'b0;
    default       : begin
                      coll_event_p = 1'b0;
                      late_coll = 1'b0;
                      excessive_coll = 1'b0;
                      coll_attempt = 4'd0;
                    end
  endcase
        
endmodule