/*************************************************************************\
*                                                                         *
*    File    : backoff_timer.v                                            *
*    Purpose : Backoff Timer Module                                       *
*    Author  : Novan Hartadi (novan@vlsi.itb.ac.id)                       *
*                                                                         *
/*************************************************************************\

/*************************************************************************\
*                                                                         *
*    Brief Description :                                                  *
*                                                                         *
*    The function of this module is to determine interval time for        *
*    backoff operation after a start_backoff is detected. This interval   *
*    time is calculated from multiplication between random number and     *
*    slot time, that is random x 128 clock cycles. This module takes      *
*    random number from Random Number Generator when start_backoff is     *
*    detected.                                                            *
*                                                                         *
\*************************************************************************/

`include "tx_var.v"

module backoff_timer (
  start_backoff,
  random,
  clk,
  reset_n,
  backoff_p
  );

input start_backoff;
input [9:0] random;
input clk;
input reset_n;
output backoff_p;

reg backoff_p;
reg [16:0] count_backoff;
reg [16:0] backoff_time;
reg [1:0] state_backoff;

always @(posedge clk or negedge reset_n)
  if (reset_n == 1'b0)
    state_backoff <= #`UD `BACKOFF_IDLE;
  else
    case (state_backoff)
      `BACKOFF_IDLE : if (start_backoff == 1'b1)
                      begin
                        state_backoff <= #`UD `BACKOFF_RUN;
                        backoff_time <= #`UD (random * `SLOT_TIME);
                      end
                      else
                        count_backoff <= #`UD 17'd0;
      `BACKOFF_RUN  : if (count_backoff == backoff_time)
                        state_backoff <= #`UD `BACKOFF_MAX;
                      else
                        count_backoff <= #`UD (count_backoff + 1'b1);
      `BACKOFF_MAX  : state_backoff <= #`UD `BACKOFF_IDLE;
  endcase

always @(state_backoff)
  case (state_backoff)
    `BACKOFF_IDLE : backoff_p = 1'b0;
    `BACKOFF_RUN  : backoff_p = 1'b0;
    `BACKOFF_MAX  : backoff_p = 1'b1;
    default       : backoff_p = 1'b0;
  endcase
       
endmodule