--==========================================================================--
-- Design units : 10/100 Ethernet MAC - RxMAC - RxFrameControl
--
-- File name    : rxframecontrol.vhd
--
-- Purpose      : The RxFrameControl module generates the RxValid signal. It
--                indicates that the current value of the RxDataRegister is a
--                valid frame data nibble. RxValid is also used to control
--                CRC checking and length counting.
--
-- Note         : None
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-09-04  Initial revision
--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

entity RxFrameControl is
  port(
    -- Receive clock and asynchronous reset
    RxReset_N:   in  Std_ULogic;
    RxClk:       in  Std_ULogic;
    -- Receive signals from RxSyncFIFO and RxDataRegister
    RxData:      in  Std_ULogic_Vector(3 downto 0);
    RxDataValid: in  Std_ULogic;
    RxFifoDV:    in  Std_ULogic;
    -- Receive frame data valid indicator
    RxValid:     out Std_ULogic
  );
end RxFrameControl;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

architecture RTL of RxFrameControl is

  ----------------------------------------------------------------------------
  -- The Starting-Frame-Delimiter (SFD) marks the beginning of the frame.
  ----------------------------------------------------------------------------
  constant SFD: Std_ULogic_Vector(3 downto 0) := "1101";

  ----------------------------------------------------------------------------
  -- Internel register to mark frame to be processed by RxMAC and Host.
  ----------------------------------------------------------------------------
  signal RxValidReg: Std_ULogic;

begin  -- Architecture RTL of RxFrameControl

  ----------------------------------------------------------------------------
  -- To detect the start of the frame this process watches the RxDataRegister
  -- and compares the stored value with the SFD. Then the frame data is valid
  -- until deassertion of RxFifoDV.
  ----------------------------------------------------------------------------
  WatchForFrame: process (RxReset_N, RxClk)
  begin
    if RxReset_N = '0' then
      RxValidReg <= '0';
    elsif Rising_Edge(RxClk) then
      if RxValidReg = '0' then
        if RxDataValid = '1' and RxData = SFD then
          RxValidReg <= '1';
        end if;
      else
        RxValidReg <= RxFifoDV;
      end if;
    end if;
  end process WatchForFrame;

  ----------------------------------------------------------------------------
  -- Propagate register contents to output and indicate valid frame data.
  ----------------------------------------------------------------------------
  RxValid <= RxValidReg;

end RTL;  -- End of RxFrameControl (RTL)
