-- PIC12C5XX modell
-- (C) 2000, Palasik Sandor

library IEEE;
use IEEE.STD_LOGIC_1164.all;

package PICTYPES is
  subtype BYTEREG is std_logic_vector (7 downto 0);
  subtype PROGADDR is std_logic_vector (9 downto 0);
  subtype REGADDR is std_logic_vector (4 downto 0);
  subtype PROGWORD is std_logic_vector (11 downto 0);

  constant ALU_WREG    : std_logic_vector := "0000";
  constant ALU_CLEAR   : std_logic_vector := "0001";
  constant ALU_SUB     : std_logic_vector := "0010";
  constant ALU_DEC     : std_logic_vector := "0011";
  constant ALU_OR      : std_logic_vector := "0100";
  constant ALU_AND     : std_logic_vector := "0101";
  constant ALU_XOR     : std_logic_vector := "0110";
  constant ALU_ADD     : std_logic_vector := "0111";
  constant ALU_FREG    : std_logic_vector := "1000";
  constant ALU_NOT     : std_logic_vector := "1001";
  constant ALU_INC     : std_logic_vector := "1010";
  constant ALU_DECFSZ  : std_logic_vector := "1011";
  constant ALU_RR      : std_logic_vector := "1100";
  constant ALU_RL      : std_logic_vector := "1101";
  constant ALU_SWAP    : std_logic_vector := "1110";
  constant ALU_INCFSZ  : std_logic_vector := "1111";

  constant OPC_BCF     : std_logic_vector := "0100";
  constant OPC_BSF     : std_logic_vector := "0101";
  constant OPC_BTFSC   : std_logic_vector := "0110";
  constant OPC_BTFSS   : std_logic_vector := "0111";
  constant OPC_RETLW   : std_logic_vector := "1000";
  constant OPC_CALL    : std_logic_vector := "1001";
  constant OPC_JMP     : std_logic_vector := "1010";
  constant OPC_JMP1    : std_logic_vector := "1011";
  constant OPC_MOVLW   : std_logic_vector := "1100";
  constant OPC_IORLW   : std_logic_vector := "1101";
  constant OPC_ANDLW   : std_logic_vector := "1110";
  constant OPC_XORLW   : std_logic_vector := "1111";

  -- Logiblox attributes
  attribute ASYNC_VAL: string;
  attribute BUS_WIDTH: string;
  attribute DEF: string;
  attribute DEPTH: string;
  attribute ENCODING: string;
  attribute LEVEL: string;
  attribute LIBVER: string;
  attribute MODTYPE: string;
  attribute OPTYPE: string;
  attribute REGISTERED: string;
  attribute STYLE: string;
  attribute USE_RPM: string;

  -- Logiblox modules
  component PC_REG
    port(
      D_IN: in std_logic_vector(9 downto 0);
      LOAD: in std_logic;
      CLOCK: in std_logic;
      ASYNC_CTRL: in std_logic;
      Q_OUT: out std_logic_vector(9 downto 0));
  end component;

  attribute DEF of PC_REG: component is "LOGIBLOX";
  attribute LEVEL of PC_REG: component is "XILINX";
  attribute LIBVER of PC_REG: component is "2.0.0";

  attribute MODTYPE of PC_REG: component is "COUNTER";
  attribute BUS_WIDTH of PC_REG: component is "10";
  attribute STYLE of PC_REG: component is "ALIGNED_RPM";
  attribute OPTYPE of PC_REG: component is "UP";
  attribute ENCODING of PC_REG: component is "BINARY";
  attribute ASYNC_VAL of PC_REG: component is "1023";

  component ADDSUB4
    port(
      ADD_SUB: in std_logic;
      C_IN: in std_logic;
      A: in std_logic_vector(3 downto 0);
      B: in std_logic_vector(3 downto 0);
      SUM: out std_logic_vector(3 downto 0);
      C_OUT: out std_logic);
  end component;

  attribute DEF of ADDSUB4: component is "LOGIBLOX";
  attribute LEVEL of ADDSUB4: component is "XILINX";
  attribute LIBVER of ADDSUB4: component is "2.0.0";

  attribute MODTYPE of ADDSUB4: component is "ADD_SUB";
  attribute BUS_WIDTH of ADDSUB4: component is "4";
  attribute OPTYPE of ADDSUB4: component is "ADD_SUB";
  attribute REGISTERED of ADDSUB4: component is "NONE";
  attribute ENCODING of ADDSUB4: component is "UNSIGNED";
  attribute STYLE of ADDSUB4: component is "ALIGNED_RPM";

  component RAM16X8
    port(
      A: in std_logic_vector(3 downto 0);
      DO: out std_logic_vector(7 downto 0);
      DI: in std_logic_vector(7 downto 0);
      WR_EN: in std_logic;
      WR_CLK: in std_logic);
  end component;

  attribute DEF of RAM16X8: component is "LOGIBLOX";
  attribute LEVEL of RAM16X8: component is "XILINX";
  attribute LIBVER of RAM16X8: component is "2.0.0";

  attribute MODTYPE of RAM16X8: component is "SYNC_RAM";
  attribute BUS_WIDTH of RAM16X8: component is "8";
  attribute DEPTH of RAM16X8: component is "16";
  attribute STYLE of RAM16X8: component is "MAX_SPEED";
  attribute USE_RPM of RAM16X8: component is "FALSE";

  component RAM32X8
    port(
      A: in std_logic_vector(4 downto 0);
      DO: out std_logic_vector(7 downto 0);
      DI: in std_logic_vector(7 downto 0);
      WR_EN: in std_logic;
      WR_CLK: in std_logic);
  end component;

  attribute DEF of RAM32X8: component is "LOGIBLOX";
  attribute LEVEL of RAM32X8: component is "XILINX";
  attribute LIBVER of RAM32X8: component is "2.0.0";

  attribute MODTYPE of RAM32X8: component is "SYNC_RAM";
  attribute BUS_WIDTH of RAM32X8: component is "8";
  attribute DEPTH of RAM32X8: component is "32";
  attribute STYLE of RAM32X8: component is "MAX_SPEED";
  attribute USE_RPM of RAM32X8: component is "FALSE";

  -- SPARTAN components
  component BUFGS port (
    I: in std_logic;
    O: out std_logic
  ); end component;

end PICTYPES;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity PICPC is port (
  Q1: in std_logic;
  RESET: in std_logic;

  PC_IN: in PROGADDR;
  PC_OUT: out PROGADDR;

  DO_GOTO: in std_logic;
  DO_CALL: in std_logic;
  DO_RETURN: in std_logic
  );
attribute CLOCK_BUFFER: boolean;
attribute CLOCK_BUFFER of Q1: signal is true;
end PICPC;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity PICQ4 is port (
  CLK: in std_logic;
  RESET: in std_logic;

  Q1_Q3: out std_logic;
  Q1_NOTQ3: out std_logic;
  Q2_NOTQ4: out std_logic
  );
attribute CLOCK_BUFFER: boolean;
attribute CLOCK_BUFFER of CLK: signal is true;
end PICQ4;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity PIC12C5X is port (
  CLK: in std_logic;
  MCLR: in std_logic;

  GPIO: inout BYTEREG;

  PROG_ADDR: out PROGADDR;
  PROG_DATA: in PROGWORD
  );
attribute CLOCK_BUFFER: boolean;
attribute CLOCK_BUFFER of CLK: signal is true;
end PIC12C5X;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity ALU_ADDSUB is port (
  Q4: in std_logic;
  WREG: in BYTEREG;
  FREG: in BYTEREG;
  OPCODE: in std_logic_vector(5 downto 0);

  RESULT: out BYTEREG;
  CARRYOUT: out std_logic;
  DCOUT: out std_logic;
  TRIS: out std_logic;
  MOD_DC: out std_logic;
  MOD_ZERO: out std_logic
); end ALU_ADDSUB;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity ALU_LOG is port (
  Q4: in std_logic;
  WREG: in BYTEREG;
  FREG: in BYTEREG;
  OPCODE: in std_logic_vector(5 downto 0);

  RESULT: out BYTEREG;
  TRIS: out std_logic;
  MOD_ZERO: out std_logic
); end ALU_LOG;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity ALU_SHIFT is port (
  Q4: in std_logic;
  FREG: in BYTEREG;
  CARRYIN: in std_logic;
  OPCODE: in std_logic_vector(5 downto 0);

  RESULT: out BYTEREG;
  CARRYOUT: out std_logic;
  TRIS: out std_logic
); end ALU_SHIFT;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity PICBITOP is port (
  Q4: in std_logic;
  BITIDX: in std_logic_vector(2 downto 0);
  OPCODE: in std_logic_vector(3 downto 0);
  FREG: in BYTEREG;

  RESULT: out BYTEREG;
  TRIS: out std_logic;
  HIT: out std_logic
); end PICBITOP;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity PICLIT is port (
  Q4: in std_logic;
  WREG: BYTEREG;
  KREG: BYTEREG;
  OPCODE: in std_logic_vector(3 downto 0);

  RESULT: out BYTEREG;
  TRIS: out std_logic;
  MOD_ZERO: out std_logic
); end PICLIT;

library IEEE;
use IEEE.STD_LOGIC_1164.all;
library METAMOR;
use METAMOR.ATTRIBUTES.all;
use METAMOR.ARRAY_ARITH.all;
library SYNOPSYS;
use SYNOPSYS.STD_LOGIC_ARITH.all;
use SYNOPSYS.STD_LOGIC_UNSIGNED.all;
use work.PICTYPES.all;

entity SPECINST is port (
  INSTR_EXECUTING: PROGWORD;
  CLRW: out std_logic;
  NOP: out std_logic;
  CLRWDT: out std_logic;
  OPTION: out std_logic;
  SLEEP: out std_logic;
  TRIS: out std_logic
); end SPECINST;

architecture PICQ4_ARCH of PICQ4 is

signal T_Q1_Q3,T_Q1_NOTQ3,T_Q2_NOTQ4: std_logic;

begin
process(CLK,RESET)
begin
if RESET='1' then
  T_Q1_Q3 <= '0';
  T_Q1_NOTQ3 <= '0';
  T_Q2_NOTQ4 <= '0';
elsif CLK'event and CLK='1' then
  T_Q1_Q3 <= not T_Q1_Q3; -- F/2

  if T_Q1_Q3='0' then
    T_Q1_NOTQ3 <= not T_Q1_NOTQ3; -- F/4
  end if;

  T_Q2_NOTQ4 <= T_Q1_NOTQ3; -- F/4
end if;
end process;

Q1_Q3 <= T_Q1_Q3;
Q1_NOTQ3 <= T_Q1_NOTQ3;
Q2_NOTQ4 <= T_Q2_NOTQ4;

end PICQ4_ARCH;

architecture PICPC_ARCH of PICPC is
signal PC,STACK1,STACK2,NEXT_PC: PROGADDR;
signal LOAD_PC,CE_PC: std_logic;

begin

PC_COUNTER: PC_REG port map(
        D_IN => NEXT_PC,
        Q_OUT => PC,
        LOAD => LOAD_PC,
	ASYNC_CTRL => RESET,
        CLOCK => Q1);

LOAD_PC <= DO_GOTO or DO_CALL or DO_RETURN;
NEXT_PC <= STACK1 when DO_RETURN='1'
	else PC_IN;

-- stack elements are NOT RESET
process(Q1)
begin
if Q1'event and Q1='1' then
  if DO_CALL='1' then
    STACK2 <= STACK1;
    STACK1 <= PC;
  elsif DO_RETURN='1' then
    STACK1 <= STACK2;
  end if;
end if;
end process;

PC_OUT <= PC;

end PICPC_ARCH;

architecture PICBITOP_ARCH_OPT of PICBITOP is
signal SET: std_logic;

signal T_INVBIT: BYTEREG;
signal T_ONE,T_HIT: std_logic;
attribute CRITICAL of T_INVBIT,T_ONE,T_HIT: signal is true;

begin

SET <= OPCODE(0);

process(SET,BITIDX,FREG,T_HIT,T_INVBIT,T_ONE)
  variable MASK: BYTEREG;
begin
  for I in RESULT'range loop
    if BITIDX=CONV_STD_LOGIC_VECTOR(I,RESULT'length) then
      T_INVBIT(I) <= not FREG(I);
    else
      T_INVBIT(I) <= '0';
    end if;
  end loop;

  if  T_INVBIT=(T_INVBIT'range => '0') then
    T_ONE <= '1';
  else
    T_ONE <= '0';
  end if;

  T_HIT <= SET xor T_ONE;

end process;

TRIS <= '0' when OPCODE(3 downto 1)="010" else '1';

process(Q4)
begin
if Q4'event and Q4='1' then
  HIT <= T_HIT;

  if T_HIT='0' then
    RESULT <= T_INVBIT xor FREG;
  else
    RESULT <= FREG;
  end if;
end if;
end process;
end PICBITOP_ARCH_OPT;

architecture ALU_ADDSUB_OPT of ALU_ADDSUB is

signal ADDSUB_B,T_RESULT: BYTEREG;
signal DO_ADD,CARRYIN: std_logic;
signal T_MOD_DC: std_logic;
signal DIGIT_CARRY: std_logic;

attribute CRITICAL of ADDSUB_B,
	DO_ADD,CARRYIN,T_MOD_DC,DIGIT_CARRY: signal is true;

signal T_TRIS,T_MOD_ZERO,T_CARRYOUT: std_logic;
attribute CRITICAL of T_TRIS,T_MOD_ZERO: signal is true;

begin

ADD_SUB_LSB: ADDSUB4 port map(
 ADD_SUB => DO_ADD,
 C_IN => CARRYIN,
 A => FREG(3 downto 0),
 B => ADDSUB_B(3 downto 0),
 SUM => T_RESULT(3 downto 0),
 C_OUT => DIGIT_CARRY);

ADD_SUB_MSB: ADDSUB4 port map(
 ADD_SUB => DO_ADD,
 C_IN => DIGIT_CARRY,
 A => FREG(7 downto 4),
 B => ADDSUB_B(7 downto 4),
 SUM => T_RESULT(7 downto 4),
 C_OUT => T_CARRYOUT);

process(T_TRIS,T_MOD_DC,OPCODE,WREG)
begin
  -- ADDER input
  for I in ADDSUB_B'range loop
    ADDSUB_B(I) <= T_MOD_DC and WREG(I);
  end loop;

  CARRYIN <= '-';
  DO_ADD <= '-';
  T_MOD_DC <= '0';
  T_MOD_ZERO <= '1';

  T_TRIS <= '0'; -- default, reset to 0, if no match for upper bits
  case OPCODE(3 downto 0) is
  when ALU_SUB => -- SUB
    DO_ADD <= '0';
    T_MOD_DC <= '1';
    CARRYIN <= '1';
  when ALU_ADD => -- ADD
    DO_ADD <= '1';
    T_MOD_DC <= '1';
    CARRYIN <= '0';
  when ALU_DEC => -- DEC
    DO_ADD <= '0';
    CARRYIN <= '0';
  when ALU_DECFSZ => -- DECFSZ
    DO_ADD <= '0';
    CARRYIN <= '0';
    T_MOD_ZERO <= '0';
  when ALU_INC => -- INC
    DO_ADD <= '1';
    CARRYIN <= '1';
  when ALU_INCFSZ => -- INCFSZ
    DO_ADD <= '1';
    CARRYIN <= '1';
    T_MOD_ZERO <= '0';
  when others =>
    T_TRIS <= '1';
    T_MOD_ZERO <= '0';
  end case;
  if OPCODE(5 downto 4)="00" then
    TRIS <= T_TRIS;
  else
    TRIS <= '1';
  end if;
end process;

process(Q4)
begin
if Q4'event and Q4='1' then
  RESULT <= T_RESULT;
  if OPCODE(5 downto 4)="00" then
    MOD_DC <= T_MOD_DC;
    MOD_ZERO <= T_MOD_ZERO;
    CARRYOUT <= T_CARRYOUT and T_MOD_DC;
  else
    MOD_DC <= '0';
    MOD_ZERO <= '0';
    CARRYOUT <= '0';
  end if;
end if;
end process;

DCOUT <= DIGIT_CARRY;
end ALU_ADDSUB_OPT;

architecture ALU_LOG_OPT of ALU_LOG is

type ONE_OPER_TYPE is (PASS_FREG,NOT_FREG,SWAP_FREG,ALL_ZERO);
type TWO_OPER_TYPE is (PASS_WREG,TWO_OR,TWO_AND,TWO_XOR);

signal T_TRIS,T_MOD_ZERO: std_logic;
signal IS_ONE: boolean;
signal ONE_OPER: ONE_OPER_TYPE;
signal TWO_OPER: TWO_OPER_TYPE;
signal ONE_RESULT,TWO_RESULT: BYTEREG;

attribute CRITICAL of T_TRIS,T_MOD_ZERO,IS_ONE,ONE_OPER,TWO_OPER,
	ONE_RESULT,TWO_RESULT: signal is true;

begin

-- output select
process(OPCODE,T_TRIS,T_MOD_ZERO)
begin
  case OPCODE(3 downto 0) is
    when ALU_WREG =>
      T_TRIS <= '0';
      T_MOD_ZERO <= '0';
    when ALU_CLEAR | ALU_OR | ALU_AND | ALU_XOR |
         ALU_FREG | ALU_NOT | ALU_SWAP =>
      T_TRIS <= '0';
      T_MOD_ZERO <= '1';
    when others => -- arithmetic, shift
      T_TRIS <= '1';
      T_MOD_ZERO <= '0';
  end case;

  if OPCODE(5 downto 4)="00" then
    TRIS <= T_TRIS;
  else
    TRIS <= '1';
  end if;
end process;

-- operator select
process(OPCODE)
begin
  ONE_OPER <= ALL_ZERO;
  TWO_OPER <= PASS_WREG;
  IS_ONE <= false;
  case OPCODE(3 downto 0) is

  when ALU_WREG =>
    TWO_OPER <= PASS_WREG;
  when ALU_OR =>
    TWO_OPER <= TWO_OR;
  when ALU_AND =>
    TWO_OPER <= TWO_AND;
  when ALU_XOR =>
    TWO_OPER <= TWO_XOR;

  when ALU_CLEAR =>
    ONE_OPER <= ALL_ZERO;
    IS_ONE <= true;
  when ALU_FREG =>
    ONE_OPER <= PASS_FREG;
    IS_ONE <= true;
  when ALU_NOT =>
    ONE_OPER <= NOT_FREG;
    IS_ONE <= true;
  when ALU_SWAP =>
    ONE_OPER <= SWAP_FREG;
    IS_ONE <= true;
  when others => -- arithmetic, shift
    null;
  end case;
end process;

-- two operands: FREG,WREG
process(WREG,FREG,TWO_OPER)
begin
  case TWO_OPER is
  when PASS_WREG =>
    TWO_RESULT <= WREG;
  when TWO_OR =>
    TWO_RESULT <= FREG or WREG;
  when TWO_AND =>
    TWO_RESULT <= FREG and WREG;
  when others => -- TWO_XOR
    TWO_RESULT <= FREG xor WREG;
  end case;
end process;

-- one operand: FREG
process(ONE_OPER,FREG)
begin
  case ONE_OPER is
  when PASS_FREG =>
    ONE_RESULT <= FREG;
  when NOT_FREG =>
    ONE_RESULT <= not FREG;
  when SWAP_FREG =>
    ONE_RESULT <= FREG(3 downto 0) & FREG(7 downto 4);
  when others => -- ALL_ZERO
    ONE_RESULT <= "00000000";
  end case;
end process;

process(Q4)
begin
if Q4'event and Q4='1' then
  if IS_ONE then
    RESULT <= ONE_RESULT;
  else
    RESULT <= TWO_RESULT;
  end if;
  if OPCODE(5 downto 4)="00" then
    MOD_ZERO <= T_MOD_ZERO;
  else
    MOD_ZERO <= '0';
  end if;
end if;
end process;
end ALU_LOG_OPT;

architecture ALU_SHIFT_ARCH of ALU_SHIFT is
signal T_TRIS: std_logic;
begin

T_TRIS <= '0' when OPCODE(5 downto 1)="00110" else '1';

process(Q4)
begin
if Q4'event and Q4='1' then
  if OPCODE(0)='0' then
    RESULT <= CARRYIN & FREG(7 downto 1);
    CARRYOUT <= FREG(0) and not T_TRIS;
  else
    RESULT <= FREG(6 downto 0) & CARRYIN;
    CARRYOUT <= FREG(7) and not T_TRIS;
  end if;
end if;
end process;

TRIS <= T_TRIS;

end ALU_SHIFT_ARCH;

architecture PICLIT_ARCH of PICLIT is
signal LIT_OPER: std_logic_vector(1 downto 0);
attribute CRITICAL of LIT_OPER:signal is true;

begin

process(OPCODE)
begin
  MOD_ZERO <= '1';
  case OPCODE is
    when OPC_IORLW => -- OR
      LIT_OPER <= "01";
    when OPC_ANDLW => -- AND
      LIT_OPER <= "10";
    when OPC_XORLW => -- XOR
      LIT_OPER <= "11";
    when others => -- PASS K
      LIT_OPER <= "00";
      MOD_ZERO <= '0';
 end case;
end process;

process(Q4)
begin
if Q4'event and Q4='1' then
  case LIT_OPER is
    when "01" => -- OR
      RESULT <= WREG or KREG;
    when "10" =>
      RESULT <= WREG and KREG;
    when "11" =>
      RESULT <= WREG xor KREG;
    when others => -- PASS K
      RESULT <= KREG;
 end case;
end if;
end process;

TRIS <= not OPCODE(3);
end PICLIT_ARCH;

architecture SPECINST_ARCH of SPECINST is

signal NOP_RANGE: std_logic;
attribute CRITICAL of NOP_RANGE: signal is true;

begin
process(INSTR_EXECUTING,NOP_RANGE)
begin
  if INSTR_EXECUTING(11 downto 5)="0000000"  then
    NOP_RANGE <= '1';
  else
    NOP_RANGE <= '0';
  end if;

  CLRWDT <= '0';
  OPTION <= '0';
  SLEEP <= '0';
  TRIS <= '0';

  if NOP_RANGE='1' then -- select low 3 bits from NOP range
    case INSTR_EXECUTING(2 downto 0) is
      when "010" =>
        OPTION <= '1';
      when "011" =>
        SLEEP <= '1';
      when "100" =>
        CLRWDT <= '1';
      when "110" =>
        TRIS <= '1';
      when others =>
        null;
      end case;
  end if;
end process;
end SPECINST_ARCH;

architecture PIC12C5X_ARCH of PIC12C5X is

signal FSR,REGFILE_MUX: std_logic_vector(5 downto 0);
signal REGFILE_GLOB,REGFILE_BANKED: BYTEREG;

signal WREG,TMR0,STATUS,OSCCAL,OPTION,TRIS: BYTEREG;
attribute CRITICAL of WREG,TMR0,STATUS,
	FSR,OSCCAL,GPIO,OPTION,TRIS: signal is true;

signal CARRY_FLAG,ZERO_FLAG,DC_FLAG,PA0: std_logic;
attribute CRITICAL of CARRY_FLAG,ZERO_FLAG,DC_FLAG,PA0: signal is true;

signal RESULT_ZERO: std_logic;

signal INSTR_EXECUTING: PROGWORD;
attribute CRITICAL of INSTR_EXECUTING: signal is true;

signal OPCODE: std_logic_vector(5 downto 0);

signal PC_IN: PROGADDR;
signal PC_OUT: PROGADDR;
attribute CRITICAL of PC_OUT: signal is true;

signal DO_GOTO, DO_CALL, DO_RETURN: std_logic;

-- 1 when next instruction treated as NOP
signal SKIPPING: std_logic;
attribute CRITICAL of SKIPPING: signal is true;

signal Q1_NOTQ3,Q2_NOTQ4,Q1,Q2,Q3,Q4: std_logic;

signal REGFILE_OUT,LIT_RESULT,BITOP_RESULT,ADDSUB_RESULT,OP_RESULT,
	LOG_RESULT,SHIFT_RESULT: BYTEREG;
attribute CRITICAL of REGFILE_OUT,LIT_RESULT,BITOP_RESULT,
	ADDSUB_RESULT,OP_RESULT,LOG_RESULT,SHIFT_RESULT: signal is true;

signal ADDSUB_TRIS,LOG_TRIS,SHIFT_TRIS,LIT_TRIS,BIT_TRIS: std_logic;
attribute CRITICAL of ADDSUB_TRIS,LOG_TRIS,SHIFT_TRIS,
	LIT_TRIS,BIT_TRIS: signal is true;

signal REGSEL: REGADDR;
signal BITIDX: std_logic_vector(2 downto 0);
signal REGDEST,BITOP_HIT,ADDSUB_CARRY,ADDSUB_DC,SHIFT_CARRY: std_logic;

signal GPIO_IN,GPIO_OUT: BYTEREG;

signal CLRWDT,WR_OPTION,SLEEP,WR_TRIS: std_logic;

signal IS_ALUOP,IS_GOTO,IS_LITOP,IS_RETLW: std_logic;

-- DECFSZ,INCFSZ, ALU on PC
signal COND_SKIP: std_logic;

signal MOD_ZERO,ADDSUB_MOD_ZERO,LOG_MOD_ZERO,LIT_MOD_ZERO: std_logic;
signal MOD_CARRY,MOD_DC,ADDSUB_MOD_DC: std_logic;

signal CE_ZERO,CE_DC,CE_CARRY: std_logic;

signal RESET: std_logic;

signal REGFILE_WEN,WREG_WEN,REGFILE_WEN_GLOB,REGFILE_WEN_BANKED: std_logic;

signal FSR_WEN,GPIO_WEN,PCL_WEN,STATUS_WEN,TMR0_WEN: std_logic;

begin

-- START

RESET <= not MCLR;

STATUS <= "00" & PA0 & "0" & ZERO_FLAG & DC_FLAG & CARRY_FLAG;

OPCODE <= INSTR_EXECUTING(11 downto 6);
REGSEL <= INSTR_EXECUTING(REGSEL'range);
REGDEST <= INSTR_EXECUTING(REGSEL'high+1);
BITIDX <= INSTR_EXECUTING(REGSEL'high+3 downto REGSEL'high+1);

MOD_Q4: entity work.PICQ4 port map(CLK => CLK, RESET => RESET,
  Q1_NOTQ3 => Q1_NOTQ3, Q2_NOTQ4 => Q2_NOTQ4);

Q1_BUF: BUFGS port map(I => Q1_NOTQ3,O => Q1);
Q2_BUF: BUFGS port map(I => Q2_NOTQ4,O => Q2);

Q3 <= not Q1;
Q4 <= not Q2;

MOD_PC : entity work.PICPC port map (Q1 => Q1,RESET => RESET,
  PC_IN => PC_IN,PC_OUT => PC_OUT,
  DO_GOTO => DO_GOTO,
  DO_CALL => DO_CALL, DO_RETURN => DO_RETURN);

MOD_LIT: entity work.PICLIT port map (Q4 => Q4,
  WREG => WREG,
  KREG => INSTR_EXECUTING(7 downto 0),
  OPCODE => OPCODE(5 downto 2),
  RESULT => LIT_RESULT,
  TRIS => LIT_TRIS,
  MOD_ZERO => LIT_MOD_ZERO);

MOD_BITOP: entity work.PICBITOP port map (Q4 => Q4,
  BITIDX => BITIDX,OPCODE => OPCODE(5 downto 2),
  FREG => REGFILE_OUT, RESULT => BITOP_RESULT,
  HIT => BITOP_HIT,
  TRIS => BIT_TRIS);

MOD_ADDSUB: entity work.ALU_ADDSUB port map (Q4 => Q4,
  WREG => WREG,FREG => REGFILE_OUT,
  OPCODE => OPCODE,
  RESULT => ADDSUB_RESULT,
  CARRYOUT => ADDSUB_CARRY,
  DCOUT => ADDSUB_DC,
  TRIS => ADDSUB_TRIS,
  MOD_DC => ADDSUB_MOD_DC,
  MOD_ZERO => ADDSUB_MOD_ZERO);

MOD_LOG: entity work.ALU_LOG port map (Q4 => Q4,
  WREG => WREG,FREG => REGFILE_OUT,
  OPCODE => OPCODE,
  RESULT => LOG_RESULT,
  TRIS => LOG_TRIS,
  MOD_ZERO => LOG_MOD_ZERO);

MOD_SHIFT: entity work.ALU_SHIFT port map (Q4 => Q4,
  FREG => REGFILE_OUT,
  CARRYIN => CARRY_FLAG,
  OPCODE => OPCODE,
  RESULT => SHIFT_RESULT,
  CARRYOUT => SHIFT_CARRY,
  TRIS => SHIFT_TRIS);

MOD_SPECINST: entity work.SPECINST port map (
  INSTR_EXECUTING => INSTR_EXECUTING,
  CLRWDT => CLRWDT,
  OPTION => WR_OPTION,
  SLEEP => SLEEP,
  TRIS => WR_TRIS);

OP_RESULT <= ADDSUB_RESULT when ADDSUB_TRIS='0' else (others => 'Z');
OP_RESULT <= LOG_RESULT when LOG_TRIS='0' else (others => 'Z');
OP_RESULT <= SHIFT_RESULT when SHIFT_TRIS='0' else (others => 'Z');
OP_RESULT <= BITOP_RESULT when BIT_TRIS='0' else (others => 'Z');
OP_RESULT <= LIT_RESULT when LIT_TRIS='0' else (others => 'Z');
RESULT_ZERO <= '1' when OP_RESULT="00000000" else '0';

process(Q1,RESET)
begin
if RESET='1' then
    INSTR_EXECUTING <= (others => '0');
    OPTION <= "11111111"; -- OPTION and TRIS resets to ALL ONES
    TRIS <= "11111111";
elsif Q1'event and Q1='1' then
  -- instruction pipeline
  for I in INSTR_EXECUTING'range loop
    INSTR_EXECUTING(I) <= not SKIPPING and PROG_DATA(I);
  end loop;

  if WR_TRIS='1' then
    TRIS <= OP_RESULT;
  end if;

  if WR_OPTION='1' then
    OPTION <= OP_RESULT;
  end if;

end if;
end process;

-- Q2 flipflops with NO RESET
process(Q2,RESET)
begin
if Q2'event and Q2='1' then
  GPIO_IN <= GPIO; -- input sampled
end if;
end process;

process(Q1,RESET)
begin
if Q1'event and Q1='1' then
  if WREG_WEN='1' then
    WREG <= OP_RESULT;
  end if;

  if TMR0_WEN='1' then
    TMR0 <= OP_RESULT;
  end if;

  if STATUS_WEN='1' then
    PA0 <= OP_RESULT(5);
  end if;

  if CE_ZERO='1' then
    if MOD_ZERO='1' then
      ZERO_FLAG <= RESULT_ZERO;
    else
      ZERO_FLAG <= OP_RESULT(2);
    end if;
  end if;

  if CE_DC='1' then
    if MOD_DC='1' then
      DC_FLAG <= ADDSUB_DC; -- flags override copy of OP_RESULT
    else
      DC_FLAG <= OP_RESULT(1);
    end if;
  end if;

  if CE_CARRY='1' then
    if MOD_CARRY='1' then
      CARRY_FLAG <= ADDSUB_CARRY or SHIFT_CARRY; -- flags override copy of OP_RESULT
    else
      CARRY_FLAG <= OP_RESULT(0);
    end if;
  end if;

  if FSR_WEN='1' then
    FSR <= OP_RESULT(FSR'range);
  end if;

  if GPIO_WEN='1' then
    GPIO_OUT <= OP_RESULT;
  end if;

end if;
end process;

process(OPCODE,BITOP_HIT,ZERO_FLAG,COND_SKIP,DO_RETURN,DO_GOTO,DO_CALL,RESULT_ZERO)
begin
  IS_ALUOP <= '0';
  IS_LITOP <= '0';
  case OPCODE(5 downto 4) is
    when "00" =>
      IS_ALUOP <= '1';
    when "11" =>
      IS_LITOP <= '1';
    when others => null;
  end case;

  DO_CALL <= '0';
  IS_GOTO <= '0';
  IS_RETLW <= '0';
  case OPCODE(5 downto 2) is
    when OPC_CALL =>
      DO_CALL <= '1';
    when OPC_JMP | OPC_JMP1 =>
      IS_GOTO <= '1';
    when OPC_RETLW =>
      IS_RETLW <= '1';
    when others => null;
  end case;

  COND_SKIP <= '0';

  if RESULT_ZERO='1' and (OPCODE="001011" or OPCODE="001111") then
    COND_SKIP <= '1'; -- DECFSZ, INCFSZ
  elsif BITOP_HIT='1' and OPCODE(5 downto 3)="011" then
    COND_SKIP <= '1'; -- BTFSC,BTFSS
  end if;

  SKIPPING <= DO_CALL or DO_GOTO or DO_RETURN or COND_SKIP;
end process;

MOD_ZERO <= ADDSUB_MOD_ZERO or LOG_MOD_ZERO or LIT_MOD_ZERO;
MOD_DC <= ADDSUB_MOD_DC;
MOD_CARRY <= (not SHIFT_TRIS) or ADDSUB_MOD_DC;

CE_ZERO <= MOD_ZERO or STATUS_WEN;
CE_DC <= MOD_DC or STATUS_WEN;
CE_CARRY <= MOD_CARRY or STATUS_WEN;

DO_GOTO <= IS_GOTO or PCL_WEN;
DO_RETURN <= IS_RETLW;

PROG_ADDR <= PC_OUT;

MOD_REGFILE_GLOB : RAM16X8 port map
(A => REGFILE_MUX(3 downto 0),
 DO => REGFILE_GLOB,
 DI => OP_RESULT,
 WR_EN => REGFILE_WEN_GLOB,
 WR_CLK => Q1);

MOD_REGFILE_BANKED : RAM32X8 port map
(A(4) => REGFILE_MUX(5), A(3 downto 0) => REGFILE_MUX(3 downto 0),
 DO => REGFILE_BANKED,
 DI => OP_RESULT,
 WR_EN => REGFILE_WEN_BANKED,
 WR_CLK => Q1);

REGFILE_WEN <= REGDEST and IS_ALUOP;
-- WREG <= WREG when NOP
WREG_WEN <= ((not REGDEST) and IS_ALUOP) or IS_LITOP or IS_RETLW; 

REGFILE_WEN_GLOB <= REGFILE_WEN and not REGFILE_MUX(4);
REGFILE_WEN_BANKED <= REGFILE_WEN and REGFILE_MUX(4);

PC_IN(7 downto 0) <= OP_RESULT; -- JMP, CALL => LIT_RESULT

PC_IN(8) <= INSTR_EXECUTING(8) when IS_GOTO='1' else '0';
PC_IN(9) <= PA0;

-- REGFILE_OUT: tristate bus
REGFILE_OUT <= REGFILE_BANKED when REGFILE_MUX(4)='1' else
	(others => 'Z');
REGFILE_OUT <= TMR0 when REGFILE_MUX(4 downto 0)="00001" else
	(others => 'Z');
REGFILE_OUT <= PC_OUT(7 downto 0) when REGFILE_MUX(4 downto 0)="00010" else
	(others => 'Z');
REGFILE_OUT <= STATUS when REGFILE_MUX(4 downto 0)="00011" else
	(others => 'Z');
REGFILE_OUT <= GPIO_IN when REGFILE_MUX(4 downto 0)="00110" else
	(others => 'Z');

REGFILE_OUT <= REGFILE_GLOB when not (
	(REGFILE_MUX(4)='1') or
	(REGFILE_MUX(4 downto 0)="00001") or
	(REGFILE_MUX(4 downto 0)="00010") or
	(REGFILE_MUX(4 downto 0)="00011") or
	(REGFILE_MUX(4 downto 0)="00110")
		) else (others => 'Z');

REGFILE_MUX <= FSR when REGSEL="00100" else PA0 & REGSEL;

-- SFR write enables
process(REGFILE_WEN,REGSEL)
begin
  TMR0_WEN <= '0';
  PCL_WEN <= '0';
  STATUS_WEN <= '0';
  FSR_WEN <= '0';
  GPIO_WEN <= '0';
  case REGSEL is
    when "00001" =>
      TMR0_WEN <= REGFILE_WEN;
    when "00010" =>
      PCL_WEN <= REGFILE_WEN;
    when "00011" =>
      STATUS_WEN <= REGFILE_WEN;
    when "00100" =>
      FSR_WEN <= REGFILE_WEN;
    when "00110" =>
      GPIO_WEN <= REGFILE_WEN;
    when others =>
      null;
  end case;
end process;

-- GPIO bit by bit output enables
process(TRIS,GPIO_OUT)
begin
  for I in GPIO'range loop
    if TRIS(I)='0' then
      GPIO(I) <= GPIO_OUT(I);
    else
      GPIO(I) <= 'Z';
    end if;
  end loop;
end process;
end PIC12C5X_ARCH;
