// ----------------------------------------------------------------------------
// FILE NAME      : xilinx_block_ram_8_32.v
// ----------------------------------------------------------------------------
// Release History
//   VERSION DATE     AUTHOR       DESCRIPTION
//   ------- -------- ------------ --------------------------------------------
//   0.00    08-07-01 John Clayton Copied and modified this file from
//                                 "xilinx_block_ram_16_32.v"
// ----------------------------------------------------------------------------
// PURPOSE: This module instantiates the Dual Port "block RAM" available within
//          the Xilinx Virtex or SpartanII FPGA.
//
//          All of the modules are dual-ported.  If you do not need one of the
//          ports, just connect the inputs to constants, and leave the outputs
//          unconnected.
//
//
//          These modules are "little endian."  That means that sequential
//          words are like this:
//
//          adr_i_s8    dat_i_s8          adr_i_s32     dat_o_s32
//          ------------------------      -----------------------
//                000   00                      000     03020100
//                001   01
//                002   02
//                003   03
//                004   04                      001     07060504
//                005   05
//                006   06
//                007   07
//
//
//          There are several modules:
//
//               ramb8_s8_s32   --  1024x8 and  256x32  ( 2 block rams)
//              ramb16_s8_s32   --  2048x8 and  512x32  ( 4 block rams)
//              ramb32_s8_s32   --  4096x8 and 1024x32  ( 8 block rams)
//              ramb64_s8_s32   --  8192x8 and 2048x32  (16 block rams)
//             ramb128_s8_s32   -- 16384x8 and 4096x32  (32 block rams)
//
// You can easily make larger modules, by just copying the last one, changing
// the name to reflect a larger size, adjusting the address lines to reflect
// one additional bit, and changing the two modules inside to be the previous
// size (which is half as large...)
// Of course, this method only handles sizes which are powers of 2 in size...
// Also, you must have enough block rams inside your part, obviously.
//
// ----------------------------------------------------------------------------
// Parameters
//   NAME                 RANGE    DESCRIPTION                DEFAULT
//   -------------------- -------- -------------------------- -----------------
//   None.
//
// ----------------------------------------------------------------------------
// REUSE ISSUES
//   Reset Strategy   : None
//   Clock Domains    : sys_clk 
//   Critical Timing  : None.
//   Test Features    : None
//   Asynchronous I/F : None
//   Scan Methodology : N/A
//   Instantiations   : RAMB4_S8_S16 FPGA Virtex Primitive
//   Other        :
// ----------------------------------------------------------------------------


//`resetall
//`timescale 1ns/100ps

// The following 'include' line must be used with Synplicity to create EDIF.
// The line must be commented for ModelSim.
//`include "C:\synplicity\synplify\lib\xilinx\virtex.v"


module ramb8_s8_s32(
dat_o_s8,
adr_i_s8,
dat_i_s8,
rst_i_s8,
we_i_s8,
clk_i_s8,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [7:0] dat_o_s8;             // 8-bit port data output
input [9:0] adr_i_s8;              // 8-bit port address
input [7:0] dat_i_s8;              // 8-bit port data input
input rst_i_s8;                    // 8-bit port reset
input we_i_s8;                     // 8-bit port write enable
input clk_i_s8;                    // 8-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [7:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [7:0] dat_o_s8_0;
wire [7:0] dat_o_s8_1;


// "Even half"
RAMB4_S8_S16 ram_0 (
  .DOA(dat_o_s8_0),
  .ADDRA({adr_i_s8[9:2],adr_i_s8[0]}),
  .CLKA(clk_i_s8),
  .DIA(dat_i_s8),
  .ENA(~adr_i_s8[1]),
  .RSTA(rst_i_s8),
  .WEA(we_i_s8),         // In combination with .ena
  .DOB(dat_o_s32[15:0]),
  .ADDRB(adr_i_s32),
  .CLKB(clk_i_s32),
  .DIB(dat_i_s32[15:0]),
  .ENB(1'b1),
  .RSTB(rst_i_s32),
  .WEB(we_i_s32)
  );


// "Odd half"
RAMB4_S8_S16 ram_1 (
  .DOA(dat_o_s8_1),
  .ADDRA({adr_i_s8[9:2],adr_i_s8[0]}),
  .CLKA(clk_i_s8),
  .DIA(dat_i_s8),
  .ENA(adr_i_s8[1]),
  .RSTA(rst_i_s8),
  .WEA(we_i_s8),         // In combination with .ena
  .DOB(dat_o_s32[31:16]),
  .ADDRB(adr_i_s32),
  .CLKB(clk_i_s32),
  .DIB(dat_i_s32[31:16]),
  .ENB(1'b1),
  .RSTB(rst_i_s32),
  .WEB(we_i_s32)
  );

// This mux selects which 8-bit data is read from the block.
assign dat_o_s8 = adr_i_s8[1]?dat_o_s8_1:dat_o_s8_0;


// The defparam initializes memory contents only for simulation.
// The "synopsys translate off/on" statements cause the synthesis tool
// (from Synopsys) to ignore the defparams...
// I left these in here so that simulations would start out with a known
// state for the memory contents of all modules in this file... (All
// of the block RAM modules in this file are build from this basic module.)

// synopsys translate_off
defparam ram_0.INIT_00 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_01 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_02 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_03 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_04 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_05 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_06 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_07 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_08 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_09 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0A =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0B =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0C =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0D =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0E =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0F =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
  
defparam ram_1.INIT_00 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_01 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_02 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_03 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_04 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_05 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_06 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_07 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_08 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_09 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0A =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0B =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0C =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0D =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0E =
   256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0F =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
//synopsys translate_on

// In order to initialize the memory in actual synthesis, using Synposys,
// the following "set attributes" do the job...  (Leave the "synopsys
// dc_script_begin" and "synopsys_dc_script_end" as comments...)
// Since you would want to modify the contents of your memory differently for
// different designs, do not set the contents inside of this file...  Instead
// use a hierarchical name, and do it inside of a dc_script from the top level
// of your design (or where ever it makes the most sense!)
//
// //synopsys dc_script_begin
// //set_attribute ram_0 INIT_00
// "0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF" -type string
// //set_attribute ram_0 INIT_01
// "0000111122223333444455556666777788889999AAAABBBBCCCCDDDDEEEEFFFF" -type string
// //synopsys dc_script_end


endmodule



// ----------------------------------------------------------------------------

module ramb16_s8_s32(
dat_o_s8,
adr_i_s8,
dat_i_s8,
rst_i_s8,
we_i_s8,
clk_i_s8,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [7:0] dat_o_s8;              // 8-bit port data output
input [10:0] adr_i_s8;              // 8-bit port address
input [7:0] dat_i_s8;               // 8-bit port data input
input rst_i_s8;                     // 8-bit port reset
input we_i_s8;                      // 8-bit port write enable
input clk_i_s8;                     // 8-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [8:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [7:0] dat_o_s8_0;
wire [7:0] dat_o_s8_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb8_s8_s32 ram_0 (
  .dat_o_s8(dat_o_s8_0),
  .adr_i_s8(adr_i_s8[9:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && ~adr_i_s8[10]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[7:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[8])
  );


// "Odd half"
ramb8_s8_s32 ram_1 (
  .dat_o_s8(dat_o_s8_1),
  .adr_i_s8(adr_i_s8[9:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && adr_i_s8[10]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[7:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[8])
  );

// This mux selects which 8-bit data is read from the block.
assign dat_o_s8 = adr_i_s8[10]?dat_o_s8_1:dat_o_s8_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[8]?dat_o_s32_1:dat_o_s32_0;



endmodule



// ----------------------------------------------------------------------------

module ramb32_s8_s32(
dat_o_s8,
adr_i_s8,
dat_i_s8,
rst_i_s8,
we_i_s8,
clk_i_s8,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [7:0] dat_o_s8;              // 8-bit port data output
input [11:0] adr_i_s8;              // 8-bit port address
input [7:0] dat_i_s8;               // 8-bit port data input
input rst_i_s8;                     // 8-bit port reset
input we_i_s8;                      // 8-bit port write enable
input clk_i_s8;                     // 8-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [9:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [7:0] dat_o_s8_0;
wire [7:0] dat_o_s8_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb16_s8_s32 ram_0 (
  .dat_o_s8(dat_o_s8_0),
  .adr_i_s8(adr_i_s8[10:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && ~adr_i_s8[11]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[8:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[9])
  );


// "Odd half"
ramb16_s8_s32 ram_1 (
  .dat_o_s8(dat_o_s8_1),
  .adr_i_s8(adr_i_s8[10:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && adr_i_s8[11]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[8:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[9])
  );

// This mux selects which 8-bit data is read from the block.
assign dat_o_s8 = adr_i_s8[11]?dat_o_s8_1:dat_o_s8_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[9]?dat_o_s32_1:dat_o_s32_0;



endmodule



// ----------------------------------------------------------------------------

module ramb64_s8_s32(
dat_o_s8,
adr_i_s8,
dat_i_s8,
rst_i_s8,
we_i_s8,
clk_i_s8,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [7:0] dat_o_s8;              // 8-bit port data output
input [12:0] adr_i_s8;              // 8-bit port address
input [7:0] dat_i_s8;               // 8-bit port data input
input rst_i_s8;                     // 8-bit port reset
input we_i_s8;                      // 8-bit port write enable
input clk_i_s8;                     // 8-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [10:0] adr_i_s32;             // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [7:0] dat_o_s8_0;
wire [7:0] dat_o_s8_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb32_s8_s32 ram_0 (
  .dat_o_s8(dat_o_s8_0),
  .adr_i_s8(adr_i_s8[11:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && ~adr_i_s8[12]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[9:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[10])
  );


// "Odd half"
ramb32_s8_s32 ram_1 (
  .dat_o_s8(dat_o_s8_1),
  .adr_i_s8(adr_i_s8[11:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && adr_i_s8[12]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[9:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[10])
  );

// This mux selects which 8-bit data is read from the block.
assign dat_o_s8 = adr_i_s8[12]?dat_o_s8_1:dat_o_s8_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[10]?dat_o_s32_1:dat_o_s32_0;



endmodule



// ----------------------------------------------------------------------------

module ramb128_s8_s32(
dat_o_s8,
adr_i_s8,
dat_i_s8,
rst_i_s8,
we_i_s8,
clk_i_s8,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [7:0] dat_o_s8;              // 8-bit port data output
input [13:0] adr_i_s8;              // 8-bit port address
input [7:0] dat_i_s8;               // 8-bit port data input
input rst_i_s8;                     // 8-bit port reset
input we_i_s8;                      // 8-bit port write enable
input clk_i_s8;                     // 8-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [11:0] adr_i_s32;             // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [7:0] dat_o_s8_0;
wire [7:0] dat_o_s8_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb64_s8_s32 ram_0 (
  .dat_o_s8(dat_o_s8_0),
  .adr_i_s8(adr_i_s8[12:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && ~adr_i_s8[13]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[10:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[11])
  );


// "Odd half"
ramb64_s8_s32 ram_1 (
  .dat_o_s8(dat_o_s8_1),
  .adr_i_s8(adr_i_s8[12:0]),
  .clk_i_s8(clk_i_s8),
  .dat_i_s8(dat_i_s8),
  .rst_i_s8(rst_i_s8),
  .we_i_s8(we_i_s8 && adr_i_s8[13]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[10:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[11])
  );

// This mux selects which 8-bit data is read from the block.
assign dat_o_s8 = adr_i_s8[13]?dat_o_s8_1:dat_o_s8_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[11]?dat_o_s32_1:dat_o_s32_0;



endmodule



