// ----------------------------------------------------------------------------
// FILE NAME      : xilinx_block_ram_16_32.v
// ----------------------------------------------------------------------------
// Release History
//   VERSION DATE     AUTHOR       DESCRIPTION
//   ------- -------- ------------ --------------------------------------------
//   0.00    08-07-01 John Clayton Adapted this file to work with Synposys,
//                                 by adding "//synposys translate on/off"
//   0.01    08-07-01 John Clayton Changed module naming.
//   0.02    08-07-01 John Clayton Added larger and larger modules.
//                                 Added comments about "set attribute" usage.
// ----------------------------------------------------------------------------
// PURPOSE: This module instantiates the Dual Port "block RAM" available within
//          the Xilinx Virtex or SpartanII FPGA.
//
//          All of the modules are dual-ported.  If you do not need one of the
//          ports, just connect the inputs to constants, and leave the outputs
//          unconnected.  They will show up as "dangling" in the Xilinx place
//          and route software, but it is only a warning, and the blocks still
//          work fine.  If you don't like that warning, build some new memory
//          blocks which instantiate single-ported constructs.
//
//          These modules are "little endian."  That means that sequential
//          words are like this:
//
//          adr_i_s16   dat_i_s16         adr_i_s32     dat_o_s32
//          ------------------------      -----------------------
//                000   0100                    000     03020100
//                001   0302
//                002   0504                    001     07060504
//                003   0706
//
//
//          There are several modules:
//
//               ramb8_s16_s32   --  512x16 and  256x32  ( 2 block rams)
//              ramb16_s16_s32   -- 1024x16 and  512x32  ( 4 block rams)
//              ramb32_s16_s32   -- 2048x16 and 1024x32  ( 8 block rams)
//              ramb64_s16_s32   -- 4096x16 and 2048x32  (16 block rams)
//             ramb128_s16_s32   -- 8192x16 and 4096x32  (32 block rams)
//
// You can easily make larger modules, by just copying the last one, changing
// the name to reflect a larger size, adjusting the address lines to reflect
// one additional bit, and changing the two modules inside to be the previous
// size (which is half as large...)
// Of course, this method only handles sizes which are powers of 2 in size...
// Also, you must have enough block rams inside your part, obviously.
//
// ----------------------------------------------------------------------------
// Parameters
//   NAME                 RANGE    DESCRIPTION                DEFAULT
//   -------------------- -------- -------------------------- -----------------
//   None.
//
// ----------------------------------------------------------------------------
// REUSE ISSUES
//   Reset Strategy   : None
//   Clock Domains    : sys_clk 
//   Critical Timing  : None.
//   Test Features    : None
//   Asynchronous I/F : None
//   Scan Methodology : N/A
//   Instantiations   : RAMB4_S16_S16 FPGA Virtex Primitive
//   Other        :
// ----------------------------------------------------------------------------


//`resetall
//`timescale 1ns/100ps

// The following 'include' line must be used with Synplicity to create EDIF.
// The line must be commented for ModelSim.
//`include "C:\synplicity\synplify\lib\xilinx\virtex.v"


module ramb8_s16_s32(
dat_o_s16,
adr_i_s16,
dat_i_s16,
rst_i_s16,
we_i_s16,
clk_i_s16,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [15:0] dat_o_s16;            // 16-bit port data output
input [8:0] adr_i_s16;              // 16-bit port address
input [15:0] dat_i_s16;             // 16-bit port data input
input rst_i_s16;                    // 16-bit port reset
input we_i_s16;                     // 16-bit port write enable
input clk_i_s16;                    // 16-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [7:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [15:0] dat_o_s16_0;
wire [15:0] dat_o_s16_1;


// "Even half"
RAMB4_S16_S16 ram_0 (
  .DOA(dat_o_s16_0),
  .ADDRA(adr_i_s16[8:1]),
  .CLKA(clk_i_s16),
  .DIA(dat_i_s16),
  .ENA(~adr_i_s16[0]),
  .RSTA(rst_i_s16),
  .WEA(we_i_s16),         // In combination with .ena
  .DOB(dat_o_s32[15:0]),
  .ADDRB(adr_i_s32),
  .CLKB(clk_i_s32),
  .DIB(dat_i_s32[15:0]),
  .ENB(1'b1),
  .RSTB(rst_i_s32),
  .WEB(we_i_s32)
  );


// "Odd half"
RAMB4_S16_S16 ram_1 (
  .DOA(dat_o_s16_1),
  .ADDRA(adr_i_s16[8:1]),
  .CLKA(clk_i_s16),
  .DIA(dat_i_s16),
  .ENA(adr_i_s16[0]),
  .RSTA(rst_i_s16),
  .WEA(we_i_s16),         // In combination with .ena
  .DOB(dat_o_s32[31:16]),
  .ADDRB(adr_i_s32),
  .CLKB(clk_i_s32),
  .DIB(dat_i_s32[31:16]),
  .ENB(1'b1),
  .RSTB(rst_i_s32),
  .WEB(we_i_s32)
  );

// This mux selects which 16-bit data is read from the block.
assign dat_o_s16 = adr_i_s16[0]?dat_o_s16_1:dat_o_s16_0;


// The defparam initializes memory contents only for simulation.
// The "synopsys translate off/on" statements cause the synthesis tool
// (from Synopsys) to ignore the defparams...
// I left these in here so that simulations would start out with a known
// state for the memory contents of all modules in this file... (All
// of the block RAM modules in this file are build from this basic module.)

// synopsys translate_off
defparam ram_0.INIT_00 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_01 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_02 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_03 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_04 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_05 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_06 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_07 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_08 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_09 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0A =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0B =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0C =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0D =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0E =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_0.INIT_0F =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
  
defparam ram_1.INIT_00 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_01 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_02 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_03 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_04 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_05 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_06 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_07 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_08 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_09 =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0A =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0B =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0C =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0D =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0E =
   256'h0000000000000000000000000000000000000000000000000000000000000000;
defparam ram_1.INIT_0F =
  256'h0000000000000000000000000000000000000000000000000000000000000000;
//synopsys translate_on

// In order to initialize the memory in actual synthesis, using Synposys,
// the following "set attributes" do the job...  (Leave the "synopsys
// dc_script_begin" and "synopsys_dc_script_end" as comments...)
// Since you would want to modify the contents of your memory differently for
// different designs, do not set the contents inside of this file...  Instead
// use a hierarchical name, and do it inside of a dc_script from the top level
// of your design (or where ever it makes the most sense!)
//
// //synopsys dc_script_begin
// //set_attribute ram_0 INIT_00
// "0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF" -type string
// //set_attribute ram_0 INIT_01
// "0000111122223333444455556666777788889999AAAABBBBCCCCDDDDEEEEFFFF" -type string
// //synopsys dc_script_end


endmodule



// ----------------------------------------------------------------------------

module ramb16_s16_s32(
dat_o_s16,
adr_i_s16,
dat_i_s16,
rst_i_s16,
we_i_s16,
clk_i_s16,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [15:0] dat_o_s16;            // 16-bit port data output
input [9:0] adr_i_s16;              // 16-bit port address
input [15:0] dat_i_s16;             // 16-bit port data input
input rst_i_s16;                    // 16-bit port reset
input we_i_s16;                     // 16-bit port write enable
input clk_i_s16;                    // 16-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [8:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [15:0] dat_o_s16_0;
wire [15:0] dat_o_s16_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb8_s16_s32 ram_0 (
  .dat_o_s16(dat_o_s16_0),
  .adr_i_s16(adr_i_s16[8:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && ~adr_i_s16[9]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[7:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[8])
  );


// "Odd half"
ramb8_s16_s32 ram_1 (
  .dat_o_s16(dat_o_s16_1),
  .adr_i_s16(adr_i_s16[8:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && adr_i_s16[9]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[7:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[8])
  );

// This mux selects which 16-bit data is read from the block.
assign dat_o_s16 = adr_i_s16[9]?dat_o_s16_1:dat_o_s16_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[8]?dat_o_s32_1:dat_o_s32_0;



endmodule


// ----------------------------------------------------------------------------

module ramb32_s16_s32(
dat_o_s16,
adr_i_s16,
dat_i_s16,
rst_i_s16,
we_i_s16,
clk_i_s16,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [15:0] dat_o_s16;            // 16-bit port data output
input [10:0] adr_i_s16;             // 16-bit port address
input [15:0] dat_i_s16;             // 16-bit port data input
input rst_i_s16;                    // 16-bit port reset
input we_i_s16;                     // 16-bit port write enable
input clk_i_s16;                    // 16-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [9:0] adr_i_s32;              // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [15:0] dat_o_s16_0;
wire [15:0] dat_o_s16_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb16_s16_s32 ram_0 (
  .dat_o_s16(dat_o_s16_0),
  .adr_i_s16(adr_i_s16[9:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && ~adr_i_s16[10]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[8:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[9])
  );


// "Odd half"
ramb16_s16_s32 ram_1 (
  .dat_o_s16(dat_o_s16_1),
  .adr_i_s16(adr_i_s16[9:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && adr_i_s16[10]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[8:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[9])
  );

// This mux selects which 16-bit data is read from the block.
assign dat_o_s16 = adr_i_s16[10]?dat_o_s16_1:dat_o_s16_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[9]?dat_o_s32_1:dat_o_s32_0;



endmodule


// ----------------------------------------------------------------------------

module ramb64_s16_s32(
dat_o_s16,
adr_i_s16,
dat_i_s16,
rst_i_s16,
we_i_s16,
clk_i_s16,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [15:0] dat_o_s16;            // 16-bit port data output
input [11:0] adr_i_s16;             // 16-bit port address
input [15:0] dat_i_s16;             // 16-bit port data input
input rst_i_s16;                    // 16-bit port reset
input we_i_s16;                     // 16-bit port write enable
input clk_i_s16;                    // 16-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [10:0] adr_i_s32;             // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [15:0] dat_o_s16_0;
wire [15:0] dat_o_s16_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb32_s16_s32 ram_0 (
  .dat_o_s16(dat_o_s16_0),
  .adr_i_s16(adr_i_s16[10:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && ~adr_i_s16[11]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[9:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[10])
  );


// "Odd half"
ramb32_s16_s32 ram_1 (
  .dat_o_s16(dat_o_s16_1),
  .adr_i_s16(adr_i_s16[10:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && adr_i_s16[11]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[9:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[10])
  );

// This mux selects which 16-bit data is read from the block.
assign dat_o_s16 = adr_i_s16[11]?dat_o_s16_1:dat_o_s16_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[10]?dat_o_s32_1:dat_o_s32_0;



endmodule


// ----------------------------------------------------------------------------

module ramb128_s16_s32(
dat_o_s16,
adr_i_s16,
dat_i_s16,
rst_i_s16,
we_i_s16,
clk_i_s16,
dat_o_s32,
adr_i_s32,
dat_i_s32,
rst_i_s32,
we_i_s32,
clk_i_s32
);
  
// I/O Declarations
output [15:0] dat_o_s16;            // 16-bit port data output
input [12:0] adr_i_s16;             // 16-bit port address
input [15:0] dat_i_s16;             // 16-bit port data input
input rst_i_s16;                    // 16-bit port reset
input we_i_s16;                     // 16-bit port write enable
input clk_i_s16;                    // 16-bit port clock

output [31:0] dat_o_s32;            // 32-bit port data output
input [11:0] adr_i_s32;             // 32-bit port address
input [31:0] dat_i_s32;             // 32-bit port data input
input rst_i_s32;                    // 32-bit port reset
input we_i_s32;                     // 32-bit port write enable
input clk_i_s32;                    // 32-bit port clock


// Local signals
wire [15:0] dat_o_s16_0;
wire [15:0] dat_o_s16_1;
wire [31:0] dat_o_s32_0;
wire [31:0] dat_o_s32_1;


// "Even half"
ramb64_s16_s32 ram_0 (
  .dat_o_s16(dat_o_s16_0),
  .adr_i_s16(adr_i_s16[11:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && ~adr_i_s16[12]),
  .dat_o_s32(dat_o_s32_0),
  .adr_i_s32(adr_i_s32[10:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && ~adr_i_s32[11])
  );


// "Odd half"
ramb64_s16_s32 ram_1 (
  .dat_o_s16(dat_o_s16_1),
  .adr_i_s16(adr_i_s16[11:0]),
  .clk_i_s16(clk_i_s16),
  .dat_i_s16(dat_i_s16),
  .rst_i_s16(rst_i_s16),
  .we_i_s16(we_i_s16 && adr_i_s16[12]),
  .dat_o_s32(dat_o_s32_1),
  .adr_i_s32(adr_i_s32[10:0]),
  .clk_i_s32(clk_i_s32),
  .dat_i_s32(dat_i_s32),
  .rst_i_s32(rst_i_s32),
  .we_i_s32(we_i_s32 && adr_i_s32[11])
  );

// This mux selects which 16-bit data is read from the block.
assign dat_o_s16 = adr_i_s16[12]?dat_o_s16_1:dat_o_s16_0;
// This mux selects which 32-bit data is read from the block.
assign dat_o_s32 = adr_i_s32[11]?dat_o_s32_1:dat_o_s32_0;



endmodule

