//-----------------------------------------------------------------------------
//
// Author: John Clayton
// Date  : August 9, 2001
// Update: August 9, 2001 Obtained this file from "build_7" project.
// Update: Aug.  10, 2001 Added ram16_s8_s32 block.  Debugged until it works.
//
// Description
//-----------------------------------------------------------------------------
// This targets an XC2S200 board which was created for educational purposes.
//
// There are:
//    8  LEDs (led[7:0])
//    4  switches (switch[3:0])
//    1  clock of 32.000 MHz clock, present on GCLK1
//    1  clock of 49.152 MHz clock, present on GCLK0
//    4  lines of ps2 clock input (port A in documentation notes)
//    4  lines of ps2 data input (port A in documentation notes)
//    16 lines of LCD panel control (port B in documentation notes)
//    2  lines of rs232 serial connection (port C in documentation notes)
//-----------------------------------------------------------------------------

// NOTE: This build is for testing DPRAM memory blocks, to make sure that they
//       are connected up properly in a dual-ported configuration.  The
//       Xilinx block RAMs are being clocked by clk_s, but they can be run
//       much faster...
//
//       These rams are configured for "little endian" transformation.
//
//       That is:  Sequential bytes, 00, 01, 02, 03 on the s8 side emerge
//                 on the s16 side:  01000302.
//                 on the s32 side:  03020100.
//
//                 Sequential words, 0100, 0302 on the s16 side emerge
//                 on the s32 side:  03020100.
//
//       This is called "little endian".  It can be quite confusing at times!

`resetall
`timescale 1ns/100ps

module top (
  sys_clk_0,
  sys_clk_1,
  switch,
  led,
  ps2_clk,
  ps2_data,
  lcd_drive,
  rs232_rxd,
  rs232_txd,
  debug_o
  );


output [14:0] debug_o;

// I/O declarations
input sys_clk_0;      // 49.152 MHz
input sys_clk_1;      // 32.000 MHz
input [3:0] switch;
input rs232_rxd;

inout [3:0] ps2_clk;
inout [3:0] ps2_data;

output [7:0] led;

output [15:0] lcd_drive;
output rs232_txd;

// Internal signal declarations

reg [9:0] crosshair_x_pos;
reg [9:0] crosshair_y_pos;

wire [7:0] lcd_control;

reg clk_s;               // 49.152/2 MHz = 24.576 MHz

wire [15:0] adr;
wire [15:0] dat;
wire we;
wire stb;
wire rst;
wire sel_1;
wire master_br;

wire ram_sel_1;
wire ram_sel_2;
wire [15:0] ram_s16_dat_o;
wire [7:0] ram_s8_dat_o;
wire [31:0] ram_s32_dat_o;
wire [8:0] ram_s32_adr;

wire reset = switch[0];  // Simply a renaming exercise

//--------------------------------------------------------------------------
// Clock generation
//--------------------------------------------------------------------------

// This uses up an additional GCLK resource.
always @(posedge sys_clk_0)
begin
  clk_s <= ~clk_s;
end

//--------------------------------------------------------------------------
// Instantiations
//--------------------------------------------------------------------------

lcd_test_pongball_crosshair lcd_block (
  .sys_clk(sys_clk_1),
  .lcd_clk(sys_clk_0),
  .sys_reset(lcd_control[1]),
  .lcd_reset(lcd_control[0]),
  .ball_x_enable(lcd_control[2]),
  .ball_y_enable(lcd_control[3]),
  .crosshair_x_pos(crosshair_x_pos),
  .crosshair_y_pos(crosshair_y_pos),
  .crosshair_color(lcd_control[6:4]),
  .lcd_drive(lcd_drive)
  );

rs232_syscon #(
               4,             // Number of Hex digits for addresses.
               4,             // Number of Hex digits for data.
               2,             // Number of Hex digits for quantity.
               16,            // Characters in the input buffer
               4,             // Bits in the buffer pointer
               255,           // Clocks before watchdog timer expires
               8,             // Bits in watchdog timer
               8,             // Number of data fields displayed per line
               3,             // Number of bits in the fields counter
               2              // Number of bits in the digits counter
               )
  syscon_1 (                  // instance name
  .clk_i(clk_s),
  .reset_i(reset),
  .master_bg_i(master_br),
  .ack_i(switch[1]),
  .err_i(switch[2]),
  .master_adr_i(16'b0),
  .rs232_rxd_i(rs232_rxd),
  .dat_io(dat),
  .rst_o(rst),
  .master_br_o(master_br),
  .stb_o(stb),
  .cyc_o(),
  .adr_o(adr),
  .we_o(we),
  .rs232_txd_o(rs232_txd)
  );


assign sel_1 = ((adr[15:3] == 0) && stb);

reg_8_pack #(
             8,               // Size of r0
             16,              // Size of r1
             16,              // Size of r2
             9,               // Size of r3
             8,               // Size of r4
             10,              // Size of r5
             10,              // Size of r6
             8,               // Size of r7
             3,               // Read only regs.
             16               // Size of the data bus.
             )
  reg_8_pack_1                // Instance name
  (
   .clk_i(clk_s),
   .rst_i(rst),
   .sel_i(sel_1),
   .we_i(we),
   .adr_i(adr[2:0]),
   .dat_io(dat),
   .r0({~ps2_clk,~ps2_data}),
   .r1(ram_s32_dat_o[31:16]),
   .r2(ram_s32_dat_o[15:0]),
   .r3(ram_s32_adr),
   .r4(led),
   .r5(crosshair_x_pos),
   .r6(crosshair_y_pos),
   .r7(lcd_control)
   );


assign ram_sel_1 = ((adr[15:10] == 4) && stb);  // 1000 - 13ff hex
assign dat = (ram_sel_1 && ~we)?ram_s16_dat_o:{16{1'bZ}};

ramb16_s16_s32 
  dp_ram_block_1 
  (
  .dat_o_s16(ram_s16_dat_o),
  .adr_i_s16(adr[9:0]),
  .dat_i_s16(dat),
  .rst_i_s16(rst),
  .we_i_s16(we && ram_sel_1),
  .clk_i_s16(clk_s),
//  .dat_o_s32(ram_s32_dat_o),
//  .adr_i_s32(ram_s32_adr),
  .dat_i_s32(32'b0),
  .rst_i_s32(1'b0),
  .we_i_s32(1'b0),
  .clk_i_s32(clk_s)
  );

//synopsys dc_script_begin
set_attribute dp_ram_block_1/ram0/ram0/ram_0 INIT_00
"0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF" -type string
set_attribute dp_ram_block_1/ram0/ram0/ram_0 INIT_01
"0000111122223333444455556666777788889999AAAABBBBCCCCDDDDEEEEFFFF" -type string
//synopsys dc_script_end

assign ram_sel_2 = ((adr[15:11] == 4) && stb);  // 2000 - 27ff hex
assign dat[7:0] = (ram_sel_2 && ~we)?ram_s8_dat_o:{8{1'bZ}};

ramb16_s8_s32 
  dp_ram_block_2 
  (
  .dat_o_s8(ram_s8_dat_o),
  .adr_i_s8(adr[10:0]),
  .dat_i_s8(dat[7:0]),
  .rst_i_s8(rst),
  .we_i_s8(we && ram_sel_2),
  .clk_i_s8(clk_s),
  .dat_o_s32(ram_s32_dat_o),
  .adr_i_s32(ram_s32_adr),
  .dat_i_s32(32'b0),
  .rst_i_s32(1'b0),
  .we_i_s32(1'b0),
  .clk_i_s32(clk_s)
  );



//--------------------------------------------------------------------------
// Module code
//--------------------------------------------------------------------------

assign debug_o[14] = switch[3];
assign debug_o[13] = ~switch[3];
assign debug_o[12:0] = switch[3];

endmodule

