//-----------------------------------------------------------------------------
// module: dac_sigma_delta.vhd
// This file contains a module for digital to analog conversion.
//
// OVERVIEW of contents:
//   The input to this DAC is considered an unsigned quantity.
//   This sigma-delta DAC is basically the same as the one shown in the
//   Xilinx App. note "XAPP154" of September 23, 1999.  It produces an output
//   which pulses at a duty cycle which is linearly proportional to the input
//   word dat_i.  Therefore, to use the analog output, a simple low-pass filter
//   filter must be used.  The component values for the low-pass filter are
//   determined by factors such as the clk_i frequency used, the number of
//   bits in the DAC, and the desired analog output signal bandwidth...
//   For further details consult Xilinx XAPP154, or for a simple set of
//   guidelines, see NOTES below.
//
//   This DAC produces 2^DAC_SIZE_PP different output voltages, from 0V to
//   Vdd*(1-1/2^DAC_SIZE_PP).  The Xilinx app. note describes how it can be
//   made to produce an additional output level of Vdd (true Vdd rail output).
//   Since I felt that Vdd*(1-1/2^DAC_SIZE_PP) was satisfactory, I did not
//   include a "delta adder."  I did this for simplicity, since the synthesizer
//   would have optimized out the delta adder in most cases anyway.  Once again
//   if you want more, refer to the Xilinx app. note.
//
//
// Author: John Clayton
// Date  : Aug.  7, 2001 Created this file.
// Update: Aug.  7, 2000 Added NOTES section, and code.  Removed delta adder.
//
//
//----------------------------------------
//
// NOTES
//
//
// This DAC generates a pulse-stream which must be low-pass filtered in order
// to arrive at the desired analog output level.  This filtering is simple
// enough, within certain limits...
//  Example schematic:
//
// dac_o ----/\/\/\----o--------O Analog output (to high-impedance load only)
//            R1       |
//                    ---
//                    --- C1
//                     |
//                     |
//                    ---
//                     -
//
// The filter cutoff frequency is 1/(2pi*R*C) Hertz.  For an audio output
// to headphones, or to the input of an audio amp., 25 kHz should be plenty.
// That makes R*C = 1/(2*pi*25000) = approx. 6.37E-6.
//
// Okay, the value of R1 determines the current draw from the pin.  Set the
// pin driver to its maximum.  Say 24mA.  The output pin is supposed to switch
// from "rail to rail" in order to maintain output linearity.  This means that
// R1 should be around 100x the output impedance of the pin, to avoid forming
// a voltage divider which would NOT produce rail-to-rail output.  The output
// pin has an impedance of around 25 ohms, so let R1 be at least 2.5 Kohms.
// For the sake of example, lets use 3.3k ohms for R1.
//
// So now C is approximately 6.37E-6/3.3E3 = 1.9 nF, about 0.002 microfarads.
//
// The output of this filter is a weak analog signal (meaning don't load it
// very much.)  Driving headphones directly will likely work, but driving a
// small speaker directly probably will not be good.  Use an amplifier in that
// case.
//
// The clock frequency of the DAC should ideally be high enough that in the
// worst case, each digital-to-analog sample has time to "resolve" or settle
// to its final value.  This means that clk_i >= 2^(DAC_WIDTH+1) * sample_rate.
// For a 12-bit audio DAC running at 44.1 kHz sample rate, the ideal clk_i
// works out to be:
//
//     2^(13) * 44100 = 8192 * 44100 = 361.2 MHz!!!  Not likely...
//
// But it turns out that the human ear is less sensitive to degradation in
// higher frequencies than lower ones -- and the higher frequencies are the
// ones which "suffer" when the clk_i is too slow...  So a much slower clk_i
// will work fine in this case (we can't always have the ideal, right?)
// (I am getting the above argument from the app-note, which also points out
//  that for 16-bit sigma-delta DACs used in CD players, the required ideal
//  clock frequency would be 2.9 GHz!  It states that "in practice a much lower
//  clock frequency is used.")
//
// Probably 40MHz or so would produce great output...  I will try it and see.
//
// Also, the worst pulse streams for the low-pass filter occur at the ends of
// the output range -- for example, dat_i = 0 or dat_i = all ones.  So, if you
// want to restrict your range of inputs, then the requirements on the low
// pass filter will be relaxed somewhat...
//-----------------------------------------


//`timescale 1ns/100ps

module dac_sigma_delta (
                  clk_i,
                  rst_i,
                  dat_i,
                  dac_o
                  );

parameter DAC_SIZE_PP = 12;    // Bit width of the DAC...

input clk_i;
input rst_i;
input[DAC_SIZE_PP-1:0] dat_i;
output dac_o;

// Local signals
reg [DAC_SIZE_PP+1:0] sigma_latch;  // the latched output of sigma adder

wire [DAC_SIZE_PP+1:0] new_term = {
                                   sigma_latch[DAC_SIZE_PP+1],
                                   sigma_latch[DAC_SIZE_PP+1],
                                   dat_i
                                   };
//-----------------------------------------------------------------------
// This is the "sigma adder."  It adds the previously latched sum with the
// new sign-adjusted-term.  The sign of the new term is always set so that
// the sequence of latched sums changes sign with a duty cycle equal to the
// desired output level.
always @(posedge clk_i)
begin
  if (rst_i) sigma_latch <= 0;
  else sigma_latch <= sigma_latch + new_term;
end

// If you want to make this DAC slightly better, try using a separate output
// flip-flop located right at the I/O pin of the programmable device.
// That way, less noise is introduced on dac_o.
// Be sure to maximize the current capability of the output driver!

assign dac_o = sigma_latch[DAC_SIZE_PP+1];

endmodule

