//-----------------------------------------------------------------------------
//
// Author: John Clayton
// Date  : April 30, 2001
// Update: April 30, 2001 Copied this file from "lcd_3.v" for use as top level
// Update: May    3, 2001 Included instantiations of VHDL code, to test mixed
//                        language support of the XILINX foundation 3.1 tools.
// Update: June   1, 2001 Brigham Young's 200th birthday!
//                        Started project "build_4" using ps2 interface and
//                        the rs232 interface code.
//
// Description
//-----------------------------------------------------------------------------
// This targets an XC2S200 board which was created for educational purposes.
//
// There are:
//    8  LEDs (led[7:0])
//    4  switches (switch[3:0])
//    1  clock of 32.000 MHz clock, present on GCLK1
//    1  clock of 49.152 MHz clock, present on GCLK0
//    4  lines of ps2 clock input (port A in documentation notes)
//    4  lines of ps2 data input (port A in documentation notes)
//    16 lines of LCD panel control (port B in documentation notes)
//    2  lines of rs232 serial connection (port C in documentation notes)
//-----------------------------------------------------------------------------


`resetall
`timescale 1ns/100ps

module top (
  sys_clk_0,
  sys_clk_1,
  switch,
  led,
  ps2_clk,
  ps2_data,
  lcd_drive,
  rs232_rxd,
  rs232_txd
  );
  
// I/O declarations
input sys_clk_0;      // 49.152 MHz
input sys_clk_1;      // 32.000 MHz
input [3:0] switch;
input rs232_rxd;

inout [3:0] ps2_clk;
inout [3:0] ps2_data;

output [7:0] led;

output [15:0] lcd_drive;
output rs232_txd;

// Internal signal declarations

wire [7:0] ps2_scan_code;
wire [7:0] ps2_ascii;
wire [7:0] ps2_status;
wire [7:0] rs232_rx_character;

wire [7:0] temp1;
wire [7:0] temp2;

wire reset = switch[0];
wire [2:0] rs232_rx_error;
wire ps2_key_data_ready;
wire ps2_key_released;
wire ps2_key_pressed = ~ps2_key_released;
wire rx232_tx_load_request;
wire rs232_rx_data_ready;

//--------------------------------------------------------------------------
// Instantiations
//--------------------------------------------------------------------------

lcd_test_pongball lcd_block (
  .sys_clk(sys_clk_1),
  .lcd_clk(sys_clk_0),
  .sys_reset(reset),
  .lcd_reset(1'b0),
  .ball_x_enable(switch[1]),
  .ball_y_enable(switch[1]),
  .lcd_drive(lcd_drive)
  );

ps2_keyboard_interface #(2950, // number of clks for 60usec.
                         12,   // number of bits needed for 60usec. timer
                         63,   // number of clks for debounce
                         6,    // number of bits needed for debounce timer
                         1     // Trap the shift keys, no event generated
                         )                       
  ps2_block (                  // Instance name
  .clk(sys_clk_0),
  .reset(reset),
  .ps2_clk(ps2_clk[0]),
  .ps2_data(ps2_data[0]),
  .rx_extended(ps2_status[5]),
  .rx_released(ps2_key_released),
  .rx_shift_key_on(ps2_status[7]),
  .rx_scan_code(ps2_scan_code),
  .rx_ascii(ps2_ascii),
  .rx_data_ready(ps2_key_data_ready),
  .rx_read(ps2_key_data_ready),
  .tx_data({rs232_rx_character[5],        // Make upper nibble E, F or 0
            rs232_rx_character[5],
            rs232_rx_character[5],
            rs232_rx_character[4],
            rs232_rx_character[3:0]}),
  .tx_write(rs232_rx_data_ready),
  .tx_write_ack_o(ps2_tx_write_ack_o),
  .tx_error_no_keyboard_ack(ps2_status[4])
  );
assign ps2_status[6] = ps2_key_released;
assign ps2_status[3] = rs232_rx_data_ready;


// These defines are for the rs232 interface
`define START_BITS 1
`define DATA_BITS 8
`define STOP_BITS 1
`define CLOCK_FACTOR 16

// This unit generates the correct 16x transmit clock (enable) frequency
// which is used for the serial transmit operation.
clock_gen_select clock_unit
  (
   .clk(sys_clk_0),
   .reset(reset),
   .rate_select(3'b100),         // 115,200 baud
   .clk_out(serial_clk_16x)
  );

// A transmitter, which asserts load_request at the end of the currently
// transmitted word.  The tx_clk must be a "clock enable" (narrow positive
// pulse) which occurs at 16x the desired transmit rate.  If load_request
// is connected directly to load, the unit will transmit continuously.
rs232tx #(
          `START_BITS,   // start_bits
          `DATA_BITS,    // data_bits
          `STOP_BITS,    // stop_bits (add intercharacter delay...)
          `CLOCK_FACTOR  // clock_factor
         )
         rs232_tx_block // instance name
         ( 
          .clk(sys_clk_0),
          .tx_clk(serial_clk_16x),
          .reset(reset),
          .load(   ps2_key_data_ready
                && ps2_key_pressed
                && rs232_tx_load_request),
          .data(ps2_ascii),                 // Connected from keyboard
          .load_request(rs232_tx_load_request),
          .txd(rs232_txd)
         );

// A receiver, which asserts "word_ready" to indicate a received word.
// Asserting "read_word" will cause "word_ready" to go low again if it was high.
// The character is held in the output register, during the time the next
//   character is coming in.
rs232rx #(
          `START_BITS,  // start_bits
          `DATA_BITS,   // data_bits
          `STOP_BITS,   // stop_bits
          `CLOCK_FACTOR // clock_factor
         )
         rs232_rx_block // instance name
         ( 
          .clk(sys_clk_0),
          .rx_clk(serial_clk_16x),
          .reset(reset || (| rs232_rx_error) ),
          .rxd(rs232_rxd),
          .read(ps2_tx_write_ack_o),
          .data(rs232_rx_character),
          .data_ready(rs232_rx_data_ready),
          .error_over_run(rs232_rx_error[0]),
          .error_under_run(rs232_rx_error[1]),
          .error_all_low(rs232_rx_error[2])
         );

//`undef START_BITS 
//`undef DATA_BITS 
//`undef STOP_BITS 
//`undef CLOCK_FACTOR


//--------------------------------------------------------------------------
// Module code
//--------------------------------------------------------------------------

assign ps2_status[2:0] = 0;

assign temp1 = switch[2]?rs232_rx_character:ps2_scan_code;
assign temp2 = switch[2]?{~ps2_clk,~ps2_data}:ps2_status;
assign led = switch[3]?temp2:temp1;



endmodule

