/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.verilogp;

import java.io.*;

/** Implements Verilog suitable tokenizer.
  * @see StreamTokenizer */
class VTokenizer extends StreamTokenizer {

	/** Number of bits last number read hold. */
	public int nbits = 0;
	
	/** constructs new Verilog tokenizer
	  * @param reader file reader to read from */
	VTokenizer(Reader reader) {
		super(reader);
		//resetSyntax();
		slashSlashComments(true); // enable C/C++ type comments
		slashStarComments(true);		
		setTable();
		wordChars(128, 255);
	}
	
	/** defines character with the same name */
	static final int EOF = TT_EOF;
	/** defines character with the same name */
	static final int NUMBER = TT_NUMBER;
	/** defines character with the same name */
	static final int LPAREN = '(';
	/** defines character with the same name */
	static final int RPAREN = ')';
	/** defines character with the same name */
	static final int COMMA  = ',';
	/** defines character with the same name */
	static final int COLON  = ':';
	/** defines character with the same name */
	static final int DOT    = '.';
	/** defines character with the same name */
	static final int SEMI   = ';';
	/** defines character with the same name */
	static final int LBRACKET='[';
	/** defines character with the same name */
	static final int RBRACKET=']';
	/** defines character with the same name */
	static final int NOT    = '~';
	/** defines character with the same name */
	static final int AND    = '&';
	/** defines character with the same name */
	static final int OR	    = '|';
	/** defines character with the same name */
	static final int XOR    = '^';
	/** defines character with the same name */
	static final int EQUAL  = '=';
	/** defines character with the same name */
	static final int BSLASH = '\\';
	/** defines character with the same name */
	static final int LCBRACKET = '{';
	/** defines character with the same name */
	static final int RCBRACKET = '}';
	
	/** Gets next token extended.
	  * <em>WARNING: <code>'-'</code> shouldn't be pushed back.</em> */
	public int nextToken() throws IOException {		
		int token = super.nextToken();
		if(token == BSLASH) {
			wordChars(32+1, 127);	// expect everything
			token = super.nextToken();			
			setTable();						// restore
		} else if((token == TT_WORD) // do we have number here?
		    &&((Character.isDigit(sval.charAt(0)))
			     ||(sval.charAt(0) == '\''))) {				
			
			nval = parseNumber(sval);
			return NUMBER;
		} else if(token == '-') {
			int t = nextToken();
			if(t == NUMBER) {nval = -nval; return NUMBER;}
			pushBack();						
		}
		return token;
	}
	
	/** setup lexer table */
	private void setTable() {
		whitespaceChars(0, 32);
		ordinaryChars(32+1,'A'-1);		
		wordChars('A', 'Z');
		ordinaryChars('Z'+1,'a');
		wordChars('a', 'z');
		ordinaryChars('z'+1,127);		
		wordChars('_','_');
		wordChars('0', '9');
		wordChars('\'', '\'');
		ordinaryChar('.');	// no floating point
		quoteChar('"');
	}	

  /** Parses <code>String</code> for binary, octal, decimal or
    * hex number. Number of bits required to store number is
    * placed into <code>nbits</code>
    * @param s String to parse
    * @return number */
	private int parseNumber(String s) {
		int ndigs = 0;
		int ls = s.length();
		while((ndigs < ls)&&(Character.isDigit(s.charAt(ndigs)))) ndigs++;
		// get first number		
		int first = Integer.parseInt(s.substring(0,ndigs));
		if(s.length() == ndigs) {
			int temp = first;
			nbits = 0;
			while(temp != 0) { temp >>= 1;	nbits++; }
			nbits = Math.max(nbits, 1);
			return first; // nothing else
		}
		if(s.charAt(ndigs) != '\'') throw new Error("\"'\" expected");		
		char type = s.charAt(ndigs+1);
		s = s.substring(ndigs+2).toUpperCase();
		ls = s.length();
		int val = 0;
		if(ls>first) throw new Error("Number too large.");
		nbits = first;
		switch(type) {
			case 'b':
				for(int i = 0; i < ls; i++) {
					switch(s.charAt(i)) {
						case '0':	val *= 2; break;
						case '1': val = val*2+1; break;
						default:throw new Error("Invalid binary number");
					}
				}
				break;
			case 'd':
				for(int i = 0; i < ls; i++) {
					if(Character.isDigit(s.charAt(i)))
						val = val * 10 + s.charAt(i) - '0';
					else throw new Error("Invalid decimal number");					
				}
				break;
			case 'o':
				for(int i = 0; i < ls; i++) {
					if((s.charAt(i) >= '0')&&(s.charAt(i) <= '9'))
						val = val * 8 + s.charAt(i) - '0';
					else throw new Error("Invalid octal number");					
				}
				break;
			case 'h':
				for(int i = 0; i < ls; i++) {
					if(Character.isDigit(s.charAt(i)))
						val = val * 16 + s.charAt(i) - '0';
					else if((s.charAt(i) >= 'A')&&(s.charAt(i) <= 'F'))
						val = val * 16 + s.charAt(i) - 'A'+10;
					else throw new Error("Invalid hex number");					
				}
				break;
			default: throw new Error("Invalid number format.");
		}		
		return val;
	}		
}
