/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.verilogp;

import java.io.*;
import java.util.Stack;
import java.util.Vector;
import org.opencores.structure.*;
import org.opencores.Conf;

/** Handwriten optimized LL(1) parser that reads data from stream
  * and builds graph <code>g</code>.<p>
  * Parser does not use standard parser or lexer builder, since
  * the number of keyword is relatively small, and there is no
  * need to parse complex objects. Only expression part is bit
  * uglier. Java.io.StreamTokenizer is used instead.<p>
  *
  * <em>NOTE: This parser accept only limited Verilog grammar, which
  * should come out of design synthesis.
  * - assign support only maxtern representation <code>X->(X)|~X|X&X</code>
  * - only one module<p>
  * - no tristate wires, etc<p>
  * - no memory (e.g. <code>reg [31:0] mem [16]</code>) </em>*/
public class Parser {
  /** currently built graph */
	private Graph g = new Graph();
	/** Java's stream tokenizer.
	  * @see java.io.StreamTokenizer */
	private VTokenizer st;
	/** <code>Dictionary</code> of all names. */
	private Dictionary d = new Dictionary();
	/** File name we are processing */
	private String filename;
		
	/** parses file and construct graph
		* @param filename file to read from */	
	public Parser(String filename) throws IOException {
	  /* setup tokenizer */
	  this.filename = filename;
		st = new VTokenizer(new BufferedReader(new FileReader(Conf.sdir+filename)));
		parse();
	}
	
	/** returns newly built graph
	  * @return graph */
	public Graph getGraph() {
		return g;
	}
	
	/** parses whole file */
	private void parse() throws IOException {
		module();		
	}
	
	/** parses one module */
	private void module() throws IOException {
		readKeyword(d.MODULE);
		String s = readNewWord(); // set graph name
		g.name = s;
		if(st.nextToken() == st.LPAREN) {
			int token;
			do {
				String port = readNewWord(); // add ports to graph				
				Range r = new Range();
				r.name = port;
				d.add(port, r);		// add entry, just to mark existence
				token = st.nextToken();
			} while (token == st.COMMA);
			if(token != st.RPAREN) throw new Error(errorPrefix()+"Expected )");
		} else {
			st.pushBack();
		}
		readSym(st.SEMI);
		int kw = d.MODULE;
		int t;
		while(kw != d.ENDMODULE) {
			kw = readKeyword();
			switch(kw) {
				case d.INPUT:	readIN_OUT(Node.INPUT); break;
				case d.OUTPUT:readIN_OUT(Node.OUTPUT); break;
				case d.WIRE:  readWIRE(); break;
				case d.ASSIGN:readASSIGN(); break;
				case d.ENDMODULE:break;				
				case -1: // non keyword => insert new module file
					readModule(st.sval);
					break;
				default: 
					if((kw >= d.PRIMITIVES)&&(kw < d.PRIMITIVES_END)) { // known module primitive
						Node n = null;
						switch(kw) { // build primitive
							case GTech.AND: n = new NodePrim(NodePrim.AND2); break;
							case GTech.OR:	n = new NodePrim(NodePrim.OR2); break;
							case GTech.XOR: n = new NodePrim(NodePrim.XOR2); break;
							case GTech.NOT: n = new NodePrim(NodePrim.INV); break;
							case GTech.AND_NOT: n = new NodePrim(NodePrim.NAND2); break;
							case GTech.ONE: n = new NodePrim(NodePrim.VCC); break;
							case GTech.ZERO:n = new NodePrim(NodePrim.GND); break;
							case GTech.BUF: n = new NodePrim(NodePrim.BUF); break;
							case GTech.FFGEN:
							case GTech.FD1: n = new NodeFF(); break;
							//case GTech.ADD_ABC: n = new NodeAdd(); break;
							default: throw new Error(errorPrefix()+"Unsupported primitive '"+st.sval+"'.");
						}
						kw -= Dictionary.PRIMITIVES;
						n.name = readNewWord();
						g.nodes.addElement(n);
						readSym(st.LPAREN);
						int k = st.COMMA;
						do {
							if(k != st.COMMA) throw new Error(errorPrefix() + "Syntax.");							
							readSym(st.DOT); // parse parameters
							k = st.nextToken();
							String name = st.sval;
							int found = -1;
							for(int i = 1; i < GTech.prim[kw].length; i++)
								if(name.equals(GTech.prim[kw][i])) found = i-1;
							if(found < 0) throw new Error(errorPrefix() + "Invalid parameter.");
							readSym(st.LPAREN);
							Object o = readParam(); // should not be array, since primitives are simple
							if(found < n.width) {
								n.ports[found] = (Net) o;
								if(n.dir[found] == n.INPUT)	((Net)o).inputs.addElement(n);
								else ((Net)o).output = n;
							} else
								if(Conf.verbose)
									Conf.log.println(errorPrefix() +	"Ignoring parameter "+GTech.prim[kw][found+1]);
									
							readSym(st.RPAREN);
							k = st.nextToken();
						} while (k != st.RPAREN );
					}	else throw new Error(errorPrefix() + "Keyword expected.");
					readSym(st.SEMI);
					break;
			}
		}
		if(st.nextToken() != st.EOF) throw new Error(errorPrefix()+"Only one module per file allowed.");
	}
	
	/** reads one module parameter inside () (no arrays {}) */
	private Object readParamSimple() throws IOException {
		Net n;		
		Object o = readWord();
		if(o instanceof Range) { // is [] array?
			Range r = (Range)o;
			int k = st.nextToken();
			if(k == st.LBRACKET) {
				int idx = readNumber();				
				k = st.nextToken();
				if(k == st.RBRACKET) {
					idx = (idx - r.msb) * r.inc;									
					if(r.array[idx] instanceof Net) {						
						return r.array[idx];
					}	else throw new Error(errorPrefix() + "Wire expected.");					
				} else if(k == st.COLON) {
					int idx2 = readNumber();					
					Range nr = new Range();
					nr.msb = idx; nr.lsb = idx2; nr.inc = (idx > idx2)?(-1):(1);
					nr.array = new Object[(nr.lsb-nr.msb)*nr.inc+1];
					for(int i = 0; i < nr.array.length; i++) { // copy selection
						nr.array[i] = r.array[(i-r.msb)*r.inc];
					}					
					return nr;
				} else throw new Error(errorPrefix() + "Syntax.");				
			} else {
				st.pushBack();				
				return r;
			}
		} else {			
			if(o instanceof Net) return o;
			else throw new Error(errorPrefix() + "Wire expected.");
		}		
	}
	
	/** reads one module parameter */
	private Object readParam() throws IOException {
		int k = st.nextToken();
		Stack s = new Stack();
		if(k == st.LCBRACKET) {				
			// assimilate port
			k = st.COMMA;
			int count = 0;
			do {
				if(k != st.COMMA) throw new Error(errorPrefix()+"Syntax.");
				Object o;
				s.push(o = readParamSimple());
				if(o instanceof Net) count++;
				else count+=((Range)o).array.length;
				k = st.nextToken();
			} while (k != st.RCBRACKET);
			if(count == 1) return s.elementAt(0);
			// now that we have all parameters listed, we will put them in
			// one range
			Range r = new Range();
			r.msb = 0; r.lsb = count-1;	r.inc = 1; r.array = new Object[count];
			count = 0;
			for(int i = 0; i < s.size(); i++) {
				if(s.elementAt(i) instanceof Net) {
					r.array[count++] = s.elementAt(i);
				} else {
					Range sr = (Range) s.elementAt(i);
					for(int j = 0; j < sr.array.length; j++)
						r.array[i] = sr.array[j];
				}
			}
			return r;
			
		} else {
			st.pushBack();			
			return readParamSimple();						
		}		
	}

  /** processes INPUT / OUTPUT statement */
	private void readIN_OUT(int dir) throws IOException {
		readRange();
		int t = st.COMMA;
		do {
			if(t != st.COMMA) throw new Error(errorPrefix()+"Syntax");
			Object id = readWord();
			if(id == null) {
				id = new Range();	// not defined yet
				((Range)id).name = st.toString();
			} else {	// defined, must be declared as Range
				if(!(id instanceof Range)) throw new Error(errorPrefix()+"Nondefined, but declared port expected.");
			}
			Range r = (Range)id;			
			if(msb == lsb) {	// no range
				NodePort np = new NodePort(dir);
				g.nodes.addElement(np);
				np.name = r.name;
				Net nt = new Net();
				if(np.dir[0] == np.INPUT)	nt.inputs.addElement(np);					
				else nt.output = np;
				np.ports[0] = nt;
				nt.name = np.name;
				g.nets.addElement(nt);
				d.add(nt.name, nt);	// replace range with port's wire
			} else {
				int inc = 1;
				if(msb > lsb) inc = -1; // keep range with same name
				r.msb = msb; r.lsb = lsb; r.inc = inc; // add
				r.array = new Object[(lsb-msb)*inc+1]; // allocate new array
				for(int i = msb; i != lsb+inc; i+=inc ) {
					NodePort np = new NodePort(dir);
					g.nodes.addElement(np);					
					np.name = r.name+" "+i; // add range instances with similar names					
					Net nt = new Net();
					if(np.dir[0] == np.INPUT)	nt.inputs.addElement(np);					
					else nt.output = np;
					np.ports[0] = nt;
					nt.name = np.name;
					g.nets.addElement(nt);
					r.array[(i-msb)*inc] = nt;
				}
			}
			t = st.nextToken();			
		} while (t != st.SEMI);
	}

	/** processes WIRE statement */
	private void readWIRE() throws IOException {
		readRange();
		int t = st.COMMA;
		do {
			if(t != st.COMMA) throw new Error(errorPrefix()+"Syntax");
			String name = readNewWord();
			Range r = new Range();	// not defined yet			
						
			int inc = 1;
			if(msb == lsb) {	// no range
				Net n = new Net();
				g.nets.addElement(n);
				n.name = name;	
				t = st.nextToken();
				//d.remove(name)				
				d.add(name, n);
			} else {
				r.name = name;
				if(msb > lsb) inc = -1; // keep range with same name
				r.msb = msb; r.lsb = lsb; r.inc = inc; // add
				r.array = new Object[(msb-lsb)*inc]; // allocate new array
				for(int i = msb; i != lsb+inc; i+=inc ) {
					Net n = new Net();
					g.nets.addElement(n);
					n.name = r.name+" "+i; // add range instances with similar names					
					r.array[(i-msb)*inc] = n;
				}
				t = st.nextToken();				
			}
		} while (t != st.SEMI);
	}

	/** count for marking added nodes and nets */
  private int expr_mark = 0;
	/** processes ASSIGN statement */
	private void readASSIGN() throws IOException {		
		readRange();
		int t = st.COMMA;
		do {
			if(t != st.COMMA) throw new Error(errorPrefix()+"Syntax");
			Object o = readParamSimple();	// must be defined prior use
			if(!(o instanceof Net)) throw new Error(errorPrefix()+"Net expected.");
			//Range r = new Range();	// not defined yet	
			
			readSym(st.EQUAL);				// = must be present
					
			//int inc = 1;
			if(msb == lsb) {					// no range
				Net nt = (Net)o;
				Net expr = readExpr();
				nt.output = expr.output;
				Node n = nt.output;			// relink to nt
				for(int i = 0; i < n.width; i++)
					if(n.ports[i] == expr) n.ports[i] = nt;									
				g.nets.remove(expr);
			} else
				throw new Error(errorPrefix()+"Array assignment not supported");			
			t = st.nextToken();
		} while (t != st.SEMI);
	}
  
	/** Reads expression from the stream
	  * @return net to assignment */
	private Net readExpr() throws IOException {
		Net a = readTerminal();
		int t = st.nextToken();		
		if(a == null) throw new Error(errorPrefix()+"Syntax");		
		
		switch(t) {
			//case st.OR:		  throw new Error(errorPrefix()+"OR not supported.");
			default:				st.pushBack(); return a;
		}		
	}
	
	/** Reads terminal from the stream
	  * @return net to assignment */
	private Net readTerminal() throws IOException {
		Net a = readFinal();
		
		while(true) {
			int t = st.nextToken();
			switch(t) {
				case st.AND:  NodePrim n = new NodePrim(NodePrim.AND2);
											Net nt = new Net();
											Net b = readFinal();	// read second parameter
											n.name = "a"+st.lineno()+"_"+expr_mark;
											nt.name = "na"+st.lineno()+"_"+expr_mark++;
											g.nodes.add(n);		  // add to graph
											g.nets.add(nt);
											n.ports[0] = a; a.inputs.add(n);// link AND2
											n.ports[1] = b; b.inputs.add(n);
											n.ports[2] = nt; nt.output = n;
											a = nt;
											break;
				default:			st.pushBack(); return a;
			}
		}
	}
	
	/** Reads final from the stream
	  * @return net to assignment */
	private Net readFinal() throws IOException {		
		int t = st.nextToken();		
		switch(t) {
			case st.NOT:		NodePrim n = new NodePrim(NodePrim.INV);
											Net nt = new Net();
											Net a = readFinal();	// read parameter											
											n.name = "a"+st.lineno()+"_"+expr_mark;
											nt.name = "na"+st.lineno()+"_"+expr_mark++;
											g.nodes.add(n);		  // add to graph
											g.nets.add(nt);
											n.ports[0] = a; a.inputs.add(n);// link INV
											n.ports[1] = nt; nt.output = n;
											return nt;
			case st.TT_WORD:st.pushBack();
											Object o = readParamSimple();
											if(o instanceof Range) throw new Error(errorPrefix()+"Arrays not supported in assign statement.");
											if(!(o instanceof Net)) throw new Error(errorPrefix()+"Wire expected.");
											return (Net)o;
			case st.LPAREN: nt = readExpr(); readSym(st.RPAREN); return nt;
			default:				throw new Error(errorPrefix()+"Syntax.");
		}
	}

	/** [msb:lsb] from last readRange */
	private int msb, lsb;
	/** reads [msb:lsb] */
	private void readRange() throws IOException {
		int token = st.nextToken();
		if(token == st.LBRACKET) {
			msb = readNumber();
			readSym(st.COLON);
			lsb = readNumber();
			readSym(st.RBRACKET);
		} else {
			msb = lsb = 0;
			st.pushBack();
		}
	}
	
	/** expects sym on input */
	private void readSym(int sym) throws IOException {
		int token = st.nextToken();
		if(token != sym) throw new Error(errorPrefix()+"Symbol expected.");		
	}
	
	/** expects non-keyword on input */
	private Object readWord() throws IOException {
		int token = st.nextToken();
		if(token != st.TT_WORD) {
			if(token == st.NUMBER) {	// assign NodeGlobal, since constant is specified
				if(st.nbits == 1) return g.global[((int)st.nval)&1];
				else {
					Range r = new Range();
					r.array = new Object[st.nbits];
					r.lsb = 0; r.msb = st.nbits; r.inc = 1;
					for(int i = st.nbits-1; i >= 0; i--) {
						r.array[i] = g.global[((int)st.nval)&1];
						st.nval = ((int)st.nval)>>1;
					}
				}
			} else throw new Error(errorPrefix()+"Word expected, found token#"+token+".");		
		}
		Object o = d.get(st.sval);
		if(o instanceof Integer) throw new Error(errorPrefix()+"ID expected, keyword found.");
		return o;
	}
	
	/** expects word on input */
	private String readNewWord() throws IOException {
		int token = st.nextToken();
		if(token != st.TT_WORD) throw new Error(errorPrefix()+"Word expected, found token#"+token+".");
		if(d.get(st.sval) != null) throw new Error(errorPrefix()+"Invalid identifier");
		return st.sval;
	}
	
	/** expects keyword k on input */
	private void readKeyword(int k) throws IOException {
		int token = st.nextToken();
		if((token != st.TT_WORD)		
		||(((Integer)d.get(st.sval)).intValue() != k))
			throw new Error(errorPrefix()+"keyword '"+d.keywords[k]+"' expected.");
	}
	
	/** read number */
	private int readNumber() throws IOException {
		readSym(st.TT_NUMBER);
		Double d = new Double(st.nval);
		return d.intValue();
	}
	
	/** expects keyword on input */
	private int readKeyword() throws IOException {
		int token = st.nextToken();
		boolean bad = false;
		if(token != st.TT_WORD) return -1;
		Object o = d.get(st.sval);
		if(!(o instanceof Integer)) return -1;	  
		return ((Integer)o).intValue();
	}	
	
	/** returns error prefix displayed */
	private String errorPrefix() {
		return filename+"("+st.lineno()+"): ";
	}
	
	/** reads module from file with same name */
	private void readModule(String name) throws IOException {
		String s = readNewWord();			// instance name
		System.out.println("Including '"+s+"'.");
		if(Conf.debug) Conf.log.println("Including '"+s+"'.");
		
		Parser p = new Parser(name+".v");		
		/* we now have module loaded, now lets look what parameters
		 * we want to pass. */					
		readSym(st.LPAREN);
		int k = st.COMMA;
		while( k == st.COMMA) {
			Range r = null;
			Net np = null;			
			readSym(st.DOT);
			k = st.nextToken();
			if(k != st.TT_WORD) throw new Error(errorPrefix()+"Syntax.");
			Object o = p.d.get(st.sval); // search for port in new graph
			if(o == null) throw new Error(errorPrefix()+"Parameter name not found.");
			if(o instanceof Range) {
				r = (Range) o;
			} else if(o instanceof Net) {
				np = (Net) o;
			} else throw new Error(errorPrefix()+"Parameter name not found.");

			readSym(st.LPAREN);

			o = readParam();	// read parameters				
			if(o instanceof Net) {
				if(np == null) throw new Error(errorPrefix()+"Single wire expected.");
				assimilate(np, (Net)o);				
			} else {
				Range nr = (Range)o; // old module array, r = new module array of nets
				if((r == null)||(r.array.length!=nr.array.length)) throw new Error(errorPrefix()+"Array of same size expected.");				
				for(int i = 0; i < r.array.length; i++) {
					assimilate((Net)r.array[i], (Net) nr.array[i]);
				}
			}			
			readSym(st.RPAREN);
			k = st.nextToken();
		}
		
		/* copy nodes and their nets, and repair structure if not
		 * all ports are used */		
		//for(int i = 0; i < p.g.nets.size(); i++) nt.link = null; //already null by parser
		for(int i = 0; i < p.g.nodes.size(); i++) {
			Node n = (Node) p.g.nodes.elementAt(i);		
			if(n instanceof NodePort) { // do not add port's net				
				Net nt = n.ports[0];
				nt.link = nt; // mark to delete
			}
		}		
		for(int i = 0; i < p.g.nodes.size(); i++) {
			Node n = (Node) p.g.nodes.elementAt(i);		
			if(!(n instanceof NodePort)) {// do not add ports
				g.nodes.addElement(n);
				for(int j = 0; j < n.width; j++) {
					if(n.ports[j].link != null) {
						if(n.dir[j] == n.OUTPUT) {
							// default output is null - not used							
							n.ports[j] = null;							
						}	else { // default input is GND, add new node
							Net nt = new Net();
							NodePrim np = new NodePrim(NodePrim.GND);
							nt.name = np.name = n.ports[j].name;
							n.ports[j] = np.ports[0] = nt;
							nt.inputs.addElement(n);							
							g.nodes.addElement(nt.output = np);
							g.nets.addElement(nt);
						}
					}
				}
			}
		}
		for(int i = 0; i < p.g.nets.size(); i++) {
			Net nt = (Net)p.g.nets.elementAt(i);
			if(nt.link == null) g.nets.addElement(nt);
		}
				
		if(k != st.RPAREN) throw new Error(errorPrefix()+"Syntax.");
		readSym(st.SEMI);
	}
	
	/** connects port's net np to net nt */
	private void assimilate(Net np, Net nt) {		
	/* if we have input port, net output should be NodePort, port.dir[0] = input */
		if((np.output instanceof NodePort)&&(np.output != null))  { // input port, copy port's inputs
			for(int i = 0; i < np.inputs.size(); i++) { // to output net
				Node n = (Node)np.inputs.elementAt(i); // port drives n
				for(int j = 0; j < n.width; j++)
					if(n.ports[j] == np) n.ports[j] = nt;
				nt.inputs.addElement(n);				// add all nodes to old net	
			}
			// port update not needed, since it will be deleted
		} else {	// output port
			Node n = np.output;
			if(n == null)
				throw new Error(errorPrefix()+"Net "+np.name+" should have defined output.");
				
			for(int j = 0; j < n.width; j++)
				if(n.ports[j] == np) n.ports[j] = nt;			
			nt.output = n; // update net's port

			for(int i = 0; i < np.inputs.size(); i++) { // to output net
				Node nn = (Node)np.inputs.elementAt(i); // port drives n
				if(!(nn instanceof NodePort)&&(nn != null)) { // only one port connected to this one
					for(int j = 0; j < nn.width; j++)
						if(nn.ports[j] == np) nn.ports[j] = nt;
					nt.inputs.addElement(nn);				// add all nodes to old net	
				}
			}
			// port update not needed, since it will be deleted			
		}	
	}	
}
