/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.util;

import java.util.Vector;
import java.util.NoSuchElementException;

/** Priority queue using heap.
  * <code>add</code> and <code>remove</code> require maximum O(log(N)) time. */ 
public class Heap extends Vector {

  /** Constructs the heap in O(N) time, using a technique similar to
    * bottom-up construction.
    * @param array constructs heap based on this array    */
  public Heap(Comparable array[]) {
		super(array.length);
		setSize(array.length);
		for (int i = 0; i < array.length; i++) setElementAt(array[i], i);		
		repair();
  }
	/** Reconstruct the heap in O(N) time. */  
  public void repair() {
		for (int i = (int)Math.floor(size() / 2) - 1; i >= 0; i--) sift(i);
  }

  /** Constructs a heap with no elements. */
  public Heap() {
		super();
  }
    
  /** Returns the Vector index of the left child.
    * @param i index */
  private static final int left(int i) {
		return ((i + 1) << 1) - 1;
  }

  /** Returns the Vector index of the right child.
    * @param i index */
  private static final int right(int i) {
		return (i + 1) << 1;
  }

  /** Returns the Vector index of the parent */
  private static final int parent(int i) {
		return ((i + 1) >> 1) - 1;
  }

  /** Exchanges the elements stored at the two locations
    * @param i index of first element
    * @param j index of second element */
  private final synchronized void exchange(int i, int j) {
		Object temp = elementAt(j);
		setElementAt(elementAt(i), j);
		setElementAt(temp, i);
  }

  /** Also known as downheap, restores the heap condition
    * starting at node <code>i</code> and working its way down.
    * @param i index of node to sift */
  private final synchronized void sift(int i) {
		int l = left(i);
		int r = right(i);
		int smallest;

		if(l < size()&&((Comparable)elementAt(l)).compareTo(elementAt(i)) < 0)
			smallest = l;
		else
			smallest = i;

		if(r < size()&&((Comparable)elementAt(r)).compareTo(elementAt(smallest)) < 0)
			smallest = r;

		if (smallest != i) {
			exchange(i, smallest);
			sift(smallest);
		}
  }

  /** Removes the minimum (top) element from the Heap, decreases the
    * size of the heap by one, and returns the minimum element.
    * @return minimum object */
  public synchronized Comparable remove() throws NoSuchElementException {
		if (size() == 0) throw new NoSuchElementException();
		Object min = elementAt(0);

		// move the last key to the top, decrease size, and downheap
		setElementAt(lastElement(), 0);
		removeElementAt(size() - 1);
		sift(0);

		return (Comparable)min;
  }

  /** Inserts key into the heap, and then upheaps that key to a
    * position where the heap property is satisfied.
    * @param elt object to insert */
  public synchronized void add(Comparable element) {
		int i = size();
		setSize(size() + 1);

		// upheap if necessary
		while (i > 0 && ((Comparable)elementAt(parent(i))).compareTo(element) > 0) {
			setElementAt(elementAt(parent(i)), i);
		  i = parent(i);
		}	
		setElementAt(element, i);
  }
  
  /** Removes specified element from the Heap, decreases the
    * size of the heap by one, and returns <code><b>true</b><code> if successful.
    * @param element element to delete
    * @return <b>true</b> if object found <b>false</b> otherwise*/
  public synchronized boolean remove(Comparable element) throws NoSuchElementException {
 		if (size() == 0) throw new NoSuchElementException();
 		int idx = indexOf(element);		// search for element
 		if(idx < 0) return false;			// invalid index => not found
				
		// move the last key to the top, decrease size, and downheap
		setElementAt(lastElement(), idx);
		removeElementAt(size() - 1);
		sift(idx);

		return true;
  }
}
