package org.opencores.util;

import java.io.OutputStream;
import java.io.IOException;

/** Class that allows writing bit by bit to file.
  * Should be constructed using <code>OutputStream</code> or
  * better <code>BufferedOutputStream</code>. */  
public class BitStreamWriter {
	/** (byte) buf which holds current bits */
	private int buf;
	
	/** Number of bits in <code>buf</code> */
	private int nbits = 0;
	
	/** Output stream */
	private OutputStream stream;
	
	/** Constructs new bitstream writer.
	  * <em>Example:</em><code>
	  * BitStreamWriter stream = new BitStreamWriter(new BufferedOutputStream(new FileOutputStream("abc.bin")));
	  * </code>
	  * @param stream output stream */
	public BitStreamWriter(OutputStream stream) {
		this.stream =  stream;		
	}
	
	/** Writes <code>nb</code> bits to the stream.
	  * <code>nb</code> &lt= 24.
	  * @param data bit sequence to write
	  * @param nbits number of bits to write */
	public void write(int data, int nb) throws IOException {
		/* should be: this.nbits < 8 */
		buf <<= nb;
		buf |= data;
		nbits += nb;		
		while(nbits >= 8) {
			nbits -= 8;
			stream.write((buf >> nbits)&0xFF);
		}		
	}

	/** Writes one bit to the stream.
	  * @param bit bit to write */
	public void write(int bit) throws IOException {
		write(bit, 1);
	}
	
	/** Closes stream. */
	public void close() throws IOException {
		stream.close();
	}
}
