/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import org.opencores.Conf;

/** SPC, consisting of Conf.ADD_GROUP_SIZE full adders ports:<p>
  * <code>0 A0		1 B0<p>
  * 2 A1		3 B1<p>
  * 4 A2		5 B2<p>
  * 6	A3		7 B3<p>
  * 8 CIN		9 S0<p>
  * 10 S1		11 S2<p>
  * 12 S3		13 COUT </code>*/
public class NodeSPC extends IndexedNode {
	/** number of inputs, they must be listed first in ports */
	public static final int NINPUTS = 9;
	/** index of carry input port */
	public static final int CARRY_IN_PORT = 8;
	/** index of carry ouput port */
	public static final int CARRY_OUT_PORT = 13;
	/** adder this one receives carry */
	public NodeSPC prev = null;
	/** adder this one passes carry */
	public NodeSPC next = null;
	/** first adder of the group */
	public NodeSPC group = null;
	
  /** constructs new adder */
	public NodeSPC() {
		super(8+1+4+1);
		for(int i = NINPUTS; i < width; i++) dir[i] = OUTPUT;
	}
	
	/** constructs new 4b adder from four full adders */
	public NodeSPC(NodeAdd a0, NodeAdd a1, NodeAdd a2, NodeAdd a3) {
		super(8+1+4+1);						// init ports, ...
		for(int i = NINPUTS; i < width; i++) dir[i] = OUTPUT;
		ports[0] = a0.ports[0];	// copy ports
		ports[1] = a0.ports[1];
		ports[2] = a1.ports[0];
		ports[3] = a1.ports[1];
		ports[4] = a2.ports[0];
		ports[5] = a2.ports[1];
		ports[6] = a3.ports[0];
		ports[7] = a3.ports[1];
		ports[8] = a0.ports[a0.CARRY_IN_PORT];
		ports[9] = a0.ports[a0.NINPUTS];
		ports[10] = a1.ports[a1.NINPUTS];
		ports[11] = a2.ports[a2.NINPUTS];
		ports[12] = a3.ports[a3.NINPUTS];
		ports[13] = a3.ports[a3.CARRY_OUT_PORT];
	}
	
	/** creates 4bit adder from four NodeAdds, looking at .next.	  
	  * Some output ports can remain <code><b>null</b></code>.
	  * @param group lowest bit full adder
	  * @param fillWith this global net is linked with unused inputs */
	public NodeSPC(NodeAdd group, NetGlobal fillWith) {	
		super(8+1+4+1);						// init ports, ...		
		for(int i = NINPUTS; i < width; i++) dir[i] = OUTPUT;
				
		NodeAdd prev = null, a = group;
		ports[CARRY_IN_PORT] = a.ports[a.CARRY_IN_PORT];
		int param = 4;
		float fx = 0, fy = 0;
		for(int i = 0; i < 4; i++) {
			prev = a;
			ports[i*2+0] = a.ports[0];	// copy ports
			ports[i*2+1] = a.ports[1];
			ports[i+NINPUTS] = a.ports[a.NINPUTS];			
			if(a.next == null) {
				param = i+1;
				break;
			}
			fx += a.fx; // place in centre
			fy += a.fy;			
			a = a.next;
		}
		for(int i = param; i < 4; i++) { // assign unused ports
			ports[i*2+0] = fillWith;
			ports[i*2+1] = fillWith;
		}
		fx /= param;
		fy /= param;
		// assign carry to last sum+1 or cout 
		ports[param+NINPUTS] = prev.ports[prev.CARRY_OUT_PORT]; 
	}

	public String toString() {
		return "SPC "+super.toString();
	}
	
	/** creates new (unlinked) node with same parameters.
	  * @return duplicated node */
	public Object clone() {
		NodeAdd n = new NodeAdd();
		n.duplicate(this);
		return n;
	}
	
	/** duplicates parameters from n to this
	  * @param n node to copy data from */
	public void duplicate(NodeAdd n) {
		duplicate((Node)n);
	}
	
	/** returns SPCs x position. 	  
	  * @param idx SPC index
	  * @return x position */
	public static int posX(int idx) {
		int x = idx % Conf.SPC_X;		
		return x*Conf.SPC_STEP_X+Conf.SPC_START_X;
	}
	
	/** returns SPCs y position. 	  
	  * @param idx SPC index
	  * @return y position */
	public static int posY(int idx) {
		int y = idx / Conf.SPC_X;
		return y;
	}
	
	/** Returns its x position.
	  * @return x position */
	public int posX() {
		return posX(idx);
	}
	
	/** returns its y position
	  * @return y position */
	public int posY() {
		return posY(idx);
	}
	
	/** returns index in array, based on position in matrix
	  * @param x x position
	  * @param y y position
	  * @return array index  */
	public static int indexOf(int x, int y) {		
		int nSPC = (x-Conf.SPC_START_X+Conf.SPC_STEP_X)/Conf.SPC_STEP_X;
		return nSPC	+ y*Conf.SPC_X;
	}
}
