/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import java.io.IOException;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;


/** Node, that can be routable - all nodes that are passed
  * to routing phase should be NodeRoutable type. */
public class NodeRoutable extends Node {
	/** Number of inputs for routable cell */
  public final static int NINPUTS_ROUTABLE = 8;
  /** designates whether this node can be moved */
  //public boolean unMoveable = false;
  
	/** Neighbours coordinates */
	public final static int neighCoor[][] = {
		{1,0}, {-1,0},  {0,1}, {0,-1},
		{3,-1}, {-3,1}, {2,3}, {-2,-3}
	};
	
	/** Cost per connection */
	private static final float costs[] = {
		1.1f, 1.1f, 1.1f, 1.1f,
		1.4f, 1.4f, 1.5f, 1.5f
	};
	
	/** Cost to get here, used by routing */
	public float cost;
	
	/** All neighbouring nodes. */
	public NodeRoutable neigh[] = new NodeRoutable[NINPUTS_ROUTABLE];
		
	/** Bit field, designating <b>0</b> for port that has
	  * routable connection and <b>1</b> for port without routable connection. */
	public int portsUnassigned;
	
	/** @return <b>true</b> whether connection is possible for port <code><i>port</i></code> using wire segment <code><i>segment/i></code> */
	public boolean isConnectable(int port, int segment) {
		return false;
	}
	
	/** Owners of wire segments.
	  * <em>IMPORTANT: NodeRoutable.segments designate where segment outputs relative
		* to node, while architecture (in architecture specification) is defined different! </em>*/
	public Net segments[] = new Net[NINPUTS_ROUTABLE];
	/** Number of free segments left. */
	public int nSegments = 0;
	
	/** Costructs new node.
	  * @param width node width */
	public NodeRoutable(int width) {
		super(width);
	}
	
	/** Costruct dummy node, which is capable of routing, but
	  * does not hold any data */
	public NodeRoutable(int x, int y) {
		super(0);
		this.x = x;
		this.y = y;
		this.name = "#";
	}
	
	/** Calculates cost of connection <i>c</i>-<i>neighbour</i> 
	  * Connection cost must be strictly greater than zero!	  
	  * @param neighbourIdx dest segment; they should be connectable
	  * @return cost of connection */
	public float calcCost(int neighbourIdx) {
		return costs[neighbourIdx];
	}	
	
	/** Returns segment index with opposite direction.
	  * @param segment
	  * @return opposite segment to s */
	public static final int opposite(int s) {
		return s^1;		
	}
	
	/** Returns less loaded neighbour index */
	protected int lessLoadedNeighbour() {
		// TODO: find best neighbour
		return 0;
	}
	
	/** Writes (necessary) bistream representation of <code>this</code> object to <code>stream</code>.
		* This method writes bitstream for unused LUT and FF.
	  * @param stream stream to write to */
	public void writeBitstream(BitStreamWriter stream) throws IOException {
		if((x>0)&&(x<Conf.X+1)&&(y>0)&&(y<Conf.X+1)) {
			// GPC dummy
			stream.write(0, 16); stream.write(0, 16); // LUT5 = 0
			
			stream.write(2, 2); // D4 = I2
			stream.write(3, 2); // D3 = I7
			stream.write(1, 2); // D2 = I1
			stream.write(3, 2); // D1 = I6
			stream.write(0, 2); // D0 = I0

			stream.write(3, 2); // CLK = GND		
			stream.write(3, 2); // RST = VCC
			stream.write(3, 2); // SET = VCC
		} else { // IOC dummy
			stream.write(3, 3); // output type
			stream.write(3, 3); // input type
			stream.write(lessLoadedNeighbour(), 3); // D = I0
		}
				
		/* write routing data
		 * IMPORTANT: NodeRoutable.segments designate where segment outputs relative
		 * to node, while architecture (in architecture specification) is defined different! */
		for(int s = NINPUTS_ROUTABLE-1; s >= 0; s--) {
			NodeRoutable n = neigh[s];
			if(segments[s]==null) { // routing resource not used
				stream.write(lessLoadedNeighbour(), 3);
			} else { // neighbour exists
				int found = -1;				// find out where is source - which neighbour point to this
				for(int i = 0; i < NINPUTS_ROUTABLE; i++) {
					Net nt = n.segments[opposite(i)];
					if(segments[s] == nt) {found = i; break; }
				}
				if(found < 0) throw new Error("Cannot find net source");
				stream.write(found, 3);				
			}
		}		
	}
}
