/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import org.opencores.Conf;

/** LUT cell type class. It implements function, and allows joining
  * last port is always output, first width-1 are always inputs */
public class NodeLUT extends Node {
	/** which port is output port */
	public int OUT;
	
  /** function data stored here parameters are ordered
	  * like in ports, 8 bits per byte */
	public byte func[];
	
	/** length of function data
		* @return number of possible inputs */
	private int size()
	{return Math.max((1<<(width-1))/8,1);}	
	
	/** Constructor, that joins two specified adjacent LUTs and creates
	  * new one. Removes matching inputs and repairs function accordingly.
	  * But it does not remove LUTs nor nets from graph.<p>
	  * <p>
	  * Parent MUST drive child's input, and parent MUST NOT have any other
	  * childs. It is not possible to do this more generally, since we may
	  * have loops.	  
	  * @param a parent LUT
	  * @param b child LUT
	  * @param updateNets whether this function should update net ports
	  */
	public NodeLUT(NodeLUT a, NodeLUT b, boolean updateNets) {	
		int width = 0;
		int wa=0,wb=0;
		int pa[],pb[];					// where does input come from		
		int la[],lb[];					// prev. LUT's input is linked to which dest. LUT
		
		/* TODO: unoptimized - we don't really need all those fancy tables */
		/* find out what input connections are */
		pa = new int[a.width-1];
		pb = new int[b.width-1];
		la = new int[a.width-1];
		lb = new int[b.width-1];
		Net aOut = a.ports[a.width-1];		
				
		for(int i = 0; i < a.width-1; i++ ) {
			pa[wa++] = i;
			la[i] = width++;
		}
		
		boolean parent = false;
		for(int i = 0; i < b.width-1; i++ ) {
			if(b.ports[i] != aOut) {		  // ignore parent-child connections
				int found = -1;
				for(int j = 0; j < wa; j++) {		// already in?
					if(a.ports[pa[j]] == b.ports[i]) { found = j;break;}
				}
				if(found < 0) {
					pb[wb++] = i;	// not in pa, add in pb
					lb[i] = width++;
				} else lb[i] = found;	// link to a's part
			} else {
				lb[i] = -1; // link to a's output
				parent = true;
			}
		}
		if(!parent) throw new Error(a.name+" is not parent of "+b.name+"!");
		
		/** we now know our width */		
		setWidth(width+1);
		OUT = width;
		func = new byte[size()]; // init function
		for(int i = 0; i < size(); i++) func[i] = 0;
		
		/* we now have build new LUTs inputs: pa+pb */
		/* Destination LUT's data will be build, giving right
			 inputs to previous functions */
		for(int v = 0; v < 1 << width; v++ ) {
			/* build b input */
			int bIn=0;
			int ao = a.calc(v&((1<<wa)-1));
			for(int i = 0; i < b.width-1; i++ ) {
			  bIn <<= 1;
			  if(lb[i] >= 0) 
					bIn |= (v >> lb[i])&1;
				else bIn |= ao;
			}
			/* build b part */
			int result = b.calc(bIn);
			if((result|1) != 1) throw new Error("Value not 0 or 1!");
			func[v>>3] |= (byte) (result << (v&7));
		}
		
	  /* nets are connected and repaired here */			
		dir[width] = OUTPUT; // width=#inputs here
		ports[width] = b.ports[b.width-1];
		int w = 0;
		for(int i = 0; i < wa; i++) {
			ports[w++] = a.ports[pa[i]];
		}
		for(int i = 0; i < wb; i++) {
			ports[w++] = b.ports[pb[i]];
		}
		name = new String(a.name+"+"+b.name);
		
		if(updateNets) {
			a.unlinkNets();
			b.unlinkNets();
			linkNets();
		}
		
		fx = (a.fx+b.fx) / 2;
		fy = (a.fy+b.fy) / 2;		
	}
	
	/** constructs empty LUT
	  * @param width number of ports (including output!) */
	public NodeLUT(int width) {
		super(width);
		OUT = width - 1;
		dir[OUT] = OUTPUT;
		func = new byte[size()];
	}
	
	/** Function, that returns number of inputs if two specified LUTs
	  * were joined, not counting  matching inputs. <p>
	  * Parent can drive child's input. It is not possible to do this
	  * more generally, since we can have loops.
	  * @param a parent LUT
	  * @param b child LUT
	  * @return number of inputs */
	public static final int NumJoinInputs(NodeLUT a, NodeLUT b) {	
		int width = a.width-1;
		
		/* find out what input connections are */
		Net aOut = a.ports[a.width-1];		
				
		for(int i = 0; i < b.width-1; i++ ) {
			if(b.ports[i] != aOut) {		  // ignore parent-child connections
				int found = -1;
				for(int j = 0; j < a.width-1; j++) {		// already in?
					if(a.ports[j] == b.ports[i]) { found = j;break;}
				}
				if(found < 0) width++;				
			}
		}
		return width;
	}
	
	/** Determines, if a is a child of b. 
	  * @param a any LUT
	  * @param b any LUT
	  * @return <b>true</b> if <i>a</i> is child of <i>b</i>, and <b>false</b> when 
	  * <i>b</i> is child of <i>a</i>, or <i>a</i> and <i>b</i> are uncorrelated. */
	public static final boolean isAChild(NodeLUT a, NodeLUT b) {	
		if((a == null)||(b == null)) return false;
		
		/* find out what input connections are */
		Net aOut = a.ports[a.width-1];		
		Net bOut = b.ports[b.width-1];
		boolean aChild = false;
		boolean bChild = false;
						
		for(int i = 0; i < b.width-1; i++ ) {
			if(b.ports[i] == aOut) {
				bChild = true; break;
			}
		}
		for(int i = 0; i < a.width-1; i++) {
			if(a.ports[i] == bOut) {
				aChild = true; break;
			}
		}
		/* they are connected in loop => one with lower hash value is child */
		if((aChild)&&(bChild)) return (a.hashCode() < b.hashCode());
			
		return aChild;		
	}
	
	/** Function, that returns number of inputs if two specified LUTs
	  * were joined, not counting  matching inputs nor inbetween connections.
	  * @param a LUT
	  * @param b  LUT
	  * @return number of inputs */
	public static final int NumJoinInputsEx(NodeLUT a, NodeLUT b) {	
		if(isAChild(a, b)) return NumJoinInputs(b, a);
		else return NumJoinInputs(a, b);
	}

	/** Permutates inputs by pi.
		* @param pi permutation (<code>pi[source_index]=destination_index<code>), must be same length as <code>width-1<code> */
	public void permutateInputs(int[] pi) throws Exception {		
		byte nf[] = new byte[func.length];
		if(pi.length != width-1) throw new Exception("Invalid permutation");
		for(int v = 0; v < 1 << (width-1); v++ ) { // for every possible input
			int nv=0;		// new input vector
			for(int i = 0; i < pi.length; i++){		   // build input vector			
				nv |= ((v >> i)&1)<<pi[i];
			}
			nf[nv/8] = (byte) (calc(nv) << (nv&7));
		}
		func = nf;	// update
	}
	
	/** Permutates inputs by inv(pi).
		* @param pi inverse permutation (<code>pi[dest_index]=source_index<code>), must be same length as <code>width-1<code> */
	public void permutateInputsInv(int[] pi) throws Exception {		
		byte nf[] = new byte[func.length];
		if(pi.length != width-1) throw new Exception("Invalid permutation");
		for(int v = 0; v < 1 << (width-1); v++ ) { // for every possible input
			int nv=0;		// new input vector
			for(int i = 0; i < pi.length; i++){		   // build input vector			
				nv |= ((v >> i)&1)<<pi[i];
			}
			nf[nv>>3] = (byte) (calc(nv) << (nv&7));
		}
		func = nf;	// update
	}
			
	/** Calculates LUTs function
	  * @param inputs inputs passed to function, LSB is first parameter
		* @return 0 or 1 */
	public final int calc(int inputs) {	
		return (func[inputs>>3]>>(inputs&7))&1;
	}
			
	public String toString() {
		String s = "LUT"+(width-1)+super.toString()+"  ";
		for(int i = size()-1; i >= 0; i--)
			s = s + Integer.toHexString(((int)func[i])&(0xFF)).toUpperCase();
		return s+Conf.NL;
	}
	
	/** creates new (unlinked) node with same parameters.
	  * @return duplicated node */
	public Object clone() {
		NodeLUT n = new NodeLUT(width);
		n.duplicate(this);
		return n;
	}
	
	/** duplicates parameters from n to this
	  * @param n node to copy data from */
	public void duplicate(NodeLUT n) {
		duplicate((Node)n);
		for(int i = 0; i < func.length; i++)
			func[i] = n.func[i];
	}
}
