package org.opencores.structure;

import java.io.IOException;
import org.opencores.structure.NodePort;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;

public class NodeIOC extends IndexedNode {
	/** input type */
	public int typeI = 3;
	/** output type */
	public int typeO = 3;

	/** Constructs new node with given direction */
	public NodeIOC() {
		super(2);
		dir[0] = INPUT;
		dir[1] = OUTPUT;
	}

	/** Constructs new node based on port.
	  * @param fillWith net to fill unused inputs with */
	public NodeIOC(NodePort p, NetGlobal fillWith) {
		super(2);
		dir[0] = INPUT;
		dir[1] = OUTPUT;
		name = p.name;
		x = p.x; y = p.y;
		fx = p.fx; fy = p.fy;			
		if(p.dir[0] == INPUT) {
			ports[0] = p.ports[0];			
		} else {			
			ports[0] = fillWith;			
			ports[1] = p.ports[0];						
		}
	}

	public String toString() {		
		return "IOC "+super.toString();
	}
	
	/** creates new (unlinked) node with same parameters.
	  * @return duplicated node */
	public Object clone() {
		NodeIOC n = new NodeIOC();
		n.duplicate(this);
		return n;
	}
	
	/** duplicates parameters from n to this
	  * @param n node to copy data from */
	public void duplicate(NodeIOC n) {
		duplicate((Node)n);
	}

	/** returns port's x position. Ports are arranged:
	  * <code>{(0..X-1,0),(0..X-1,Y-1),(0,0..Y-1),(X-1,0..Y-1)}</code>
	  * @param pi port index
	  * @return x position */
	public static int posX(int pi) {		
		if(pi < 2*Conf.X) { // upper of bottom line 						
			if(pi >= Conf.X) return pi - Conf.X;
				else return pi;				
		} else { // left or right ports
			pi -= 2*Conf.X;
			if(pi >= Conf.Y) return Conf.X;
			else return -1;			
		}		
	}
	
	/** Returns port's y position. Ports are arranged:
	  * <code>{(0..X-1,0),(0..X-1,Y-1),(0,0..Y-1),(X-1,0..Y-1)}</code>
	  * @param pi port index
	  * @return y position */
	public static int posY(int pi) {		
		if(pi < 2*Conf.X) { // upper of bottom line 						
			if(pi >= Conf.X) return Conf.Y;
			else return -1;			
		} else { // left or right ports
			pi -= 2*Conf.X;
			if(pi >= Conf.Y)	return pi - Conf.Y; else return pi;							
		}		
	}
	
  /** Returns its x position.
	  * @return x position */
	public int posX() {
		return posX(idx);
	}
	
	/** Returns its y position.
	  * @return y position */
	public int posY() {
		return posY(idx);
	}
	
	/** Returns index in array, based on position in matrix.
	  * @param x x position
	  * @param y y position
	  * @return array index  */
	public static int indexOf(int x, int y) {		
		if(y == -1) return x;					 //top
		if(y == Conf.Y) return x+Conf.X; // bottom
		if(x == -1) return y + 2*Conf.X; // left
		return y + 2*Conf.X+Conf.Y;		
	}
	
	/** @return <b>true</b> whether connection is possible for pin <code><i>pin</i></code> using wire segment <code><i>segment/i></code> */
	public boolean isConnectable(int pin, int segment) {
		return true;	// any input can be connected to IO
	}
	
	/** Writes (necessary) bistream representation of <code>this</code> object to <code>stream</code>.
		* This method writes bitstream for unused LUT and FF.
	  * @param stream stream to write to */
	public void writeBitstream(BitStreamWriter stream) throws IOException {
		stream.write(typeO, 3); // output type
		stream.write(typeI, 3); // input type
		
		int found = -1;					// find matching input
		for(int s = 0; s < NINPUTS_ROUTABLE; s++)
			if(segments[s] == ports[0]) {found = s; break; }
		
		if(found < 0) throw new Error("Cannot find IOC's "+name+" input.");
		stream.write(found, 3); // D
				
		/* write routing data */
		for(int s = NINPUTS_ROUTABLE-1; s >= 0; s--) {
			NodeRoutable n = neigh[s];
			if(n == null) { // routing resource could not be used
				stream.write(lessLoadedNeighbour(), 3);
			} else { // neighbour exists
				Net nt = n.segments[opposite(s)];
				if(nt == null) { // n doesn't need its neighbour
					stream.write(lessLoadedNeighbour(), 3);
				} else {
					// find out where is source
					found = -1;
					for(int i = 0; i < NINPUTS_ROUTABLE; i++)
						if(segments[i] == nt) {found = i; break; }
					if(found < 0) throw new Error("Cannot find net source");
					stream.write(found, 3);
				}
			}
		}		
	}
}
