/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import java.io.IOException;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;

/** GPC, consisting of one 5->1 LUT and one FFs.
  * @see IndexedNode */
public class NodeGPC extends IndexedNode {  
	/** number of inputs for LUT */
	public final static int NINPUTS_LUT = Conf.K;
	/** a's clock */
	public final static int CLK = NINPUTS_LUT + 0;
	/** a's async set */
	public final static int SET = NINPUTS_LUT + 1;
	/** a's async reset */
	public final static int RST = NINPUTS_LUT + 2;	
	/** a's output */
	public final static int OUT = NINPUTS_LUT + 3;
		
  /** LUT inside <code><i>this</i></code> */
	public NodeLUT a;
	/** FF inside <code><i>this</i></code> */
	public NodeFF ra;
		
	/** groups specified elements into one GPC
	  * @param a LUT
	  * @param ra FF connected to a's output	  
	  * @param updateNets whether nets should be relinked to new GPC 
	  * @param fillWith this global net is set to unassigned inputs
	  * If parameters are null, they are ignored. <p>
	  * <em>WARNING: no checking for FFs. Nets between x-rx have to
	  * be deleted manually</em> */	  
	public NodeGPC(NodeLUT a, NodeFF ra, boolean updateNets, NetGlobal fillWith) {
		super(NINPUTS_LUT+3+1);		 // setup ports and link instances				
		this.a = a;	// parent		
		this.ra = ra;		
		dir[OUT] = dir[OUT] = OUTPUT;

		int w = 0;
		Net aOut;
		name = "";
		
		/* copy all inputs from a and b to this */
		if(a != null) {
			name = "("+a.name+",";
			aOut = a.ports[a.width-1];				
			for(int i = 0; i < a.width-1; i++ ) ports[w++] = a.ports[i];
		} else {
			if(ra != null) throw new Error("FF does not have preceding LUT.");
			aOut = null;
			name = "(null,";
		}
	
		if(w > NINPUTS_LUT) throw new Error("Formed GPC too large.");
		
		/* fill unused input ports */
		for(int i = w; i < NINPUTS_LUT; i++) {
			ports[i] = fillWith;
		}
		
		/* copy LUT output, if FF not used, otherwise FF's output */
		if(ra != null) {
			ports[CLK] = ra.ports[ra.CLK];
			ports[SET] = ra.ports[ra.SET];
			ports[RST] = ra.ports[ra.RST];
			ports[OUT] = ra.ports[ra.OUT];
			name += ra.name+")";
		} else {
			ports[CLK] = ports[SET] = ports[RST] = fillWith; // not used
			if(a != null) ports[OUT] = a.ports[a.width-1]; // a's output
			else ports[OUT] = null;// else not used
			name += "null)";
		}

	  if(updateNets) {
			if(a != null) a.unlinkNets();			
			if(ra != null) ra.unlinkNets();			
			linkNets();
		}
		/* calculate node gravity point */		
		int cnt = 0;

		fx = a.fx; fy = a.fy;
		if(ra != null) {
			fx += ra.fx; fy += ra.fy;
			fx /= 2; fy /= 2;
		}
	}
	
	public String toString() {		
		return "GPC "+super.toString();
	}
		
	/** returns GPCs x position.
	  * @param idx GPC index
	  * @return x position */
	public static int posX(int idx) {
		int x = idx % Conf.X;
		return x;		
	}
	
	/** returns GPCs y position. 	  
	  * @param idx GPC index
	  * @return y position */
	public static int posY(int idx) {
		int y = idx / Conf.X;
		return y;
	}
	
	/** Returns its x position.
	  * @return x position */
	public int posX() {
		return posX(idx);
	}
	
	/** returns its y position
	  * @return y position */
	public int posY() {
		return posY(idx);
	}
	
	/** Returns index in array, based on position in matrix.
	  * @param x x position
	  * @param y y position
	  * @return array index  */
	public static int indexOf(int x, int y) {
		return x + y*Conf.X;
	}
	
	/** @return <b>true</b> whether connection is possible for pin <code><i>pin</i></code> using wire segment <code><i>segment/i></code> */
	public boolean isConnectable(int pin, int segment) {
		// LUT input can connect to any input (otherwise inputs can be interchanged)
		if(pin < NINPUTS_LUT) return true;		
		if(pin == CLK) return segment == 4;
		if(pin == RST) return (segment|1) == 1; // 0 or 1
		if(pin == SET) return (segment|1) == 3; // 2 or 3
		throw new Error("Invalid Pin");
	}
	
	/** Precalculated table for input selection. */
	private static final int INPUT_CONF[] = {
    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   0, 273,   1, 546,   2, 274,  18,
    -1,  68, 341,  69, 614,  70, 342,  86,  -1,   4, 277,   5, 550,   6, 278,  22,
    -1, 136, 409, 137, 682, 138, 410, 154,  -1,   8, 281,   9, 554,  10, 282,  26,
    -1,  72, 345,  73, 618,  74, 346,  90, 819,  56, 313,  57, 570,  58, 314,  -1,
    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  12, 285,  13, 558,  14, 286,  30,
    -1,  76, 349,  77, 622,  78, 350,  94,  -1,  60, 317,  61, 574,  62, 318,  -1,
    -1, 140, 413, 141, 686, 142, 414, 158,  -1, 143, 415, 159, 687, 175, 431,  -1,
    -1, 188, 445, 189, 702, 190, 446,  -1, 831, 191, 447,  -1, 703,  -1,  -1,  -1,
    -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1, 192, 465, 193, 738, 194, 466, 210,
    -1, 196, 469, 197, 742, 198, 470, 214,  -1, 199, 471, 215, 743, 231, 487,  -1,
    -1, 200, 473, 201, 746, 202, 474, 218,  -1, 203, 475, 219, 747, 235, 491,  -1,
    -1, 248, 505, 249, 762, 250, 506,  -1,1011, 251, 507,  -1, 763,  -1,  -1,  -1,
    -1, 204, 477, 205, 750, 206, 478, 222,  -1, 207, 479, 223, 751, 239, 495,  -1,
    -1, 252, 509, 253, 766, 254, 510,  -1,  -1, 255, 511,  -1, 767,  -1,  -1,  -1,
    -1, 972, 989, 973,1006, 974, 990,  -1,  -1, 975, 991,  -1,1007,  -1,  -1,  -1,
    -1,1020,1021,  -1,1022,  -1,  -1,  -1,1023,  -1,  -1,  -1,  -1,  -1,  -1,  -1
  };
	
	/** Writes (necessary) bistream representation of <code>this</code> object to <code>stream</code>.		
	  * @param stream stream to write to */
	public void writeBitstream(BitStreamWriter stream) throws IOException {
		for(int f = a.func.length-1; f >= 0; f--)	// LUT5
			stream.write(a.func[f], 8);

		int inputs = 0;	// determine which inputs we use
		for(int i = 0; i < NINPUTS_LUT; i++)
			for(int s = 0; s < NINPUTS_ROUTABLE; s++)
				if(ports[i] == segments[s]) { inputs |= 1<<s; break; }

		stream.write(INPUT_CONF[inputs], 10); // D0..5

		if(ra == null) {
			stream.write(3, 2); // CLK = GND
			stream.write(3, 2); // RST = VCC
			stream.write(3, 2); // SET = VCC
		} else {
			if(ports[CLK] == segments[4]) stream.write(0, 2); // CLK = I4
			else if(ports[CLK] instanceof NetGlobal) {
				NetGlobal ng = (NetGlobal)ports[CLK];
				if(ng.type == ng.GCLK0)	stream.write(1, 2); // CLK = GCLK0
				else if(ng.type == ng.GCLK1)	stream.write(2, 2); // CLK = GCLK1
				else throw new Error("Cannot connect GCLKx signal to GPC "+name);
			} else throw new Error("Cannot connect CLK signal to GPC "+name);
			
			if(ports[RST] == segments[0]) stream.write(0, 2); // RST = I0
			if(ports[RST] == segments[1]) stream.write(1, 2); // RST = I1
			else if(ports[RST] instanceof NetGlobal) {
				NetGlobal ng = (NetGlobal)ports[RST];
				if(ng.type == ng.GRST)	stream.write(2, 2);	 // RST = GRST
				else throw new Error("Cannot connect GRST signal to GPC "+name);
			} else stream.write(3, 2); // RST = VCC
			
			if(ports[RST] == segments[2]) stream.write(0, 2); // SET = I2
			if(ports[RST] == segments[3]) stream.write(1, 2); // SET = I3
			else if(ports[SET] instanceof NetGlobal) {
				NetGlobal ng = (NetGlobal)ports[SET];
				if(ng.type == ng.GSET)	stream.write(2, 2);	 // SET = GSET
				else throw new Error("Cannot connect GSET signal to GPC "+name);
			} else stream.write(3, 2); // SET = VCC
		}
		
		/* write routing data */
		for(int s = NINPUTS_ROUTABLE-1; s >= 0; s--) {
			NodeRoutable n = neigh[s];
			if(n == null) { // routing resource could not be used
				stream.write(lessLoadedNeighbour(), 3);
			} else { // neighbour exists
				Net nt = n.segments[opposite(s)];
				if(nt == null) { // n doesn't need its neighbour
					stream.write(lessLoadedNeighbour(), 3);
				} else {
					// find out where is source
					int found = -1;
					for(int i = 0; i < NINPUTS_ROUTABLE; i++)
						if(segments[i] == nt) {found = i; break; }
					if(found < 0) throw new Error("Cannot find net source");
					stream.write(found, 3);
				}
			}
		}		
	}
}
