/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import java.util.Vector;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;

/** node structure suitable for P&R process */
public class Node {	
	/** input type wire */
	public final static int INPUT = 1;	
	/** output type wire */
	public final static int OUTPUT = 2; 
	/** input/output type wire */
	public final static int INOUT = 3;  
	
	/** node name */
	public String name;
	/** nets that are connected to ports */
	public Net ports[];		 
	/** weights of specific ports */ 
	public float weight[]; 
	/** directions of ports Edifp.Port.* */
	public int dir[];			 
	/** placed position */
	public int x,y;	
	/** RxR position */
	public float fx,fy;		 
	/** misc flag */
	public boolean flag = false;
	/** visited flag */
	public boolean visited = false;
	/** misc used temp */
	public int temp;
	/** temp RxR position */ 
	public float nfx,nfy;	 
	/** # function inputs */
	public int width = 0;
	/** node link - for graph duplication, etc.*/
	public Node link = null;	 

	/** Allocates new node.
	  * @param width number of ports */
	public Node(int width) {
		setWidth(width);
	}
	
	/** Constructs new node. <p>
		*	<em>WARNING: this function should only be called by child constructors.</em> */
	Node() {
	}
	
	/** changes Node's width, set all ports to INPUT direction
	  * @param width number of ports */
	public void setWidth(int width) {
		this.width = width;
		ports = new Net[width];
		weight = new float[width];
		dir = new int[width];
		for(int i = 0;i < width;i++) {
			weight[i] = 1.f;
			dir[i] = INPUT;						
			ports[i] = null;
		}
	}
		
	/** @return graph structure text representation */
	public String toString() {
		String s = "Node: "+name+"(x"+x+" y"+y+" temp"+temp+") nets("+width+")";
		char comma = ':';
		for(int i = 0; i < width; i++) {
			if(ports[i] != null) 
				if(dir[i] == OUTPUT)
					s = s + comma + " *" + ports[i].name;
				else s = s + comma + " " + ports[i].name;
			else 
				if(dir[i] == OUTPUT)
					s = s + comma + " *null";
				else s = s + comma + " null";
			comma = ',';
		}
		return s + Conf.NL;
	}
	
	/** creates new (unlinked) node with same parameters.
	  * @return duplicated node */
	public Object clone() {
		Node n = new Node();
		n.duplicate(this);		
		return n;
	}
	
	/** duplicates parameters from n to this
	  * @param n node to copy data from */
	public void duplicate(Node n) {
		setWidth(n.width);
		name = n.name; flag = n.flag; temp = n.temp;
		x = n.x; y = n.y; fx = n.fx; fy = n.fy;
		for(int i = 0; i < n.width; i++) {
			dir[i] = n.dir[i];
			weight[i] = n.weight[i];			
			ports[i] = n.ports[i];
		}
	}
	
	/** Disconnect nets to this node, based on ports */
	public void unlinkNets() {
		for(int i = 0; i < ports.length; i++) {
			if(ports[i] != null)
				if(ports[i].output == this) ports[i].output = null;
				else {
					ports[i].inputs.removeElement(this);
					ports[i] = null;
				}
		}
	}
	
	/** Connect nets to this node, based on ports */
	public void linkNets() {
	  for(int i = 0; i < ports.length; i++)
			if(ports[i] != null)
				if(dir[i] == INPUT)
					ports[i].inputs.addElement(this);
				else ports[i].output = this;
	}
}
