/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import java.util.Vector;
import java.io.IOException;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;

/** Graph, that holds all nodes and nets. */
public class Graph {	
	/** graph name */
	public String name;									
	/** all the nodes */
	public Vector nodes = new Vector();
	/** all the nets */ 	
	public Vector nets = new Vector();	
	/** all global nets */
	public NetGlobal global[] = new NetGlobal[NetGlobal.NUM_GLOBAL];	
	/** Nodes by position. Not initialized on start.
	  * @see costructSegments */
	public NodeRoutable pos[][];
	/** special resources */
	public NodeSR SR[] = new NodeSR[Conf.NUM_SR];
	/** specifies if graph has been mapped */
	public boolean mapped = false;
	
	/** Constructs neighb field of all NodeRoutable nodes. */
	public void constructSegments() {
		pos = new NodeRoutable[Conf.X+2][];
		for(int x = 0; x < pos.length; x++) {
			pos[x] = new NodeRoutable[Conf.Y+2];
			for(int y = 0; y < pos[x].length; y++)
				pos[x][y] = null;
		}
		
		/* build hash table based on Node's position */
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node)nodes.elementAt(i);
			if(n instanceof NodeRoutable) {
				NodeRoutable r = (NodeRoutable)n;
				pos[r.x+1][r.y+1] = r;
			}
		}
		
		for(int x = 0; x < pos.length; x++)
			for(int y = 0; y < pos[x].length; y++) {
				if(pos[x][y] == null) pos[x][y] = new NodeRoutable(x-1, y-1);
				pos[x][y].nSegments = 0;
			}
				
		/* allocate segments and neighbours */
		for(int x = 0; x < pos.length; x++)
			for(int y = 0; y < pos[x].length; y++) {
				NodeRoutable r = pos[x][y];
				for(int j = 0; j < r.NINPUTS_ROUTABLE; j++) {
					r.segments[j] = null;							// mark unused
					if(r.neigh[j] != null) r.nSegments++;
					int nx = r.x+1+r.neighCoor[j][0];
					int ny = r.y+1+r.neighCoor[j][1];					
					boolean corner = ((nx==0)||(nx==Conf.X+1))&&((ny==0)||(ny==Conf.Y+1));						
					if((nx>=0)&&(ny>=0)&&(nx<Conf.X+2)&&(ny<Conf.Y+2)&&(!corner)) {
						r.neigh[j] = pos[nx][ny];
						r.nSegments++;
					} else r.neigh[j] = null;
				}
			}
					
		/* add special resources and link them with the first line */		
		for(int i = 0; i < Conf.NUM_SR; i++) {
			NodeSR r = SR[i];
			if(r == null) continue; // not used => skip
			for(int s = 0; s < r.NINPUTS_ROUTABLE; s++) {
				int nx = r.x + r.neighCoor[s][0] + 1;
				int ny = r.y + r.neighCoor[s][1] + 1;
				boolean corner = ((nx==0)||(nx==Conf.X+1))&&((ny==0)||(ny==Conf.Y+1));
				if((nx>=0)&&(ny>=0)&&(nx<Conf.X+2)&&(ny<Conf.Y+2)&&(!corner)) {
					r.neigh[s] = pos[nx][ny];	// create bidirectional link
					r.nSegments++;
					pos[nx][ny].neigh[r.opposite(s)] = r;
					pos[nx][ny].nSegments++;
				}
			}
		}
	}

	/** Sets all segments costs to MAX_VALUE. */
	public void clearNodes() {
		for(int x = 0; x < pos.length; x++)
			for(int y = 0; y < pos[x].length; y++) {
				NodeRoutable r = pos[x][y];
				r.temp = 0;			
				for(int j = 0; j < r.NINPUTS_ROUTABLE; j++)
					r.cost = Float.MAX_VALUE;	
			}
		for(int i = 0; i < Conf.NUM_SR; i++)
			if(SR[i] != null) SR[i].temp = 0;	
	}

	/** Empty all channel segments, and allocates those
	  * (on perimeter) that cannon be used. */
	public void emtpySegments() {
		for(int x = 0; x < pos.length; x++)
			for(int y = 0; y < pos[x].length; y++) {
				NodeRoutable r = pos[x][y];				
				r.nSegments = 0;
				for(int j = 0; j < r.NINPUTS_ROUTABLE; j++) {
					r.segments[j] = null;
					if(r.neigh[j] != null) r.nSegments++;
				}
				r.portsUnassigned = 0;
				for(int j = r.ports.length-1; j >= 0; j--) {
					r.portsUnassigned <<= 1;
					if(r.dir[j] == r.INPUT) r.portsUnassigned|=1;
				}
			}	
		for(int i = 0; i < Conf.NUM_SR; i++)
			if(SR[i] != null) {
				NodeRoutable r = SR[i];
				r.portsUnassigned = 0;
				for(int j = r.ports.length-1; j >= 0; j--) {
					r.portsUnassigned <<= 1;
					if(r.dir[j] == r.INPUT) r.portsUnassigned|=1;
				}
			}
	}

	/** Constructs new empty graph. */
	public Graph() {
		for(int i = 0; i < NetGlobal.NUM_GLOBAL; i++)
			global[i] = new NetGlobal(i);	
	}
	
	/** outputs graph to String */
	public String toString() {
		return toString("");
	}
	
	/** outputs graph to String
	  * @param comment graph comment */
	public String toString(String comment) {
		String s = "Graph: "+name+" "+comment+Conf.NL 
			+"Number of nodes: "+nodes.size()+Conf.NL+Conf.NL;
		for(int i = 0; i < nodes.size(); i++)	s = s + nodes.elementAt(i).toString();
		for(int i = 0; i < NetGlobal.NUM_GLOBAL; i++)	s = s + global[i].toString();
		s = s + "Nets("+nets.size()+"):" + Conf.NL + Conf.NL;
		for(int i = 0; i < nets.size(); i++)	s = s + nets.elementAt(i).toString();
		return s;
	}

	/** sets random values for x from 0..X-1 and y from 0..Y-1
	  * @param X maximum horizontal position
	  * @param Y maximum vertical position */
	public void setRandomPositions(int X, int Y) {	
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node) nodes.elementAt(i);
			n.x = (int)(Math.random()*X);
			n.y = (int)(Math.random()*Y);
		}
	}
	
	/** checks graph for possible annomalies (unidirected lines) and
	  * can posts warnings.
	  * @param report reports warnings to Conf.log if true
	  * @return number of warnings */
	public int check(boolean report) {
		int nerrs = 0;
		for(int i = 0; i < nodes.size(); i++) {			
			Node n = (Node) nodes.elementAt(i);
			n.temp = 0;			
			/* check for broken (unidirectional) links => not repairable */
			for(int j = 0; j < n.width; j++) {
				if(n.dir[j] == n.INPUT) {
					if(!n.ports[j].inputs.contains(n)) {
						if(report) Conf.log.println("(node) "+n.name+" - (net) "+n.ports[j].name+" Input connection not bidirectional.");				
						nerrs++;
					}
				} else if(n.ports[j] != null) {
					if(n.ports[j].output != n) {
						if(report) Conf.log.println("(node) "+n.name+" - (net) "+n.ports[j].name+" Output connection not bidirectional.");
						nerrs++;
					}									
				} else n.temp++;
			}		
		}
		/* check broken (unidirectiona) links in opposite direciton */
		for(int i = 0; i < nets.size(); i++) {
			Net nt = (Net) nets.elementAt(i);
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				n.temp++; // update number of ports
			}
			Node n = (Node) nt.output; // update number of ports
			if(n!=null) n.temp++;
		}
		
		/* update also inputs from global nets */
		for(int i = 0; i < global.length; i++) {
			Net nt = (Net) global[i];
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				n.temp++; // update number of ports
			}			
		}

		/* check for number true number of references */
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node) nodes.elementAt(i);
			if(n.temp != n.width) {			
				if(report) Conf.log.println(n.name+" has inconsistent number of net references.");
				nerrs++;
			}
			
			for(int j = 0; j < n.width; j++) { // mark all used nets
				Net nt = n.ports[j];
				if(nt != null) nt.link = nt;
			}
		}
		
		/* unmark nets in graph
		 * at the same time mark nodes referenced by nets in graph */
		for(int i = 0; i < nets.size(); i++) {
			Net nt = (Net) nets.elementAt(i);
			nt.link = null;	
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n != null)	n.flag = true; // mark used
			}
			if(nt.output != null) nt.output.flag = true;
		}
		
		/* search for marked nets
		 * at the same time unmark nodes in graph */
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node) nodes.elementAt(i);
			n.flag = false;	// unmark node
			for(int j = 0; j < n.width; j++)
			  if(!((n.ports[j]) instanceof NetGlobal)) {
				if((n.ports[j] == null)||
					((n.ports[j] != null)&&(n.ports[j].link != null))) { // net not in graph?
					if(n.dir[j] == n.OUTPUT) {	// default output is null - not used						
						if(n.ports[j] != null) {
							if(report) Conf.log.println(n.name+"'s output port("+j+"+) not in graph!");
							nerrs++;
						}
					}	else { // default input is GND, add new node											
						if(report) Conf.log.println(n.name+"'s input port("+j+"+) not in graph!");
						nerrs++;
					}					
					
				}
			}
		}

		/** search for referenced nodes, not in graph */
		for(int i = 0; i < nets.size(); i++) {
			Net nt = (Net)nets.elementAt(i);
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n == null) { n = new Node();	n.flag = true; }
				if((n.flag)&&(!(n instanceof NodeSR))) {
					if(report) Conf.log.println("Node "+n.name+" not in graph.");
					nerrs++;
				}
			}
			Node n = nt.output;
			if(n == null) { n = new Node();	n.flag = true; }
			if((n.flag)&&(!(n instanceof NodeSR))) {
				if(report) Conf.log.println("Node "+n.name+" not in graph.");
				nerrs++;
			}
		}		
		return nerrs;
	}
	
	/** repairs graph, removing floating nodes and nets
	  * @param report reports warnings to Conf.log if true
	  * @return number of repaired annomalies */
	public int repair(boolean report) {				
		int nreps = 0;
	  /* mark all used nets */
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node) nodes.elementAt(i);
			for(int j = 0; j < n.width; j++) {
				Net nt = n.ports[j];
				if(nt != null) nt.link = nt; // mark all used nets
			}			
		}
		/* unmark nets in graph
		 * at the same time mark nodes referenced by nets in graph */
		for(int i = 0; i < nets.size(); i++) {
			Net nt = (Net) nets.elementAt(i);
			nt.link = null;	 // unmark net
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n != null)	n.flag = true; // mark used
			}
			if(nt.output != null) nt.output.flag = true;
		}
		
		/* search for marked nets
		 * at the same time unmark nodes in graph */
		for(int i = 0; i < nodes.size(); i++) {
			Node n = (Node) nodes.elementAt(i);
			n.flag = false; // unmark node
			for(int j = 0; j < n.width; j++)
			  if(!((n.ports[j]) instanceof NetGlobal)) {
				if((n.ports[j] == null)||
					((n.ports[j] != null)&&(n.ports[j].link != null))) { // net not in graph?
					if(n.dir[j] == n.OUTPUT) {	// default output is null - not used						
						if(n.ports[j] != null) {
							if(report) Conf.log.println("Marking "+n.name+"'s output port("+j+"+) floating");
							n.ports[j] = null; // unlink
						} else nreps--; // not count this one
					}	else { // default input is GND, add new node					
						Net nt = new Net(); // add new net and floating node
						NodePrim np = new NodePrim(NodePrim.VCC);
						if(n.ports[j] == null) nt.name = np.name = n.name+"_fi"+j;
						else nt.name = np.name = n.ports[j].name;						
						n.ports[j] = np.ports[0] = nt;
						nt.inputs.addElement(n);
						nodes.addElement(nt.output = np);
						nets.addElement(nt);
						if(report) Conf.log.println("Marking net "+nt.name+" floating");
					}
					nreps++;
				}
			}
		}
		return nreps;
	}
	
	/** copies (x,y) to (fx,fy) suitable for display */
	public void createFloatPositions() {
		for(int i = 0; i < nodes.size(); i++) {
		  Node n = (Node) nodes.elementAt(i);
		  n.fx = n.x; n.fy = n.y;
		}
	}
	
	/** Calculates positions (.x, .y) of IndexedNodes */
	public void calcPositions() {
		for(int i = 0; i < nodes.size(); i++) {
		  Node n = (Node) nodes.elementAt(i);
		  if(n instanceof IndexedNode) { // need to calculate index?
		 	  n.x = ((IndexedNode)n).posX();
		 	  n.y = ((IndexedNode)n).posY();
		  }
		}
	}
	
	/** Writes (necessary) bistream representation of <code>this</code> object to <code>stream</code>.
	  * <em>This method should be called after routing</em>
	  * @param stream stream to write to
	  * @param clear whether should reset unused elements */
	public void writeBitstream(BitStreamWriter stream, boolean clear) throws IOException {
		/** write all GPC columns */
		for(int x = 0; x < Conf.X; x++) {
			boolean empty;
			if(clear)	empty = false;
			else { // check if column has something to be programmed
				empty = true; 
				for(int y = 0; y < Conf.Y; y++)
					if(pos[x+1][y+1].width > 0) {empty = false; break;}
					else {
						for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++)
							if(pos[x+1][y+1].segments[s] != null) {empty = false; break;}
						if(!empty) break;
					}
			}
			if(empty) { // write column if not empty
				stream.write(Conf.BSH_GPC, 16);	stream.write(x, 16);
				stream.write(0, 16); stream.write(54, 16);
				for(int y = 0; y < Conf.Y; y++)
					pos[x+1][y+1].writeBitstream(stream);			
			}
		}
		
		/** write top IOC row */
		boolean empty;
		if(clear)	empty = false;
		else { // check if column has something to be programmed
			empty = true; 
			for(int x = 0; x < Conf.X; x++)
				if(pos[x+1][0].width > 0) {empty = false; break;}
				else {
					for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++)
						if(pos[x+1][0].segments[s] != null) {empty = false; break;}
					if(!empty) break;
				}
		}
		if(empty) { // write column if not empty
			stream.write(Conf.BSH_IOC, 16);	stream.write(0, 16);
			stream.write(0, 16); stream.write(25, 16);
			for(int x = 0; x < Conf.X; x++)
				pos[x+1][0].writeBitstream(stream);			
			stream.write(0, 8);
		}
		
		/** write bottom IOC row */		
		if(clear)	empty = false;
		else { // check if column has something to be programmed
			empty = true; 
			for(int x = 0; x < Conf.X; x++)
				if(pos[x+1][Conf.Y+1].width > 0) {empty = false; break;}
				else {
					for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++)
						if(pos[x+1][Conf.Y+1].segments[s] != null) {empty = false; break;}
					if(!empty) break;
				}
		}
		if(empty) { // write column if not empty
			stream.write(Conf.BSH_IOC, 16);	stream.write(1, 16);
			stream.write(0, 16); stream.write(25, 16);
			for(int x = 0; x < Conf.X; x++)
				pos[x+1][Conf.Y+1].writeBitstream(stream);			
			stream.write(0, 8);
		}	
		
		/** write left IOC column */		
		if(clear)	empty = false;
		else { // check if column has something to be programmed
			empty = true; 
			for(int y = 0; y < Conf.Y; y++)
				if(pos[0][y+1].width > 0) {empty = false; break;}
				else {
					for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++)
						if(pos[0][y+1].segments[s] != null) {empty = false; break;}
					if(!empty) break;
				}
		}
		if(empty) { // write column if not empty
			stream.write(Conf.BSH_IOC, 16);	stream.write(2, 16);
			stream.write(0, 16); stream.write(25, 16);
			for(int y = 0; y < Conf.Y; y++)
				pos[0][y+1].writeBitstream(stream);			
			stream.write(0, 8);
		}
		
		/** write right IOC column */		
		if(clear)	empty = false;
		else { // check if column has something to be programmed
			empty = true; 
			for(int y = 0; y < Conf.Y; y++)
				if(pos[Conf.X+1][y+1].width > 0) {empty = false; break;}
				else {
					for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++)
						if(pos[Conf.X+1][y+1].segments[s] != null) {empty = false; break;}
					if(!empty) break;
				}
		}
		if(empty) { // write column if not empty
			stream.write(Conf.BSH_IOC, 16);	stream.write(3, 16);
			stream.write(0, 16); stream.write(25, 16);
			for(int y = 0; y < Conf.Y; y++)
				pos[Conf.X+1][y+1].writeBitstream(stream);			
			stream.write(0, 8);
		}		
	}
}
