/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.structure;

import java.util.Vector;
import org.opencores.Conf;

/** Class that represents horizontal or vertical channel. <p>
  * Channel connect two adjacent switch boxes. <p>
  * Each channel is attached to node (share same source position) and
  * consists of multiple wire segments. <i>nets</i> Net array
  * identifies which of them are still free.<p>
  * Nodes are adjacent when there exist wire segment between them.<p>
  * List of all nodes channel connects to can be constructed.
  * @see adjacentChannels */
public class Channel {	
	/** This channel position. */
	public int x, y;
	/** Node channel is connected to. */
	public Node node;
	/** List of all adjacent channels. */
	public Channel adjacentChannels[];
	/** Used by wave algorithm - cost to this node */
	public float cost;	
	/** Nets going through this channel, each allocating one wire segment. */
	public Net nets[] = new Net[Conf.NUM_SEGMENTS*Conf.NUM_SEG_WAY];
	/** Number of nets allocated on this channel. */
	public int nNets = 0;	
	
	/** Returns max. segments per way
	  * @return number of used segments */
	public int maxSegments() {
		int max = 0;
		for(int dir = 0; dir < Conf.NUM_SEG_WAY; dir++) {
			int sum = 0;
			for(int sub = 0; sub < Conf.NUM_SEGMENTS; sub++)
				if(nets[dir * Conf.NUM_SEGMENTS + sub] != null) sum++;			
			if(sum > max) max = sum;
		}
		return max;
	}

	/** Constructs new channel.
	  * @param idx channel position */
	public Channel(int x, int y) {
		this.x = x;
		this.y = y;
	}
	
	/** Identifies which segment we will reach, if we are on segment
	  * <i>segment</i> and going to neighbour index <i>channel</i>.
	  * This transform is based on USM.
	  * @see "Universal Switch Modules for FPGA Design, Yao-Wen Chang, D. F. Wong, C. K. Wong"
	  * @param segment segment we are on
	  * @param channel index of neighbour
	  * @return neighbours segment, negative if invalid move */
	public static final int switchsTo(int segment, int channel) {
		int ssub = segment % Conf.NUM_SEGMENTS;
		int sdir = segment / Conf.NUM_SEGMENTS;		
		switch(Conf.ADJACENT_DIR[channel]*Conf.NUM_SEG_WAY + sdir) {
			case 0+0:  // N-N
			case 1+4:  // S-S			
			case 3+12: // E-E
			case 2+8:  // W-W
			case 2+0:  // W-N									
			case 3+4:  // E-S
			case 0+12: // N-E
			case 1+8:  // S-W
				return ssub + sdir*Conf.NUM_SEGMENTS;

			case 3+0:  // E-N				
			case 2+4:  // W-S				
			case 0+8:  // N-W
			case 1+12: // S-E
				return sdir*(Conf.NUM_SEGMENTS+1) - ssub-1;						

			case 1+0:  // S-N
			case 0+4:  // N-S
			case 3+8:  // E-W
			case 2+12: // W-E			
				return -1;	// direct return not possible
			default:
				throw new Error();
		}
	}
}
