/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.routing;

import java.util.Vector;
import java.util.Arrays;
import java.util.Comparator;
import org.opencores.util.Heap;
import org.opencores.structure.*;
import org.opencores.Conf;
import java.util.NoSuchElementException;

/** Just places wires using frontwave technique.
  * @see "Vaughn Betz, Jonathan Rose: VPR: A New Packing, Placement and Routing Tool for FPGA Research, 1997 International Workshop on Field Programmable Logic and Applications"<p>
  * @see "Michael J. Alexander, Gabriel Bobins: New Performance-Driven FPGA Routing Algorithms"<p>
  * @see "Guy G. Lemieux, Stephen D. Brown: A Detailed Routing Algorithm for Allocating Wire Segments in Field-Programmable Gate Arrays" <p>*/
public class Wavefront {
	/** Number of iteration of ripping-up and re-routing. */
	public final static int NUM_ITERATIONS = 45;
	/** Cost modifier, if we failed to place it */	
	public final static float COST_MODIFIER = 1.2f;
	
	/** graph to work on */
	private Graph g;
	/** Current net's tree is stored here. Vector of Segments. */
	private Vector tree;
	/** Vector of all unrouted nets. (failed to route) */
	public Vector unroutedNets;
	
	/** Distribution of used wire connections by length */
	public int statLenCnt[] = new int[5*Conf.X];
	/** Distribution of wire length. */
	public int statWireCnt[] = new int[5*Conf.X];
	/** Distribution of #hops. */
	public int statHopCnt[] = new int[5*Conf.X];

	/** starts new prerouting session based on graph g
	  * @param g graph, it should have channels initialized */
	public Wavefront(Graph g) {
		this.g = g;
		for(int i = 0; i < 5*Conf.X; i++)
			statLenCnt[i] = statWireCnt[i] = statHopCnt[i] = 0;		
	}
	
	/** Performs one iteration of routing.
	  * @return <b>true</b> if successful, <b>false</b> otherwise */
	public boolean route() {
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			n.temp = 0;
		}
		/* create Array from Vector */
		Net nets[] = new Net[g.nets.size()];
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net) g.nets.elementAt(i);
			nt.cost = 1f;			//TODO: build costs before
			nets[i] = nt;			
		}		
		
		for(int i = 0; i < NUM_ITERATIONS; i++) { 
			/* descending sort based on cost*/
			Arrays.sort(nets, new Comparator() { 
				public int compare(Object o1, Object o2) {
					Net a = (Net)o1, b = (Net)o2;
					if(a.cost < b.cost) return 1;
					else if(a.cost > b.cost) return -1; else return 0;
				}		
				public boolean equals(Object o) {throw new Error();}
			});
			spread(nets);
			
			//TODO: check for failed constraints and add then to unroutedNets list
			
			if(unroutedNets.size() == 0) return true; // great - solution was found			
			for(int j = 0; j < unroutedNets.size(); j++) {
				Net nt = (Net)unroutedNets.elementAt(j);
				nt.cost *= COST_MODIFIER;
				Conf.log.print(nt.name+" ");
			}
			Conf.log.println(i+": "+unroutedNets.size());
			System.out.print(i+": "+unroutedNets.size()+"   ");
		}
		Conf.log.println();
		System.out.println();
		return false;
	}

	/** Spread wave around all nets in graph */
	public void spread(Net nets[]) {
		g.emtpySegments();	// clear all allocated nets
		unroutedNets = new Vector();
		for(int i = 0; i < nets.length; i++) {
			Net nt = nets[i];
			if(Conf.debug) Conf.log.println("Routing "+nt.name+" ");
			if(spread(nt)) { // was successful? => result in tree
				/** allocate found path */				
				for(int j = 0; j < tree.size(); j++) {
					Segment s = (Segment) tree.elementAt(j);
					if(s.prev != null) {
						if(s.prev.c.segments[s.s] != null) throw new Error("Segment should be free!");
						s.prev.c.segments[s.s] = nt;
						s.prev.c.nSegments--;
						if(s.prev.c.nSegments < 0) throw new Error("Invalid count.");
					}
					if(Conf.debug) Conf.log.print("("+s.c.x+","+s.c.y+") ");
				}
				if(Conf.debug) Conf.log.println();
				if(Conf.debug) Conf.log.println("OK");
			} else { // unable to find route				
				if(Conf.debug) Conf.log.println("FAILED");
				unroutedNets.add(nt);
			}
		}
	}

	/** Spread wave around specified net in graph */
	public boolean spread(Net nt) {
		if(nt.output == null) throw new Error("Errorneous net "+nt.name+"!");		
		//TODO: handle special resource
		NodeRoutable sc = (NodeRoutable) nt.output;
			
		/* Searches all available segments from CO, if they can
		 * reach sinks. */		
		Heap wave = new Heap();					// we will keep wave in heap
		tree = new Vector(1);
						
		Segment source = new Segment(sc, -1, null);
		wave.add(source);								// source node starts the wave					
		g.clearNodes();									// set all costs to MAX_VALUE
		source.c.cost = 0.f;						// sink has zero cost
			
		if(Conf.debug) Conf.log.println("O:"+nt.output.name+"("+nt.output.x+","+nt.output.y+")");
		/* mark nodes, when we reach them we will unmark them */
		for(int k = 0; k < nt.inputs.size(); k++) {
			Node n = ((Node)nt.inputs.elementAt(k));
			n.temp++;
			if(Conf.debug) Conf.log.print(n.name+"("+n.x+","+n.y+") ");
		}
		if(Conf.debug) Conf.log.println();

		//TODO: calculate bounding box, and limit search

		int toCover = nt.inputs.size(); // # of sink nodes to be covered
		while(toCover > 0) {						// do wave search until all covered
			if(wave.isEmpty()) break;
			Segment cur = (Segment)wave.remove();
			if(Conf.debug) {
				if(cur.prev != null) Conf.log.print(cur.prev.c.name+"["+cur.prev.c.x+","+cur.prev.c.y+"]-");
				Conf.log.print(cur.c.name+"["+cur.c.x+","+cur.c.y+"] ");
			}
			for(int j = 0; j < cur.c.neigh.length; j++) 
			if(cur.c.segments[j] == null) { // segment still free?
				if(Conf.debug) Conf.log.print(j);
				NodeRoutable neigh = cur.c.neigh[j];
				if(neigh != null) {
					if(Conf.debug) Conf.log.print("!");
					float nc = cur.c.cost + cur.c.calcCost(j);
					if(neigh.temp > 0) { // we found one sink
						boolean found = false;
						/* add path and set zero costs, so that wave spreads
						 * around it first */
						for(int p = 0; p < neigh.ports.length; p++)
							if((neigh.dir[p] == neigh.INPUT)&&(neigh.ports[p] == nt)
							 &&(((neigh.portsUnassigned>>p)&1)==1)
							 &&(neigh.isConnectable(p, NodeRoutable.opposite(j)))) {
								neigh.portsUnassigned ^= 1>>p;
								neigh.temp--;
								toCover--;	// we just covered one node port
								found = true;
								/* do statistics */
								int wireLen = (int)nc;
								if(wireLen >= statHopCnt.length) wireLen = statHopCnt.length - 1;
								statHopCnt[wireLen]++;
							}	
						
						if(found)	findWayBack(new Segment(neigh, j, cur), wave);
					} else {
						if(nc < neigh.cost) {		// do not go back
							if(neigh.cost != Float.MAX_VALUE) { // already in wave								
								Segment found = null;
								for(int i = 0; i < wave.size(); i++) {
									Segment s = (Segment)wave.elementAt(i);
									if(s.c == neigh) { found = s; break; }
								}								
								if(found != null) {
									wave.remove(found);
									neigh.cost = nc;			// add to wave
									found.prev = cur;			// we have better predecessor
									found.s = j;
									wave.add(found);
								}	else {
									neigh.cost = nc;			// add to wave
									wave.add(new Segment(neigh, j, cur));								
								}
							} else {
								neigh.cost = nc;			// add to wave
								wave.add(new Segment(neigh, j, cur));
							}
						}
					}
				}
			}
			if(Conf.debug) Conf.log.println(" >"+cur.c.cost);
		}
		if(Conf.debug) Conf.log.println();
		if(toCover == 0) return true;		 // all covered => solution found
		return false;											 // we've tried all free CO's segments
	} // route()
	
	/** Sets path cost we've searched so far to zeroes.
	  * @param s sink Segment to start from	  	  
	  * @param wave wave-front currently processed */	
	void findWayBack(Segment s, Heap wave) {		
		Segment first = s;
		while(s.c.cost != 0f) {						
			tree.add(s);
			s.c.cost = 0.f;	// add new path and mark it to be searched first			
			Segment t = s.prev;
			if(t!=null) {
				if(t.c.segments[s.s] != null) throw new Error("Segment should be free!");
				statLenCnt[Math.abs(s.c.x-t.c.x+s.c.y-t.c.y)]++;
			}
			try {
				wave.remove(s);	// removes same element first, if it exists
			} catch (NoSuchElementException e) {}	// just ignore if it doesn't exist
			wave.add(s);
			s = t;
		}
		wave.repair();
		int dist = Math.abs(first.c.x - s.c.x) + Math.abs(first.c.y - s.c.y);
		if(dist >= statWireCnt.length) dist = statWireCnt.length - 1;
		statWireCnt[dist]++;
	}
	
	/** Returns number of lowest free segments
	  * @return min free segments */
	public int minFreeSegment() {
		int min = NodeGPC.NINPUTS_ROUTABLE;
		for(int x = 0; x < g.pos.length; x++)
			for(int y = 0; y < g.pos[x].length; y++)
				if(min > g.pos[x][y].nSegments)
					min = g.pos[x][y].nSegments;
		return min;
	}
}

/** Class used for spreading wave.
  * <code>c</code> is accessible from <code>prev</code> via segnent index <code>s</code>*/
class Segment implements Comparable {
	/** During search - where we came from */
	Segment prev;
	/** channel we are on */
	NodeRoutable c;
	/** segment we are on */
	int s;
	/** constructs new segment */
	Segment(NodeRoutable c, int s, Segment prev) {
		this.c = c;
		this.s = s;
		this.prev = prev;
	}	
	
	/** Method required by <i>Compareable</i> interface.
	* @param c channel to compare to
	* @see Comparable */
	public int compareTo(Object o) {
		NodeRoutable ch = ((Segment)o).c;
		if(c.cost < ch.cost) return -1;
		if(c.cost > ch.cost) return 1;
		return 0;		
	}
}
