/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.placement;

import java.util.Vector;
import java.util.Arrays;
import java.util.Comparator;
import org.opencores.structure.*;
import org.opencores.Conf;

/** This algorithm tries to arrange graph nodes similarly to
  * graph drawing algorithms. Nodes may overlap and they are
  * placed in RxR space. Also this class contains some preplacement
  * functions, that are based on RxR placement. After preplacement
  * we have valid placement.
  *
  * Annealing uses two forces - temperature and distancing force.
  * First nets acts like elastics and tries to approach nodes together,
  * on the other hand nodes are distanced by the power of distancing force.
  *
  * Preplacement consist of three steps:
  *  - preplacement of SPCs
  *  - preplacement of GPCs
  *  - preplacement of ports */
public class PrePlacement {
	/** graph representation we are working on */
	private Graph g;
	/** Number of GPCs detected by preplacement */
	public int nGPC = 0;
	/** Number of Ports detected by preplacement */
	public int nIOC = 0;
	/** Number of total LUT inputs. */
	public int nInputs = 0;
		
	/** Initializes RxR placement.
	  * @param g graph to do placement on. */
	public PrePlacement(Graph g) {
		this.g = g;
	}
	
	/** Changes working graph.
	  * @param g graph to work on
	  * @see Graph */
	public void setGraph(Graph g) {
		this.g = g;
	}
	
	/** vector, that contains all GPCs */
	public Vector vnl = new Vector();
	/** position array, knowing number of LUTs at that position */
	private int pos[] = new int[Conf.X*Conf.Y];
	
	/** Assigns valid positions for GPCs. This placement is
	  * improved by main SA placement. */
	public void prePlacementGPC() {		
		int nluts = 0;		
		nInputs = 0;
						
		/* build indexes from position */
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodeGPC) {	// round positions to ints
				nInputs += ((NodeGPC)n).a.width-1;
				int nx = Math.round(n.fx);
				int ny = Math.round(n.fy);
				if(nx < 0) nx = 0;
				if(ny < 0) ny = 0;
				if(nx >= Conf.X) nx = Conf.X-1;
				if(ny >= Conf.Y) ny = Conf.Y-1;				
				((NodeGPC)n).idx = NodeGPC.indexOf(nx, ny);
				pos[((NodeGPC)n).idx]++; // place here
				nluts++;
				vnl.addElement(n);
			}
		}
		nGPC = nluts;
		if(nluts > Conf.NUM_GPC) // pretty obvious
			throw new Error("Number of GPCs too large for this architecture.");
			
		/* We will work with indexes from now on */
			
		//TODO: sort by importance (ascending!)
		
		for(int i = 0; i < vnl.size(); i++) { // check if alone, otherwise place at first
			NodeGPC n = (NodeGPC) vnl.elementAt(i);// empty position
			if(pos[n.idx] > 1) {
				int r = 0, nidx = -1;	// search around for a new place
				/* searches in index 1D space instead of 2D matrix,
				 * but this is preplacement */
				while(nidx < 0) {
					r++;
					if((n.idx-r>0)&&(pos[n.idx-r]==0)) {nidx = n.idx-r; break; }
					if((n.idx+r<pos.length)&&(pos[n.idx+r]==0)) {nidx = n.idx+r; break; }
				}
				pos[n.idx]--; // change position
				n.idx = nidx;
				pos[n.idx]++;
			}
			n.x = n.posX();
			n.y = n.posY();
		}
	}
	
	/** Assigns valid positions for IOC. This placement is
	  * improved by main SA placement. */
	public void prePlacementIOC() {
		int nIOs = 0;
		int pos[] = new int[Conf.NUM_IOC];
		
		Vector vnp = new Vector();
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodeIOC) {	// round positions to ints
				nIOs++;
				vnp.addElement(n);
				n.nfy = n.weight[0];		  // importance factor
			}
		}
		
		nIOC = nIOs;
		if(nIOs > Conf.NUM_IOC) // pretty obvious
			throw new Error("Number of ports too large for this architecture.");
		
		/* descending sort groups based on impotance */
		Object[] ni = vnp.toArray();		
		Arrays.sort(ni, new Comparator() { // sort based on .nfy
			public int compare(Object o1, Object o2) {
				NodeIOC a = (NodeIOC)o1, b = (NodeIOC)o2;
				if(a.nfy < b.nfy) return 1;
				else if(a.nfy > b.nfy) return -1; else return 0;
			}		
			public boolean equals(Object o) {throw new Error();}
		});
				
		/* greedy placement */
		for(int i = 0; i < pos.length; i++) pos[i] = 0; // clear positions
		for(int i = 0; i < ni.length; i++) {			
			int bestj = 0;
			float best = Float.MAX_VALUE;
			/* search all empty positions */
			for(int j = 0; j < pos.length; j++)	if(pos[j] == 0) {
				float tmp = portDist(((NodeIOC)ni[i]).fx, ((NodeIOC)ni[i]).fy, j);
				if(tmp < best) {	best = tmp; bestj = j; }				
			}
			pos[bestj]++;// assign port to position bestj
			((NodeIOC)ni[i]).idx = bestj;
			((NodeIOC)ni[i]).x = ((NodeIOC)ni[i]).posX(); // calc position
			((NodeIOC)ni[i]).y = ((NodeIOC)ni[i]).posY();
		}
	}
	
	/** returns manhattan distance between specified position (x,y) and
	  * port with index pi. Ports are arranged:
	  * <code>{(0..X-1,0),(0..X-1,Y-1),(0,0..Y-1),(X-1,0..Y-1)}</code>
	  * @param x x position
	  * @param y y position
	  * @param pi port index
	  * @return manhattan distance */
	private float portDist(float x, float y, int pi) {
		int px, py;
		if(pi < 2*Conf.X) { // upper of bottom line 						
			if(pi >= Conf.X) {
				px = pi - Conf.X;
				py = Conf.Y;
			}	else {
				px = pi;
				py = -1;
			}			
		} else { // left or right ports
			pi -= 2*Conf.X;
			if(pi >= Conf.Y) {
				py = pi - Conf.Y;
				px = Conf.X;
			}	else {
				py = pi;
				px = -1;
			}
		}
		return Math.abs(px-x)+Math.abs(py-y);
	}
	
	/** Does preplacement.	  
	  * @see prePlacementGPC	  
	  * @see prePlacementPort
	  * <em>NOTE: Netlist must be mapped first.</em>*/
	public void prePlacement() {		
		prePlacementGPC();
		prePlacementIOC();
	}
}
