/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.placement;

import org.opencores.structure.*;
import org.opencores.Conf;

/** This is main class, consisting of simulated annealing routine.
  * Nodes must be are already placed correctly (using preplacement)
  * and their pos (.x, .y) and  index .idx are set. All work here
  * is done using indexes, but coordinates are also maintaned to
  * speed up placement. <p>
  * Alpha functions and some other dinamic parameters are adopted from VPR.
  * @see "Vaughn Betz, Jonathan Rose: VPR: A New Packing, Placement and Routing Tool for FPGA Research, 1997 International Workshop on Field Programmable Logic and Applications" */
public class Annealing {
	/** Coefficient, which linearly changes number of inner loop executions. */
	private static final int INNER_NUM_K = 10;
	/** Initial temperature coefficient. Larger, more moves will be accepted at start. */
	private static final int INITIAL_TEMP_K = 1;// orig 20
	/** Number of inner loop execution times. Set at reset(). */
	private int INNER_NUM; 
	/** Stoping treshold. Larger it is sooner we will stop. <em>Should be positive.</em> */
	private static final double DONE_CONSTANT = .001; //orig .005
	
	/** graph we are working on */
	private Graph g;
	/** table of all possible GPC positions (and current assignments) */
	private NodeGPC posGPC[] = new NodeGPC[Conf.NUM_GPC];	
	/** table of all possible Port positions (and current assignments) */
	private NodeIOC posIOC[] = new NodeIOC[Conf.NUM_IOC];
	/** current temperature, @see setTemp */
	private double temp;
	/** current total wiring cost, @see calcTotalWiringCost, getWiringCost */
	private int wiringCost;
	/** number of (possible) moves performed per current temperature */
	private int tempMoves = 0;
	/** number of (accepted) moves performed per current temperature */
	private int tempAccepted = 0;
	/** total wiring change per current temperature */
	private int tempWiringChange = 0;
	
	/** Construct and resets new SA object on graph g.
	  * @param g mapped, preplaced graph to work on */
	public Annealing(Graph g) {
		this.g = g;
		buildDist();
		reset();		
	}
	
	/** Resets SA. Should be called every time, when SA is restarted. */
	public void reset() {
		for(int i = 0; i < Conf.NUM_GPC; i++) posGPC[i] = null;		
		for(int i = 0; i < Conf.NUM_IOC; i++) posIOC[i] = null;
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodeGPC) {
				int idx = ((NodeGPC)n).idx;
				if(posGPC[idx] != null) throw new Error("GPC pos not unique");
				posGPC[idx] = (NodeGPC)n;
			}	else {
				int idx = ((NodeIOC)n).idx;
				if(posIOC[idx] != null)
					throw new Error("Port pos not unique");
				posIOC[idx] = (NodeIOC)n;
			}
		}
		wiringCost = calcTotalWiringCost();	
		
		/* set annealing default parameters */
		INNER_NUM = (int)(INNER_NUM_K * Math.pow(g.nodes.size(), 4.d/3.d));
		setTemp(INITIAL_TEMP_K * stdDev());				
	}
	
	/** Changes temperature and resets temperature specific data. <p>
	  * <em> Should be called after each temp change. </em>
	  * @param temp new temperature */
	public void setTemp(double temp) {
		wiringCost += tempWiringChange;
		tempMoves = tempAccepted = tempWiringChange = 0;
		this.temp = temp;
	}
	
	/** Discrete temperature update - alpha function. <p>
	  * First number is acceptance ratio, second coefficient K,
	  * and T = K * T */
	private final double alpha[][]={
		{.15,.8},
		{.8,.95},
		{.96,.9},
		{1.0,.5}};
	/** Changes temperature - function alpha(T) */
	public void updateTemperature() {
		if(tempMoves == 0) return;	// shouldn't occur, but ignore
		int segment = 0;
		double acceptRatio = ((double)tempAccepted) / tempMoves;// find segment, in
		while(acceptRatio > alpha[segment][0]) segment++;		// which acc. ratio falls
		setTemp(temp * alpha[segment][1]);	// update temperature
	}
	
	/** simulates anneal at current temperature <i>temp</i>. */
	public void anneal() {
		while(!stopCriteria()) {
			Node n = (Node) g.nodes.elementAt((int)(Math.random()*g.nodes.size()));
			if(n instanceof NodeGPC) move((NodeGPC)n);
			else move((NodeIOC)n);
			tempMoves++; // one possible move occured
		}
	}
	
	/** Decides whether we should stop annealing at current temperature.
	  * <em>Also called InnerLoop criterion.</em>
	  * @return <b>true</b> when annealing at current temperature should be stopped, <b>false</b> otherwise */
	private final boolean stopCriteria() {
		return (tempMoves > INNER_NUM);
	}
	
	/** Decides whether annealing has good solution and it is unlikely
	  * that we will find it, so we should stop. <p>
	  * <em>NOTE: call this before updateTemperature, since it destroys temperature
	  * dependent data, needed here </em>
	  * @return <b>true</b> when annealing should be stopped, <b>false</b> otherwise */
	public final boolean doneCriteria() {
		return ((tempWiringChange >= 0)
					&&((temp < DONE_CONSTANT*tempWiringChange/g.nets.size())||(tempWiringChange==0)));
	}
	
	/** Tries to move node <i>n</i>, using current Conf.
	  * @param temp current temperature
	  * @param a GPC node to move */
	private void move(NodeGPC a) {		
		int bidx = (int)(Math.random()*Conf.NUM_GPC);
		NodeGPC b = posGPC[bidx];		
		int bx, by;
		if(b != null) {
			bx = b.x;
			by = b.y;
		} else {
			bx = NodeGPC.posX(bidx);
		  by = NodeGPC.posY(bidx);
		}		
		
		int aidx = a.idx;
		/* make decision based on temperature */
		if(move(a, b, bx, by, bidx)) {			
			posGPC[aidx] = b;		// exchange (have in mind that indexes changed by move!)
			posGPC[bidx] = a;			
		}
	}
		
	/** Tries to move node <i>n</i>, using current configuration
	  * @param an IOC node to move
	  * @return wiring cost change */
	private void move(NodeIOC a) {
		int bidx = (int)(Math.random()*Conf.NUM_IOC);
		NodeIOC b = posIOC[bidx];		
		int bx, by;
		if(b != null) {
			bx = b.x;
			by = b.y;
		} else {
			bx = NodeIOC.posX(bidx);
		  by = NodeIOC.posY(bidx);
		}
		
		int aidx = a.idx;
		/* make decision based on temperature */
		if(move(a, b, bx, by, bidx)) {
			posIOC[aidx] = b;	// exchange (have in mind that indexes changed by move!)
			posIOC[bidx] = a;
		}
	}	
	
	/** Decides whether move with delta cost <i>dCost</i> should be taken.
	  * @param dCost movement delta cost
	  * @return <b>true</b> if decision should be taken, <b>false</b> otherwise. */
	private final boolean allowMove(float dCost) {			
		if(dCost < 0) return true; // always move to better state
		/* otherwise accept with probability e^(-dC/T) */
		return (Math.random() <= Math.exp(-dCost/temp));
	}
	
	private float costs[] = null;
	/** Estimates cost (wiring cost change) of move, after <i>n</i> is moved to (<i>x</i>,<i>y</i>). <p>
	  * <em>NOTE: no replacement is made. Does not calculate replacement cost of node at (x,y).</em><p>
	  * This function stores computed costs in <i>costs</i>, which can be used later if move is not accepted.<p>
	  * Proceed as following:<p>
	  * 1. find closest pin on each a's port net, and calculate
	  *    cost to move it to next neighbour or if no other to new position<p>
	  * 2. find closest pin for each a's port net, based on new position
	  * @param n node to calculate movement cost on
	  * @param dest node, we will exchange n with
	  * @param x destination x position
	  * @param y destination y position
	  * @return movement cost */
	private final float moveCost(Node n, int x, int y) {		
		float total_cost = 0;
		float prev_total_cost = 0;		
		
		costs = new float[n.width];
		/* move */		
		n.x = x;
		n.y = y;
		for(int i = 0; i < n.width; i++) { // check all nets connected to n
			Net nt = n.ports[i];
			if((nt == null)||(nt instanceof NetGlobal)) continue;
			nt.link = nt;
		}
		boolean used[] = new boolean[n.width];
		/* save previous state */
		for(int i = 0; i < n.width; i++) { // check all nets connected to n
			Net nt = n.ports[i];			
			if((nt == null)||(nt instanceof NetGlobal)) continue;
			costs[i] = nt.cost;
			used[i] = false;
			if(nt.link==null) continue; // handle special nets
			nt.link = null;
			used[i] = true;
			prev_total_cost += nt.cost;	// prev cost sum for equal number of nodes			
		}
		/* do new ones in separate loop, since nets can be updated inbetween
		 * (many ports on same net!) */
		for(int i = 0; i < n.width; i++) { // check all nets connected to n
			Net nt = n.ports[i];			
			if((nt == null)||(nt instanceof NetGlobal)||(!used[i])) continue; // handle special nets
			nt.cost = MST(nt); // store this port's cost for later
			total_cost += nt.cost;
		}
		return total_cost-prev_total_cost;
	}
	
	/** This function is called when move is not accepted, and updates
	  * net MST costs, to previous moveCost <i>costs</i>	  
	  * @see moveCost */
	private void takeBackMove(Node n) {
		if(costs.length != n.width) throw new Error();
		for(int i = 0; i < n.width; i++) {
			Net nt = n.ports[i];
			if((nt == null)||(nt instanceof NetGlobal)) continue; // skip special nets			
			nt.cost = costs[i];	// just copy them
		}
	}
	
	/** Returns <b>true</b> if node has been moved from <i>a</i> to <i>b</i>*/
	private boolean move(IndexedNode a, IndexedNode b, int bx, int by, int bidx) {
		if(b != null) return false;		//debugging	
		int ax = a.x; // original a's position
		int ay = a.y;		
		//int pwc = calcTotalWiringCost();
		//if(pwc != wiringCost + tempWiringChange) throw new Error();		
		
		/* calculate cost a->b */
		float cost = moveCost((Node)a, bx, by);
		float a_costs[] = costs;
		/* calculate cost b->a */
		if(b != null)	cost += moveCost((Node)b, ax, ay);
			
		//if(calcTotalWiringCost() != pwc + cost) throw new Error();// debugging
		
		/* make decision based on temperature */		
		if(allowMove(cost)) {
			if(b != null) b.idx = a.idx; // coordinates have been moved,							
			a.idx = bidx;								 // update indexes
			
			/* update temp specific parameters */
			tempAccepted++;
			tempWiringChange += cost;
			return true;
		} else {			
			if(b != null) {
				takeBackMove(b);	// update b's MST
				b.x = bx;
				b.y = by;
			}
			costs = a_costs;
			takeBackMove(a);
			a.x = ax;
			a.y = ay;			
			return false;
		}		
	}
	
	/** Precalculated distance based on (x,y) coordinates. */
	private int distance[][];
	
	/** Build estimated #hops for each vector (dx, dy).
	  * Calculated table is stored in <code>distance</code> */	
	private final void buildDist() {
		distance = new int[Conf.X+3][];
		for(int x = 0; x < distance.length; x++) {
			distance[x] = new int[Conf.Y+3];
			for(int y = 0; y < distance[x].length; y++) {
				int nhops = 0, dx = x, dy = y;
				while(manhattan(dx, dy) > 0) {
					int bests = -1, bestd = Integer.MAX_VALUE;
					for(int s = 0; s < NodeRoutable.NINPUTS_ROUTABLE; s++) {
						int dist = Math.abs(dx-NodeRoutable.neighCoor[s][0])
						           + Math.abs(dy-NodeRoutable.neighCoor[s][1]);
						if(dist < bestd) { bestd = dist; bests = s;	}				
					}
					dx -= NodeRoutable.neighCoor[bests][0];
					dy -= NodeRoutable.neighCoor[bests][1];
					nhops++;
				}
				distance[x][y] = nhops;
			}
		}
	}
	
	/** Returns estimated distance of vector (dx, dy).
	  * @param dx vector x component
	  * @param dy vector y component
	  * @return min distance */	
	private final int dist(int dx, int dy) {		
		return distance[Math.abs(dx)][Math.abs(dy)];
	}
		
	/** Returns manhattan distance of vector (dx, dy).
	  * @param dx vector x component
	  * @param dy vector y component
	  * @return distance */	
	private static final int manhattan(int dx, int dy) {
		return Math.abs(dx) + Math.abs(dy);
	}
	
	/** Calculates minimum spanning tree size.	  
	  * @param net to compute MST on
	  * @return MST size */
	private int MST(Net nt) {	  
		/* find all nodes on the net and put it into mst */
		Node mst[]; 
		int dist [];	// current shortest distance to given node
		int size = nt.inputs.size();	
		if(nt.output != null) {		
			if(nt.inputs.size() == 1)	{// handle simple nets faster
				Node a = (Node) nt.inputs.firstElement(); 
				return dist(nt.output.x - a.x, nt.output.y - a.y);
			}
			mst = new Node[++size];
			mst[size-1] = nt.output;
		} else mst = new Node[size];
		dist = new int[size];
		for(int i = 0; i < nt.inputs.size(); i++) //copy inputs
			mst[i] = (Node)nt.inputs.elementAt(i);			
		
		//Conf.log.print(nt.name+":");	//!
		int idx = 0;
		for(int i = 0; i < size; i++) {
			dist[i] = Integer.MAX_VALUE;
			//Conf.log.print("("+mst[i].x+","+mst[i].y+"),");
		}
		
		idx = 0;
		dist[idx] = 0;				//!
		int cost = 0;	// calculate cost using Prim's algorithm
		for(int i = 0; i < size; i++) {			
			int min = Integer.MAX_VALUE;	// find closest available	nodes
			int bestj = 0;
			int x = mst[idx].x;
			int y = mst[idx].y;
			mst[idx] = null;
			for(int j = 1; j < size; j++) if(mst[j] != null) {
				dist[j] = Math.min(dist[j], dist(x - mst[j].x, y - mst[j].y));
				if(min > dist[j]) {
					min = dist[j];
					bestj = j;
				}
			}
			
			cost += dist[idx];
			idx = bestj;
		}
		//Conf.log.println(" = "+cost);
		return cost;		
	}
	
	/** Wiring length is sum of sizes of all minimum spanning trees.
	  * @return total wiring cost */
	private int calcTotalWiringCost() {
		int wiringCost = 0;
		for(int i = 0; i < g.nets.size(); i++) {			
			Net nt = (Net)g.nets.elementAt(i);
			if((nt == null)||(nt instanceof NetGlobal)) continue; // handle special nets
			wiringCost += nt.cost = MST(nt);	// store cost for later
		}
		return wiringCost;
		//Conf.log.println();
	}
		
	/** Checks MST trees currently built with stored number in nets.cost
	  * @return difference
	  * @deprecated, used for debugging only */
	private float checkWiringCost() {
		float cost = 0;
		float tcost = 0;		
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net)g.nets.elementAt(i);
			if((nt == null)||(nt instanceof NetGlobal)) continue; // handle special nets
			float a,b;
			cost += a = nt.cost;
			tcost += b = MST(nt);	// store cost for later
			if(a != b)
				Conf.log.println("!!! "+nt.name+"("+a+','+b+")");
		}		
		return cost-tcost;
	}

	
	/** Calculates standard deviation of one random move for each graph node.
	  * @return standard deviation of cost */
	private double stdDev() {
		double sum = 0;
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);			
			int x = n.x, y = n.y; // save orig pos - just testing			
			int nx = (int)(Math.random()*Conf.X);
			int ny = (int)(Math.random()*Conf.Y);
			double cost = moveCost(n, nx, ny);			
			takeBackMove(n);			
			n.x = x; n.y = y;			// restore			
			sum += cost*cost;
		}		
		Conf.log.println();
		return sum / g.nodes.size();
	}
	
	/** Returns current total wiring cost.
	  * @return total wiring cost */
	public int getWiringCost() {
		return wiringCost;
	}
	
	/** Returns current temperature.
	  * @return current temperature
	  * @see temp */
	public double getTemp() {
		return temp;
	}
}
