/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.mapping;

import org.opencores.structure.*;
import java.util.Vector;

/** Class that groups one bit full adders into four bit adders. */
public class SPCMap {
	/** list of all adders in graph. @see group   */
	public Vector adderList = new Vector();
	/** list of all adder groups @see group */
	public Vector groupList = new Vector();
	/** graph used for mapping */	
	public Graph g;
	
	/** constructs new Adders
	  * @param g graph to work on later */
	public SPCMap(Graph g) {
		this.g = g;
	}
	
	/** disjoins adders 
	  * @param g graph to perform ungroupping  */
	public void ungroup() {
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodeAdd) {
				NodeAdd a = (NodeAdd)n;
				a.prev = a.next = a.group = null;
			}
		}
	}
	
	/** joins adders into groups - some groups may need to
	  * be splitted if too long. Also adderList is built.
	  * @param g graph to perform groupping
	  * @return vector of all adders */
	public Vector group() {
		adderList.removeAllElements();
		groupList.removeAllElements();
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodeAdd) adderList.addElement(n);
			if((n instanceof NodeAdd)&&(((NodeAdd) n).group == null)) {// ungroupped
				/* we've  found unassigned node, now we go forward and backwards
				 * and try to join NodeAdd path into same group */				
				NodeAdd first = (NodeAdd)n; // start with this and go backwards
				first.next = null;
				boolean ok = true;
				do {
					ok = false;					
					if(first.ports[first.CARRY_IN_PORT].output instanceof NodeAdd) { // get pred. adder
						NodeAdd prev = (NodeAdd) first.ports[first.CARRY_IN_PORT].output;
						if((prev.group != null)								// not used, and used properly?
						 ||(first.ports[first.CARRY_IN_PORT]
						     == prev.ports[prev.CARRY_OUT_PORT] )) continue;
						first.prev = prev; // link together
						prev.next = first;
						first = prev;			 // go back
						ok = true;
						break;
					}
				}	while(ok);
				first.prev = null;	// group leader
				first.group = first;
				groupList.addElement(first);
				/* now we must mark visited nodes to group first */
				NodeAdd a = first;
				while(a.next != null) {
					a.group = first;
					a = a.next;
				}
				
				/* search for any adders behind */
				do {
					ok = false;
					Net nt = a.ports[a.CARRY_OUT_PORT]; // search through carry
					if(nt == null) break;
					for(int j = 0; j < nt.inputs.size(); j++) {
						Node m = (Node) nt.inputs.elementAt(j); // search for all adders connected to
						if(m instanceof NodeAdd) {							// this net
							NodeAdd next = (NodeAdd) m;
							if((next.group != null)								// not used and used properly?
							 ||(next.ports[next.CARRY_IN_PORT] != nt)) continue;
							next.group = first;		// link together and assign group
							a.next = next;
							next.prev = a;
							a = next;						// go forward
							ok = true;
							break;
						}
					}
				} while(ok);
				a.next = null;	// no successors
			}
		}
		return adderList;
	}
	
	/** Joins adders together into SPCs. <p>
	  * <em>WARNING: Call group first.</em>
	  * @return adderList - Vector of all NodeSPC.
	  * @see group */
	public Vector map() {
		adderList = new Vector();
		for(int i = 0; i < groupList.size(); i++) {
			NodeAdd na = (NodeAdd)groupList.elementAt(i);
			NodeSPC prev = null;
			NodeSPC group = null;
			NodeSPC n4 = null;
			while(na!=null) {
				n4 = new NodeSPC(na, g.global[NetGlobal.VCC]);
				n4.name = new String();
				for(int j = 0; (j < 4)&&(na!=null); j++) { // join up to four together					
					n4.name = n4.name+'+'+na.name;
					g.nodes.remove(na);
					if((na.next != null)&&(j < 3)) { // delete carry chains inbetween
						g.nets.remove(na.ports[na.CARRY_OUT_PORT]);
					}
					na.unlinkNets();
					na = na.next;
				}
				n4.name = n4.name.substring(1);
				n4.linkNets();
				g.nodes.add(n4);
				adderList.add(n4);
				if(prev == null) {	// update list
					n4.prev = null;	// first in list
					group = n4;
					n4.group = group;
				} else {					// not first in list
					n4.prev = prev;
					prev.next = n4;
					n4.group = group;
					/* connect to predecessors with carry chain */
					n4.ports[n4.CARRY_IN_PORT] = prev.ports[prev.CARRY_OUT_PORT];
					//n4.ports[n4.CARRY_IN_PORT].inputs.add(n4); // already set
				}
				prev = n4;
			}
			n4.next = null;
		}
		groupList = null; // dispose NodeAdds
		return adderList;
	}
}
