/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.mapping;

import java.util.Vector;
import org.opencores.structure.*;
import org.opencores.Conf;

/** Joins LUTs together so they fit into same GLCs.
  * NodeLUT.roomate is used.
  * Expects mapped graph, partially placed (RxR).
  * O(|V|^2*|K|) */
public class Roommates {
	/** number of LUTs in graph last processed */
	public static int numLUTs = 0;
	/** number of pairs (two LUTs connected) in graph last processed */
	public static int numPairs = 0;
	/** total sum of LUT inputs */
	public static int sumLUTInputs = 0;
	
	/** performs bucket sort on LUTs, based on number of inputs.
	  * @param g graph with LUTs
	  * @return array of Vectors, containing nodes, with same number of
	  * inputs */
	public static Vector[] bucketSort(Graph g) {
		// do bucket sort
		numLUTs = 0;
		sumLUTInputs = 0;
		Vector bucket[] = new Vector[Conf.K+1];
		for(int i = 0; i < Conf.K+1; i++)
			bucket[i] = new Vector(g.nodes.size());		
		for(int i = 0; i < g.nodes.size(); i++ ) {
			if(g.nodes.elementAt(i) instanceof NodeLUT) {
				NodeLUT n = (NodeLUT) g.nodes.elementAt(i);
				bucket[n.width-1].addElement(n);
				n.roommate = null; // no roommate at start
				numLUTs++;
				sumLUTInputs +=  n.width-1;
			}			
		}
		return bucket;
	}
			
	/** Performs roomate joining, based on greedy algorithm.
	  * @param g graph to work on */
	public static void joining(Graph g) {		
		numPairs = 0;
			
		Vector bucket[] = bucketSort(g);
		/* allocate descendingly */				
		for(int i = Conf.K; i >= 0; i-- ) {		  
		  while(!bucket[i].isEmpty()) {
				/* get LUT and find its best roommate */
				NodeLUT n = (NodeLUT) bucket[i].elementAt(bucket[i].size()-1);
				bucket[i].removeElementAt(bucket[i].size()-1);
				for(int k = i; k >= 0; k--) {
					int bestj = -1;
					float bestdist = Conf.X;
					for(int j = 0; j < bucket[k].size(); j++) {
						NodeLUT mate = (NodeLUT) bucket[k].elementAt(j);
						float d;
						if((NodeLUT.NumJoinInputsEx(n, mate) <= Conf.K)
						 &&((d = manhattan(n, mate)) < bestdist)) {
							bestj = j; bestdist = d; // remember
						}
					}
					if(bestj >=0) {
						numPairs++;
						NodeLUT mate = (NodeLUT) bucket[k].elementAt(bestj);						
						n.roommate = mate;
						mate.roommate = n;
						bucket[k].removeElementAt(bestj); // already assigned
						break;
					}
				}
			}
		}
	}
	
	/** calculates (manhattan) distance between two nodes,
	  * using .fx, .fy.
	  * @param a first node
	  * @param b second node
	  * @return distance >= 0 */
	public static float manhattan(Node a, Node b) {
		return Math.abs(a.fx - b.fx) + Math.abs(a.fy - b.fy);
	}
}
