/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.mapping;

import java.util.Vector;
import org.opencores.Conf;
import org.opencores.structure.*;

/** Collection of functions, that allows global (special) node mapping.
  * @see NodeGlobal */
public class GlobalMap {
	/** Name definition for Wishbone clock. */
	public final static String NAME_WB_CLK = "WB_CLK";
	/** Name definition for PIO clock. */
	public final static String NAME_PIO_CLK = "PIO_CLK";
	
	/** graph we are working on */
	private Graph g;
	
	/** Constructs new Global mapper */
	public GlobalMap(Graph g) {
		this.g = g;
	}

  /** Converts all LUT0s, that don't drive FF.DATA into NodeGlobal.
    * <em>This function should be called after optimal mapping and before
    * SPC and GPC groupping. <p>
    * SPC, GPC and FF mapping should link to global nets instead of
    * assigning null, to reduce complexity.</em>*/
	public void mapLUT0() {
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if((n instanceof NodeLUT)&&(n.width == 1)) {// LUT0?
				NetGlobal ng;
			  NodeLUT l = (NodeLUT)n;
			  /* only two possibilities for zero input function */
			  if(l.func[0] == 0) ng = g.global[NetGlobal.GND]; //GND
			  else ng = g.global[NetGlobal.VCC]; //VCC
			
				Vector inputs = l.ports[0].inputs;
				int ninputs = inputs.size();
				for(int j = 0; j < ninputs; j++) {
					Node next = (Node)inputs.elementAt(j);			  
					if(next instanceof NodeFF) {
						NodeFF ff = (NodeFF)next; // skip LUT0 before FF.DATA
						if(ff.ports[ff.DATA] == l.ports[0]) continue;						
					}
				
					/* now, that we have checked everything we can replace node */			  
					ng.inputs.add(next);
					for(int k = 0; k < next.width; k++) 
						if(next.ports[k] == n.ports[0]) { next.ports[k] = ng; break; }
					inputs.removeElementAt(j--);
					ninputs--;
				}
				if(ninputs == 0) {
					g.nodes.removeElementAt(i); i--; // delete node and net in graph
					g.nets.remove(n.ports[0]);
				}
			}
		}
	}
	
	/** Finds best net that could have use of <code>GLCK0</code>
	  * and it assigns to <code>NetGlobal GLCK0</code>.
	  * @param g graph to work on */
	public void mapGCLK0() {
		int max = Conf.GNET_MIN-1;	// should be at least GNET_MIN for global net
		Net ntMax = null;
	  /* search through all nets for that one with maximum FFs
	   * (with port CLK) attached */
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net) g.nets.elementAt(i);
			if(nt.inputs.size() < max) continue; // surely cannot be max => skip
			int cnt = 0;
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.CLK] == nt) cnt++;					
				}
			}
			if(cnt > max) { max = cnt; ntMax = nt;	}
		}
		
		if(ntMax != null) { // net found
			/* add global net generator node */
			NodeSR gen = new NodeSR(1);
			g.SR[1] = gen;
			gen.ports[0] = ntMax;
			ntMax.inputs.add(gen);
			
			NetGlobal GCLK0 = g.global[NetGlobal.GCLK0];
			gen.ports[1] = GCLK0;
			GCLK0.output = gen;			
			
			/* set global net */
			for(int j = 0; j < ntMax.inputs.size(); j++) {
				Node n = (Node) ntMax.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.CLK] == ntMax) {
						ff.ports[ff.CLK] = GCLK0;
						GCLK0.inputs.add(ff);
						ntMax.inputs.removeElementAt(j); j--;
					}
				}
			}
		}		
	}
	
	/** Finds best net that could have use of <code>GLCK1</code>
	  * and it assigns to <code>NetGlobal GLCK1</code>.
	  * @param g graph to work on */
	public void mapGCLK1() {
		int max = Conf.GNET_MIN-1;	// should be at least GNET_MIN for global net
		Net ntMax = null;
	  /* search through all nets for that one with maximum FFs
	   * (with port CLK) attached */
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net) g.nets.elementAt(i);
			if(nt.inputs.size() < max) continue; // surely cannot be max => skip
			int cnt = 0;
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.CLK] == nt) cnt++;					
				}
			}
			if(cnt > max) { max = cnt; ntMax = nt;	}
		}
		
		if(ntMax != null) { // net found
			/* add global net generator node */
			NodeSR gen = new NodeSR(3);
			g.SR[3] = gen;
			gen.ports[0] = ntMax;
			ntMax.inputs.add(gen);
			
			NetGlobal GCLK1 = g.global[NetGlobal.GCLK1];
			gen.ports[1] = GCLK1;
			GCLK1.output = gen;
			
			/* set global net */
			for(int j = 0; j < ntMax.inputs.size(); j++) {
				Node n = (Node) ntMax.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.CLK] == ntMax) {
						ff.ports[ff.CLK] = GCLK1;
						GCLK1.inputs.add(ff);
						ntMax.inputs.removeElementAt(j); j--;
					}
				}
			}
		}		
	}
	
	/** Finds best net that could have use of <code>GRST</code>
	  * and it assigns to <code>NetGlobal GRST</code>.
	  * @param g graph to work on */
	public void mapGRST() {
		int max = Conf.GNET_MIN-1;	// should be at least GNET_MIN for global net
		Net ntMax = null;
	  /* search through all nets for that one with maximum FFs
	   * (with port RST) attached */
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net) g.nets.elementAt(i);
			if(nt.inputs.size() < max) continue; // surely cannot be max => skip
			int cnt = 0;
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.RST] == nt) cnt++;					
				}
			}
			if(cnt > max) { max = cnt; ntMax = nt;	}
		}
		
		if(ntMax != null) { // net found
			/* add global net generator node */			
			NodeSR gen = new NodeSR(5);
			g.SR[5] = gen;
			gen.ports[0] = ntMax;
			ntMax.inputs.add(gen);
			
			NetGlobal GRST = g.global[NetGlobal.GRST];
			gen.ports[1] = GRST;
			GRST.output = gen;
			
			/* set global net */
			for(int j = 0; j < ntMax.inputs.size(); j++) {
				Node n = (Node) ntMax.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.RST] == ntMax) {
						ff.ports[ff.RST] = GRST;
						GRST.inputs.add(ff);
						ntMax.inputs.removeElementAt(j); j--;
					}
				}
			}
		}		
	}
	
	/** Finds best net that could have use of <code>GSET</code>
	  * and it assigns to <code>NetGlobal GSET</code>.
	  * @param g graph to work on */
	public void mapGSET() {
		int max = Conf.GNET_MIN-1;	// should be at least GNET_MIN for global net
		Net ntMax = null;
	  /* search through all nets for that one with maximum FFs
	   * (with port SET) attached */
		for(int i = 0; i < g.nets.size(); i++) {
			Net nt = (Net) g.nets.elementAt(i);
			if(nt.inputs.size() < max) continue; // surely cannot be max => skip
			int cnt = 0;
			for(int j = 0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.SET] == nt) cnt++;					
				}
			}
			if(cnt > max) { max = cnt; ntMax = nt;	}
		}
		
		if(ntMax != null) { // net found
			/* add global net generator node */
			NodeSR gen = new NodeSR(4);
			g.SR[4] = gen;
			gen.ports[0] = ntMax;
			ntMax.inputs.add(gen);
			
			NetGlobal GSET = g.global[NetGlobal.GSET];
			gen.ports[1] = GSET;
			GSET.output = gen;
			
			/* set global net */
			for(int j = 0; j < ntMax.inputs.size(); j++) {
				Node n = (Node) ntMax.inputs.elementAt(j);
				if(n instanceof NodeFF) {
					NodeFF ff = (NodeFF)n;
					if(ff.ports[ff.SET] == ntMax) {
						ff.ports[ff.SET] = GSET;
						GSET.inputs.add(ff);
						ntMax.inputs.removeElementAt(j); j--;
					}
				}
			}
		}		
	}
	
	/** Searches for port named PIO_CLK and assigns it to SR.
	  * <em>NOTE: should be called before IOCMap</em> */
	public void mapPIO_CLK() {
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node)g.nodes.elementAt(i);
			if((n instanceof NodePort)&&(n.name.compareTo(NAME_PIO_CLK)==0)&&(n.dir[0] == n.OUTPUT)) {
				g.nodes.removeElementAt(i);	// remove port from netlist and replace it with SR0
				NodeSR sr = g.SR[0] = new NodeSR(0);
				n.ports[0].output = sr;
				sr.ports[1] = n.ports[0];
				return;
			}
		}
	}
	
	/** Searches for port named WB_CLK and assigns it to SR.
	  * <em>NOTE: should be called before IOCMap</em> */
	public void mapWB_CLK() {
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node)g.nodes.elementAt(i);
			if((n instanceof NodePort)&&(n.name.compareTo(NAME_WB_CLK)==0)&&(n.dir[0] == n.OUTPUT)) {
				g.nodes.removeElementAt(i);	// remove port from netlist and replace it with SR2
				NodeSR sr = g.SR[2] = new NodeSR(2);
				n.ports[0].output = sr;
				sr.ports[1] = n.ports[0];
				return;
			}
		}
	}
	
	/** Does complete global net mapping on graph <code>g</code> */
	public void mapping() {
		/* unmark all global nets */
		for(int i = 0; i < Conf.NUM_SR; i++) g.SR[i] = null;
		mapLUT0();
		mapGCLK0();
		mapGCLK1();
		mapGRST();
		mapGSET();
		mapPIO_CLK();
		mapWB_CLK();
	}
}
