/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.mapping;

import java.util.Stack;
import java.util.Vector;
import java.util.Arrays;
import java.util.Comparator;
import org.opencores.structure.*;
import org.opencores.Conf;
import org.opencores.util.Heap;

/** Performs unit-delay optimal mapping for LUTs - FlowMap algorithm.  
  * @see "Cong, J. and Y. Ding, "An Optimal Technology Mapping Algorithm for Delay optimization in Lookup-table Based FPGA Designs" in UCLA Computer Science Department Technical Report CSD-920022, (May 1992)"<p>
  * @see "Harvey J. Greenberg: Ford-Fulkerson Max Flow Labeling Algorithm, December 22, 1998" */
public final class FlowMap {
	/** Whether FlowMap should optimize space */
	public static boolean OPTIMIZE_SIZE = false;

	/** current temp flag */
	private int tempCnt = 0;
	
	/** first step of algorithm - labeling
	  * @param g graph to label. Labels are stored in node.y */
	public void labeling(Graph g) {
		Stack zeroInputs = new Stack();
		
		int topoOrder = 0; // current node topological order
		/* node.x denotes number of inputs left */
		/* node.y denotes label (= minimum height in optimal LUT mapping) */
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			n.temp = tempCnt;
			n.x = 0;
			n.y = -1;
			if(n instanceof NodeLUT) // treat all non-LUTs as PI and PO
				for(int j = 0; j < n.dir.length; j++) // count inputs
					if(n.dir[j] == n.INPUT) n.x++;			
			if(n.x == 0) zeroInputs.push(n); // add to zero inputs
		}
		
		int min = 1; // minimal number of inputs left (node.x)
		int mini = 0;
		while(mini >= 0) {
			while(!zeroInputs.isEmpty()) {	// do topo-sort
				Node n = (Node) zeroInputs.pop();
				n.y = label(n); 							// label current node
				n.nfy = topoOrder++;					// store topological order
				for(int i = 0;i < n.ports.length; i++) {
					if(n.dir[i] == n.OUTPUT) {
						Net ns = n.ports[i];
						if(ns != null) {
							for(int j = 0; j < ns.inputs.size(); j++) {
								Node next = (Node) ns.inputs.elementAt(j);
								if(next.x <= 0) continue; // can happen in NodeCycleBreak
								if(--next.x == 0) zeroInputs.push(next);
							}
						}
					}
				}
			}
			
			/* now that topo sort is finished we have to figure out
			   if we have a cycle or we ran out of nodes */
			min = g.nodes.size();
			mini = -1;
			for(int i = 0; i < g.nodes.size(); i++) {
				Node n = (Node) g.nodes.elementAt(i);
				if((n.x != 0)&&(n.x < min)) {
					min = n.x; mini = i;
					if(min == 1) break; // optimal candidate found
				}
			}
			if(mini > 0) { // cycle found, delete cycle links
				// insert NodeCycleBreak
				Node critical = (Node)g.nodes.elementAt(mini);				
				for(int i = 0; i < critical.width; i++)	{
					// insert cycle break node before node port if there is a problem
					if((critical.ports[i] != null)&&(critical.dir[i] == critical.INPUT)
					 &&(critical.ports[i].output.y<0)) {
						Net cbNet = new Net();						
						NodeCycleBreak ncb = new NodeCycleBreak();
						Net cNet = critical.ports[i];
						cbNet.name = "NCBN"+topoOrder;
						ncb.name = "NCB"+topoOrder;
						cNet.inputs.remove(critical);	// cNet->ncb
						cNet.inputs.add(ncb);
						ncb.ports[ncb.IN] = cNet;
						
						cbNet.output = ncb;						// ncb->cbNet
						ncb.ports[ncb.OUT] = cbNet;
						
						critical.ports[i] = cbNet;		// cbNet->critical
						cbNet.inputs.add(critical);						
						
						critical.x--;	// delete references
					
						g.nodes.add(ncb);
						g.nets.add(cbNet);
						ncb.nfy = topoOrder++;
						ncb.x = 0;
					}										
				}
				if(critical.x != 0) throw new Error();
				zeroInputs.push(critical);
			}
		}
	}
	
	/** Calculates label of node t.
		* @param n node to calculate label
	  * @return label of t */
	private int label(Node t) {		
		if(!(t instanceof NodeLUT)) return 0; // treat non-LUT nodes as PI, PO
		if(t.y >= 0) throw new Error("Node already labeled.");
		Vector PI = new Vector();

		int mil = 0; // maximum input node label
		for(int i = 0; i < t.width-1; i++) {
			mil = Math.max(mil, t.ports[i].output.y);
			if(t.ports[i].output.y < 0) throw new Error("Invalid topo order.");
		}
				
		/* first we have to mark cone(n) - all predecessors. If cycle is detected
		   we stop searching in that direction and mark it as PI add to sn.inputs */
		tempCnt++;
		mark(t, PI, mil);
		
		/* now that we know which nodes are accessible, we perform max-flow
		   on them from node s (all PIs) */		   
		if(PI.size() < Conf.K) return 1; // trivial case: #PI<K
		
		if(flow(PI, t, mil)) { //is there maximum flow larger than Conf.K? => pack t in new LUT
			return mil+1;
		} else { // there is enough room left 
			return Math.max(mil, 1);
		}
	}
	
	/** Mark predecessor nodes ot t with tempCnt.
		* @param t node to mark predecessors
		* @param PI vector to put primary input nets in */
	private void mark(Node t, Vector PI, int mil) {
		if(t.temp != tempCnt) {
			t.temp = tempCnt;
			t.flag = false;	// unvisited (prepare for flow function)
			t.visited = false;
			t.nfx = 0f;
			for(int i = 0; i < t.width-1; i++) // search all inputs recursively
				if(t.ports[i].output instanceof NodeLUT) {
					if((!OPTIMIZE_SIZE)||(t.ports[i].output.y == mil)
					||(t.ports[i].output.link == null)||(t.ports[i].inputs.size() == 1))
						mark(t.ports[i].output, PI, mil);
					else if(!PI.contains(t.ports[i])) PI.add(t.ports[i]);
				} else if(!PI.contains(t.ports[i])) PI.add(t.ports[i]);			
		}
	}
	
	/** Calculates flow from source s to sink t or to node with label 'label'.
	  * Also mark function should be called to set .temp and .flag = false for all predecessors	  
	  * @see mark
	  * @param s source node
	  * @param t sink node
	  * @param label minimum label to stop
	  * @return true if there is flow > Conf.K between source and sink */
	private boolean flow(Vector PI, Node t, int label) {		
		t.y = label;
		/** current s-t flow */
		int flowCnt = 0;
		
		Vector GPI = new Vector(PI.size());
		for(int i = 0; i < PI.size(); i++) {// search all inputs recursively
			Net nt = (Net)PI.elementAt(i);
			NodePrim np = new NodePrim(NodePrim.BUF);	// create new temporary PI node
			np.ports[np.OUT] = nt;
			np.link = nt.output;
			np.flag = false;
			np.temp = tempCnt;
			np.name = "FI-"+nt.name;
			GPI.add(np);			
		}
		
		boolean bad = false;
		for(int i = 0; (i < PI.size())&&(!bad); i++) {
			Net nt = (Net)PI.elementAt(i);
			NodePrim np = (NodePrim) GPI.elementAt(i);
			nt.output = np;						
			boolean direct = false;
			for(int j = 0; j < np.ports[1].inputs.size(); j++) {
				if(((Node)np.ports[1].inputs.elementAt(j)).y == label)
					{direct = true; break; }				
			}
			if(direct) {
				np.flag = true;
				if(++flowCnt > Conf.K) { bad = true; break; } // direct connection			
			} else {
				while(flowR(np, true, label))									// breakthrough
					if(++flowCnt > Conf.K) { bad = true; break; }
			}
		}
			
		for(int i = 0; i < GPI.size(); i++) {							// restore PIs
			NodeLUT n = (NodeLUT)GPI.elementAt(i);
			n.ports[n.OUT].output = n.link;
		}
		
		return bad;
	}
	
	/** Tries to find flow from source s to sink t. <p>
	  * Each node is splitted into two parts - top and bottom. Top can access
	  * bottoms of predecessors, bottom can access successor tops. Top-bottom
	  * link is valid if node <code>.flag == <b>false</b></code>. <p>
	  * Mark function should be called to set .temp and .flag = false for all predecessors	  
	  * @see mark
	  * @param sink source node	  
	  * @param top designates top access to sink node
	  * @param label minimum label to stop
	  * @return true if there is a flow between source and sink
	  * @see markAccessible
	  * @see L.R. Ford, Jr. and D.R. Fulkerson: Maximal Flow Trough a Network */
	private boolean flowR(Node sink, boolean top, int label) {
		if(Conf.debug) Conf.log.print("("+sink.name+" "+top+label+sink.flag+")");
		/* if a LUT node is part of cone(t) and not yet visited */		
		if((sink.y >= label)) {
			if(Conf.debug) Conf.log.println();
			return true;	// are we there yet?
		}	
		sink.visited = true; // mark visited				
		
		if((!top)||(!sink.flag)) {	// on bottom or cannot pass trough
			// update all successors
			Vector inputs = sink.ports[sink.width-1].inputs;
			for(int i = 0; i < inputs.size(); i++) {
				Node next = (Node)inputs.elementAt(i);
				if((next instanceof NodeLUT)&&(!next.visited)&&(next.temp == tempCnt)){
					if(Conf.debug) Conf.log.print(sink.name+"+"+next.name+" ");
					if(flowR((NodeLUT) next, true, label)) {
						if(Conf.debug) Conf.log.println(sink.name+"+"+next.name+" ");
						if(top) sink.flag = true; // set flow
						sink.visited = false;		
						return true;
					}
				}
			}
		}
		
		if((top)||(sink.flag)) {	// on top, or can pass trough
			// also update all predecessors
			for(int i = 0; i < sink.width-1; i++) if(sink.ports[i] != null) {
				Node pred = sink.ports[i].output;
				if((pred instanceof NodeLUT)&&(!pred.visited)&&(pred.temp == tempCnt)) {
				  if(Conf.debug) Conf.log.print(sink.name+"-"+pred.name+" ");
					if(flowR((NodeLUT)pred, false, label)){
						if(Conf.debug) Conf.log.println(sink.name+"-"+pred.name+" ");
						if(!top) sink.flag = false;	// set flow
						sink.visited = false;
						return true;
					}
				}
			}		
		}
		sink.visited = false;
		return false;
	}
			
	/** Second step of the algorithm - mapping. Call labeling first.
	  * @see labeling
	  * @param g labeled graph
	  * @return mapped graph (completely rebuilt) */
	public Graph mapping(Graph g) {
	  /* NOTE: new graph variables have _ before name */
	  
	  /* graph is built in following steps:
	     1. all nodes in _g are built and output nets are allocated, so
	        every node should have accesible its input nets and nodes.
					Since all outputs are allocated, ALL nets are also built.
	     2. part of LUT tree is built for every _NodeLUT
	     3. connect inputs of _nodes
	   */
	  tempCnt = 0;	// timestamp
		Graph _g = new Graph();	// newly built graph
		_g.name = g.name;
		Stack todo = new Stack(); // list of unbuilded LUTs
		
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);			
			n.temp = tempCnt;
			if(!(n instanceof NodeLUT)) {	// non LUT node
				Node _n = (Node) n.clone();				
				_n.temp = tempCnt;
				_n.link = n; n.link = _n; // link nodes
				_g.nodes.addElement(_n);				
				
				/* create all output nets for non-LUTs */
				for(int j = 0; j < n.ports.length; j++) {					
					if((n.ports[j]!=null)&&(n.dir[j] == n.OUTPUT)) {
						Net _nt = new Net();
						_g.nets.addElement(_nt);
						_nt.name = n.ports[j].name; // copy net name
						_n.ports[j] = _nt;
						_nt.output = _n;
						_nt.link = n.ports[j];			// link nets
						n.ports[j].link = _nt;
					}
				}
			} else { // LUT node
				n.link = null;									// mark unassigned
				if(n.ports[n.width-1] == null) continue;	// no output - will not be created
				Vector inputs = n.ports[n.width-1].inputs;
				for(int j = 0; j < inputs.size(); j++)
					if(!(inputs.elementAt(j) instanceof NodeLUT)) {
						/*if(!todo.contains(n))*/ todo.push(n);
						break;
					}				
			}
		}
		
		if(Conf.debug) {
			for(int i = 0; i < todo.size(); i++)
				Conf.log.print(((Node)todo.elementAt(i)).name+" ");
			Conf.log.println();
		}
		
		/** process LUTs */
		while(!todo.isEmpty()) {
			NodeLUT n = (NodeLUT)todo.pop();
			
			tempCnt++;						
			NodeLUT _n = joinLUTs((NodeLUT)n); // join LUTs together ("collapses" tree)
			for(int i = 0; i < _n.width-1; i++) {
				Node cand = _n.ports[i].output;				
				if((cand.link == null)&&(cand instanceof NodeLUT)
				 &&(!todo.contains(cand))) todo.push(cand);
			}
			if(_n.width-1 > Conf.K) throw new Error("Invalid number of inputs!");
			_n.link = n; n.link = _n;
			_g.nodes.add(_n);
			_n.name = n.name;
			_n.temp = tempCnt;
			Net _nt = new Net();
			_nt.name = n.ports[n.width-1].name; // copy net name
			_g.nets.add(_nt);
			_n.ports[_n.width-1] = _nt;			
			_nt.output = _n;
			_nt.link = n.ports[n.width-1];			 // link nets
			n.ports[n.width-1].link = _nt;
		}

		for(int i = 0; i < _g.nodes.size(); i++) {
			Node _n = (Node) _g.nodes.elementAt(i);	 // get node			
			// built _n's inputs
			for(int j = 0; j < _n.ports.length; j++) {
				if(_n.dir[j] == _n.INPUT) {
					if(_n.ports[j] != null) {
						_n.ports[j] = _n.ports[j].link; // use net's link to reconnect 					
						if(_n.ports[j] != null) _n.ports[j].inputs.add(_n);			// bidirectional - update net's inputs
					}
				}
			}
			if(Conf.debug) System.out.print(_n.name + " ");
		}
		
		/** unlink NodeCycleBreak nodes */
		for(int i = 0; i < _g.nodes.size(); i++) {
			Node _n = (Node) _g.nodes.elementAt(i);	 // get node			
			if(_n instanceof NodeCycleBreak) {
				NodeCycleBreak n = (NodeCycleBreak)_n;
				Net nt = n.ports[n.OUT];
				n.ports[n.IN].inputs.remove(n);	// unlink	NCB
				for(int j = 0; j < nt.inputs.size(); j++) {
					Node next = (Node)nt.inputs.elementAt(j);
					for(int k = 0; k < next.width; k++)
						if(next.ports[k] == nt) {
							/* link succ node with pred node */
							next.ports[k] = n.ports[n.IN];
							n.ports[n.IN].inputs.add(next);
						}
				}				
			}
		}
		/** remove NodeCycleBreak nodes and their nets */
		for(int i = 0; i < _g.nodes.size(); i++) {
			Node _n = (Node) _g.nodes.elementAt(i);	 // get node			
			if(_n instanceof NodeCycleBreak) {
				_g.nodes.removeElementAt(i);
				_g.nets.remove(_n.ports[((NodeCycleBreak)_n).OUT]);
				i--;
			}
		}
		return _g;
	}
	
	/** Collapses tree of same labeled nodes into s, using shortest-first joining.
	  * @param child sink node
	  * @return collapsed tree node */
	private NodeLUT joinLUTs(NodeLUT child) {		
		Vector PI = new Vector();
		int mil = 0; // maximum input node label
		for(int i = 0; i < child.width-1; i++) {
			mil = Math.max(mil, child.ports[i].output.y);
			if(child.ports[i].output.y < 0) throw new Error("Invalid topo order.");
		}
		mark(child, PI, mil);	// mark predecessor cone
		if(Conf.debug) {
			Conf.log.println("----------------------------");
			Conf.log.println(child.name);
			for(int i = 0; i < PI.size(); i++) {
				Net pin = (Net) PI.elementAt(i);			
				Conf.log.print(pin.name+" ");
			}	
			Conf.log.println(PI.size());
		}		
		
		/* Find flow again. */
		int nFlows = 0;
		Vector GPI = new Vector(PI.size());
		for(int i = 0; i < PI.size(); i++) {// search all inputs recursively
			Net nt = (Net)PI.elementAt(i);
			NodePrim np = new NodePrim(NodePrim.BUF);
			np.ports[np.OUT] = nt;
			np.link = nt.output;			
			np.flag = false;
			np.temp = tempCnt;			
			np.name = "FI-"+nt.name;
			np.nfx = 0;
			GPI.add(np);
		}
		
		for(int i = 0; i < PI.size(); i++) {
			Net nt = (Net)PI.elementAt(i);
			NodePrim np = (NodePrim) GPI.elementAt(i);
			nt.output = np;
			boolean direct = false;
			for(int j = 0; j < np.ports[1].inputs.size(); j++) {
				if(((Node)np.ports[1].inputs.elementAt(j)).y == child.y)
					{direct = true; break;}				
			}
			if(direct) {np.flag = true; nFlows++; }// direct connection			
			else while(flowR(np, true, child.y)) nFlows++; // breakthrough
		}
				
		if(Conf.debug) Conf.log.println("nFlows: "+nFlows);
		//if(nFlows > Conf.K) throw new Error("Number of flows larger than number of inputs !");
			
		/** At this point we have nodes in X marked with tempCnt, nodes
		  * in _X marked with tempCnt-1. We will add all nodes marked with
		  * tempCnt-1 in same LUT */		 
		NodeLUT newLUT = buildLUT(child, GPI);
		for(int i = 0; i < GPI.size(); i++) {
			NodeLUT n = (NodeLUT)GPI.elementAt(i);
			n.ports[n.OUT].output = n.link;
		}
		return newLUT;
	}
	
	/** Similar to routine A of Ford Fulerson max-flow algorithm,
	  * marks visited nodes with .visited flag. <p>
	  * Each node is splitted into two parts - top and bottom. Top can access
	  * bottoms of predecessors, bottom can access successor tops. Top-bottom
	  * link is valid if node <code>.flag == <b>false</b></code>.
	  * @param sink sink node
	  * @param top designates top access to sink node
	  * @see flowR
	  * @see "L.R. Ford, Jr. and D.R. Fulkerson: Maximal Flow Trough a Network" */
	private void markAccessible(NodeLUT sink, boolean top) {
		if(Conf.debug) Conf.log.print("("+sink.name+" "+top+(int)sink.nfx+sink.flag+")");		
		sink.visited = true; // mark visited		
		sink.nfx = 1;				 // mark in X set
		
		if((!top)||(!sink.flag)) {	// on bottom or cannot pass trough
			// update all successors
			Vector inputs = sink.ports[sink.width-1].inputs;
			for(int i = 0; i < inputs.size(); i++) {
				Node next = (Node)inputs.elementAt(i);
				if((next instanceof NodeLUT)&&(!next.visited)&&(next.temp == tempCnt)){
					if(Conf.debug) Conf.log.print(sink.name+"+"+next.name+" ");
					markAccessible((NodeLUT) next, true);
				}
			}
		}
		
		if((top)||(sink.flag)) {	// on top, or can pass trough
			// also update all predecessors
			for(int i = 0; i < sink.width-1; i++) if(sink.ports[i] != null) {
				Node pred = sink.ports[i].output;
				if((pred instanceof NodeLUT)&&(!pred.visited)&&(pred.temp == tempCnt)) {
				  if(Conf.debug) Conf.log.print(sink.name+"-"+pred.name+" ");
					markAccessible((NodeLUT)pred, false);
				}
			}		
		}
		sink.visited = false;		
	}	
		
	/** adds all childs predecessors with same label into LUTs Vector */
	private void buildSame(NodeLUT child, int label, Vector LUTs) {
		// same label or bottom unaccessible from sink		
		//if(child.y == 0) return; // ignore primary inputs
		if((child.y == label)||(child.nfx == 0)) {
			if(!LUTs.contains(child)) LUTs.add(child);	// add to list
		} else return;
		for(int i = 0; i < child.width-1; i++) if(child.ports[i] != null){
			Node parent = child.ports[i].output;
			if(Conf.debug) Conf.log.print(child.name+":"+parent.name+" ");
			if((parent instanceof NodeLUT)&&(parent.temp == tempCnt))
				buildSame((NodeLUT)parent, label, LUTs);
		}
	}
	
	/** Collapses DAG of same nodes with <code>temp==tempCnt-1</code> into <i>child</i>,
	  * using shortest-first joining. <code>.temp</code> is incremented, to disable looping.
	  * @param child sink node
	  * @return collapsed tree node */
	private NodeLUT buildLUT(NodeLUT child, Vector GPI) {
		Vector PI = new Vector();
		Vector LUTs = new Vector();
						
		for(int i = 0; i < GPI.size(); i++) {// search all inputs recursively
			NodeLUT np = (NodeLUT) GPI.elementAt(i);
			/*boolean direct = false;
			for(int j = 0; j < np.ports[1].inputs.size(); j++) {
				if(((Node)np.ports[1].inputs.elementAt(j)).y == child.y)
					{direct = true; break;}				
			}
			if(direct) np.nfx = 1;		// direct connection
			else */			
			markAccessible(np, true);	// start on top of forced primary inputs
		}
		if(Conf.debug) Conf.log.println();

		buildSame(child, child.y, LUTs);
		if(Conf.debug) Conf.log.println();

		for(int i = 0; i < LUTs.size(); i++) {
			NodeLUT nl = (NodeLUT)LUTs.elementAt(i);
			for(int j = 0; j < nl.width-1; j++) {
				Net nt = nl.ports[j];				
				Node next = nt.output;
				if(next instanceof NodeLUT) {
					if((!LUTs.contains(next)&&(!PI.contains(nt)))) PI.add(nt);					
				} else if(!PI.contains(nt)) PI.add(nt);
			}
		}
		
		if(Conf.debug) {
			Conf.log.println();
			for(int i = 0; i < LUTs.size(); i++)
				Conf.log.print(((Node)LUTs.elementAt(i)).name +"("+((Node)LUTs.elementAt(i)).y+") ");
			Conf.log.println();
			Conf.log.flush();
		}
				
		NodeLUT l[] = new NodeLUT[LUTs.size()];
		LUTs.toArray(l);
		Arrays.sort(l, new Comparator() { // sort ascending on topo sort
			public int compare(Object o1, Object o2) {
				NodeLUT a = (NodeLUT)o1, b = (NodeLUT)o2;
				if(a.nfy < b.nfy) return -1;
				else if(a.nfy > b.nfy) return 1; else return 0;
			}		
			public boolean equals(Object o) {throw new Error();}		
		});
		Variable var[] = new Variable[PI.size()];
		for(int i = 0; i < PI.size(); i++) {
			Net pin = (Net) PI.elementAt(i);
			var[i] = new Variable();
			var[i].actualNode = pin.output;
			var[i].actualNet = pin;			
		}
	
		for(int i = 0; i < PI.size(); i++) {
			Net pin = (Net) PI.elementAt(i);
			pin.output = var[i];
			if(Conf.debug) Conf.log.print(pin.name+" ");
		}
		if(Conf.debug) {		
			Conf.log.println(PI.size());
			Conf.log.flush();
		}

		if(PI.size() > Conf.K)
			throw new Error("Invalid number of inputs!");		
		
		/* calculate gravity point */
		NodeLUT nl = new NodeLUT(var.length+1); 
		float fx = 0, fy = 0;
		for(int i = 0; i < LUTs.size(); i++) {
			fx += ((NodeLUT)LUTs.elementAt(i)).fx;
			fy += ((NodeLUT)LUTs.elementAt(i)).fy;
		}
		nl.fx = fx / LUTs.size();
		nl.fy = fy / LUTs.size();
		
		for(int i = 0; i < 1<<var.length; i++) {	// generate function
			// setup inputs
			int input = i;
			for(int v = var.length-1; v >= 0; v--) {
				var[v].x = input&1;
				input >>= 1;
			}
			for(int j = 0; j < l.length; j++) {
				int li = 0;
				for(int k = 0; k < l[j].width-1; k++) {
					Node n = l[j].ports[k].output;
					li <<= 1;
				  li |= n.x;
				}
				l[j].x = l[j].calc(li);
			}
			nl.func[i>>3] |= (byte) (child.x << (i&7));
		}
		
		for(int v = 0; v < var.length; v++) { // restore primary inputs
			var[v].actualNet.output = var[v].actualNode;
			nl.ports[v] = var[v].actualNet;
		}
		return nl;
	}
	
	/** Internal class used for building larger LUTs,
	  * representing directed acyclic subgraph input. <p>
	  * Value is stored in .x */
	class Variable extends Node {
	  /** node for which this variable stands for */
		Node actualNode;
		/** net for which this variable stands for */
		Net actualNet;		
		
		/** construct new variable */
		Variable() {
			super(0);
		}
	}
}
