/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.graphics;

import java.awt.*;
import java.awt.event.*;
import org.opencores.structure.*;
import org.opencores.Conf;

/** Component used for graph drawing. */
public class Draw extends Canvas {
  /** grid size */
	private static final int GRID = 16;
	/** linear drawing mode - no grid - mapping */
	public static final int DM_LINEAR = 1;
	/** grid drawing mode - placement */
	public static final int DM_GRID = 2;
	/** segment drawing mode - routing */
	public static final int DM_SEGMENT = 3;
	/** graph to work on */
	private Graph g;
	/** current mode */
	private int mode = DM_LINEAR;
	/** Object mouse is holding */
	private Object holding = null;
	
	/** Sets drawing mode.
	  * @see Draw for parameters.*/
	public void setMode(int mode) {
		this.mode = mode;
	}

  /** Constructs new frame, showing graph g
    * @param g graph to display
    * @see org.opencores.structure.Graph */
	public Draw(Graph g) {
		this.g = g;
		addMouseListener(new MouseListener() {
			public void mouseEntered(MouseEvent e) {}			
			public void mouseExited(MouseEvent e) {}
			public void mouseClicked(MouseEvent e) {}
			public void mousePressed(MouseEvent e) {				
				holding = pick(e.getX(), e.getY());
			}
			public void mouseReleased(MouseEvent e) {
				moveTo(holding, e.getX(), e.getY());
				repaint();
			}
		});
	}
	
	/** Changes graph displayed
	  * @param g graph to display
    * @see org.opencores.structure.Graph */
	public void setGraph(Graph g) {
		this.g = g;
		repaint();
	}

  /** Draws	graph to specified graphics.
    * @param gr graphics object */
	public void paint(Graphics gr) {
		gr.setColor(Color.darkGray);
		gr.fillRect(0,0,GRID*(Conf.X+2),GRID*(Conf.Y+2));
		//gr.drawLine(0,0,GRID*Conf.X,GRID*Conf.Y);
		
		if(mode == DM_GRID) {
			/** draw grid */
			for(int i=0; i < Conf.X; i++) {
				if(i % 8 == 0) gr.setColor(Color.lightGray);
				else gr.setColor(Color.gray);
				int x = GRID*(1+i)-1;
				gr.drawLine(x, GRID, x, GRID*(Conf.Y+1)-1);
			}
			gr.setColor(Color.gray);	
			gr.drawLine(GRID*(1+Conf.X)-1, GRID,
				GRID*(1+Conf.X)-1, GRID*(Conf.Y+1)-1);
		
			for(int i=0; i < Conf.Y; i++) {			
				int y = GRID*(1+i)-1;
				if(i % 8 == 0) gr.setColor(Color.lightGray);
				else gr.setColor(Color.gray);
				gr.drawLine(GRID-1, y, GRID*(Conf.X+1)-1, y);
			}
			gr.drawLine(GRID, GRID*(1+Conf.Y)-1,
				GRID*(Conf.X+1)-1, GRID*(1+Conf.Y));
		}
		
		if((mode == DM_GRID)||(mode == DM_LINEAR)) {
			/** draw nets - direct lines */
			for(int i=0; i < g.nets.size(); i++) {
				Net nt = (Net) g.nets.elementAt(i);			
				if(nt.output == null) {
					System.out.println("Errorneous net: "+nt.name);
					continue;
				}
				float x=nt.output.fx, y=nt.output.fy;

				for(int j=0; j < nt.inputs.size(); j++) {
					Node n = (Node) nt.inputs.elementAt(j);
					x += n.fx;
					y += n.fy;
				}
				x /= nt.inputs.size()+1;
				y /= nt.inputs.size()+1;
				
				gr.setColor(Color.lightGray);
				for(int j=0; j < nt.inputs.size(); j++) {
					Node n = (Node) nt.inputs.elementAt(j);
					gr.drawLine((int)(n.fx*GRID)+GRID/2+GRID,(int)(n.fy*GRID)+GRID/2+GRID,(int)(x*GRID)+GRID/2+GRID,(int)(y*GRID)+GRID/2+GRID);
				}
				Node n = nt.output;
				gr.setColor(Color.yellow);
				gr.drawLine((int)(n.fx*GRID)+GRID/2+GRID,(int)(n.fy*GRID)+GRID/2+GRID,(int)(x*GRID)+GRID/2+GRID,(int)(y*GRID)+GRID/2+GRID);
				gr.fillRect((int)(x*GRID-1)+GRID/2+GRID,(int)(y*GRID-1)+GRID/2+GRID,3,3);
			}
		}
		
		/* draw nodes */		
		for(int i = 0; i < g.nodes.size(); i++) {
			Node n = (Node) g.nodes.elementAt(i);
			if(n instanceof NodePrim) gr.setColor(Color.magenta);
			else if(n instanceof NodeGPC) gr.setColor(Color.orange);
			else if(n instanceof NodeLUT) gr.setColor(Color.red);
			else if(n instanceof NodePort) gr.setColor(Color.cyan);
			else if(n instanceof NodeIOC) gr.setColor(Color.blue);
			else if(n instanceof NodeFF) gr.setColor(Color.yellow);
			else gr.setColor(Color.gray);
			
			gr.fill3DRect((int)(n.fx*GRID-1)+GRID+1,(int)(n.fy*GRID-1)+GRID+1,GRID-1,GRID-1,true);
		}
		
		/** draw segments */
		if((mode == DM_SEGMENT)&&(g.pos != null)) {						
			for(int x = 0; x < g.pos.length; x++) {
				for(int y = 0; y < g.pos[x].length; y++) {
					if(((x==0)||(x==Conf.X+1))&&((y==0)||(y==Conf.Y+1))) continue;
					NodeRoutable r = g.pos[x][y];
										
					/* draw segments - segment is divided  */
					for(int s = 0; s < r.NINPUTS_ROUTABLE; s++) if(r.segments[s] != null) {
						int dx = r.neighCoor[s][0];
						int dy = r.neighCoor[s][1];
						int ax = (Math.abs(dx)>Math.abs(dy))?(0):(1);
						int ay = 1-ax;
						if((s&1) == 1) { ax = -ax; ay = -ay; } 
						gr.setColor(Color.white);
						gr.drawLine((int)(r.x*GRID)+GRID/2+GRID+dx+ax,(int)(r.y*GRID)+GRID/2+GRID+dy+ay,
											  (int)((r.x+dx)*GRID)+GRID/2+GRID-dx+ax,(int)((r.y+dy)*GRID)+GRID/2+GRID-dy+ay);
						gr.setColor(Color.red);
						gr.drawLine((int)(r.x*GRID)+GRID/2+GRID+dx+ax,(int)(r.y*GRID)+GRID/2+GRID+dy+ay,
												(int)(r.x*GRID)+GRID/2+GRID+dx+ax,(int)(r.y*GRID)+GRID/2+GRID+dy+ay);
					}
				}
			}
			/* draw number */
			for(int x = 0; x < g.pos.length; x++) {
				for(int y = 0; y < g.pos[x].length; y++) {
					if(((x==0)||(x==Conf.X+1))&&((y==0)||(y==Conf.Y+1))) continue;
					NodeRoutable r = g.pos[x][y];

					int nSeg = r.nSegments;
					int i = nSeg*256/r.NINPUTS_ROUTABLE;
					if(i > 255) i = 255;
					gr.setColor(new Color(255 - i, i, 10));
					gr.drawString(nSeg+"", x*GRID+1, (y+1)*GRID-2);
				}
			}
		}
	}
	
	/** function that returns object at window position (x,y)
	  * @param x mouse x (relative to this frame)
	  * @param y mouse y (relative to this frame)*/
	public Object pick(int x, int y) {		
		float fx = ((float)x)/GRID-1.5f;
		float fy = ((float)y)/GRID-1.5f;
		for(int i = g.nodes.size()-1; i >= 0; i--) {
			Node n = (Node)g.nodes.elementAt(i);
			if((Math.abs(n.fx-fx)<.5f)&&(Math.abs(n.fy-fy)<.5f)) return n;
		}
		if(mode == DM_SEGMENT) return null;
		for(int i = g.nets.size()-1; i >=0; i--) {
			Net nt = (Net) g.nets.elementAt(i);			
			if(nt.output == null) continue;			
			float sx=nt.output.fx, sy=nt.output.fy;

			for(int j=0; j < nt.inputs.size(); j++) {
				Node n = (Node) nt.inputs.elementAt(j);
				sx += n.fx;
				sy += n.fy;
			}
			sx /= nt.inputs.size()+1;
			sy /= nt.inputs.size()+1;
			if((Math.abs(sx-fx)<.2f)&&(Math.abs(sy-fy)<.2f)) return nt;
		}
		return null;
	}
	
	/** moves object to window position (x,y)
		* @param o object to move 
	  * @param x mouse x (relative to this frame)
	  * @param y mouse y (relative to this frame) */
	public void moveTo(Object o, int x, int y) {		
		float fx = ((float)x)/GRID-1.5f;
		float fy = ((float)y)/GRID-1.5f;
		if(o instanceof Node) {
			Node n = (Node) o;
			n.fx = fx;
			n.fy = fy;		
		}
	}
}
