/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores.edifp;

import java.util.Vector;
import java.util.Stack;
import java.io.PrintWriter;
import java.io.FileInputStream;
import org.opencores.structure.Graph;
import org.opencores.Conf;

/** Parser class provides basic LL(1) parser for EDIF file */
public class Parser {
	/** Are we doing debug here? */
	private static boolean debug = false;
	private Lex lex;
	/** writter, copied from Conf.log */
	private static PrintWriter err; 

  /** parses EDIF file. 
    * @param infile file to parse
    * @return graph read from file
    * @see Graph */
	public static Graph main(String infile) throws Exception {				
		Graph graph = null;
		Lex y = new Lex(new FileInputStream(Conf.sdir + infile));				
		err = Conf.log;
		debug = Conf.debug;
		
		Parser p = new Parser(y);		
		graph = p.parse();				
		return graph;
	}

	/** constructs new parser.
	  * @param lex lexical analizer parser is going to be attached to */
	public Parser(Lex lex) {
		this.lex = lex;
	}	

	/* ****** LL part - utilities ****** */

	/** get next token */
	private Symbol next() throws Exception {
		Symbol ss=lex.getToken();		
		if(debug) err.print(ss);
		return ss;
	}

	/** get specific token, otherwise return error */
	private Symbol get(int id) throws Exception {
		Symbol ss = next();
		if(ss.s != id) error(id);
		return ss;
	}

	/** get NUMBER */
	private int getNum() throws Exception {
		return ((Integer)get(sym.NUMBER).o).intValue();
	}

	/** look for LPAREN */
	private void lp() throws Exception {
		get(sym.LPAREN);
	}

	/** look for RPAREN */
	private void rp() throws Exception {
		get(sym.RPAREN);
	}
	
	/* check for LPAREN - new block or RPAREN - end of this block */
	private boolean checkEmpty() throws Exception {
		Symbol s = next();
		
		if(s.s == sym.LPAREN) return false;
		else
			if(s.s == sym.RPAREN) return true;
			else error(sym.RPAREN);						
		return false;
	}

  /** skip to last RPAREN */
	private void skipBlock(Symbol s) throws Exception {
		int level = 1;
		if(s.s!=26)	err.println("Unknown token found: "+s);
		while(level>0) {
			Symbol ns=next();
			switch(ns.s) {
				case sym.LPAREN:level++;break;
				case sym.RPAREN:level--;break;
			}
		}
	}
	
	/** real last object name */
	private String realName;
	/** num elements for last object ID */ 
	private int numArray;		 
	/** element in array */
	private int arrayElt;		 
	
	/** gets identifier extended */
	private String getID(boolean allowArrays) throws Exception {		
		Symbol ss = next();
		numArray = 1;
		arrayElt = 0;
		switch (ss.s) {			
			case sym.NET:
				ss.o = new String("net");				
				ss.s = sym.ID;realName = (String)ss.o;break;
			case sym.NETLIST:
				ss.o = new String("netlist");				
				ss.s = sym.ID;realName = (String)ss.o;break;				
			case sym.ID: realName = (String)ss.o;break;
			case sym.LPAREN: ss.o = rename(allowArrays);break;
			default: error("Invalid ID"); break;
		}
		return (String)ss.o;
	}	
	
	/** gets identifier */
	private String getID() throws Exception {
		return getID(false);
	}	
	
	/** dissolves rename, member or array */
	private String rename(boolean allowArray) throws Exception {
		Symbol ss = next();
		String s = null;
		switch (ss.s) {
			case sym.RENAME:
				s = getID();
				Symbol srn = get(sym.STRING);
				realName = (String) srn.o;
				rp();
				break;
			case sym.MEMBER:
				s = getID();
				arrayElt = getNum();
				rp();
				break;
			case sym.ARRAY:
				if(!allowArray) error("Array not allowed here.");
				s = getID();
				numArray = getNum();
				rp();
				break;
			default:				
				error("Renaming, member or array expected.");
				break;
		}				
		return s;
	}

	/** reports an error */
	void error(String msg) {
		err.println();
		err.println(msg);		
	  err.close();	  
		throw new Error(msg);
	}
	
	/** reports token # expected */
	void error(int token) {	  
		error("Invalid token found (expected #"+token+").");
	}

	public Graph parse() throws Exception {
		return edif();
	}
	
	/*--------- LL(1) parser ------*/	
	
	/** current namescope */
	private PathScope curPath=new PathScope();
	/** referenced namescope */ 
	private PathScope refPath=new PathScope();
	
	/** dissolves token with same name using LL(1) structure */
	private Graph edif() throws Exception {		
		lp();get(sym.EDIF);
		Edif e = new Edif();
		curPath.edif = e;				
		e.name = getID();
		if(debug) err.println(e.name);
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.EDIFV:										// get version
					e.version[0]=getNum();
					e.version[1]=getNum();
					e.version[2]=getNum();
					rp();break;
				case sym.LIBRARY: library();break; // new library
				case sym.EXTERNAL: external();break;
				case sym.DESIGN: design();break;	// new design
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();			
		}
		Convert cnv = new Convert(e);
		return cnv.graph;
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void library() throws Exception {
		Library l = new Library();
		l.external = false;
		curPath.lib = l;
		curPath.edif.libraries.addElement(l);
		l.name = getID();				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.CELL: cell();break; // new cell				
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void external() throws Exception {
		Library l = new Library();
		l.external = true;
		curPath.lib = l;
		curPath.edif.libraries.addElement(l);
		l.name = getID();				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.CELL: cell();break; // new cell				
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void cell() throws Exception {
		Cell c = new Cell();
		curPath.cell = c;
		c.name = getID();
		curPath.lib.cells.addElement(c);
				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.VIEW:view();break; // new view net
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void view() throws Exception {				
		curPath.cell.view = getID();		
		boolean empty = checkEmpty();
		boolean netlist = false;
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.VIEWTYPE:
					s=next(); // new view netlist?
					if(s.s == sym.NETLIST) {rp();netlist=true;}
					else skipBlock(s);
					break;
				case sym.INTERFACE:_interface();break;
				case sym.CONTENTS:contents();break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void _interface() throws Exception {				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.PORT:port();break; // new port onto cur cell interface
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void port() throws Exception {
		Port p = new Port();
		curPath.port = p;
		p.name = getID(true);
		p.width = numArray;		
		curPath.cell.ports.addElement(p);

		boolean empty = checkEmpty();
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.DIRECTION:
					s = next();
					switch (s.s) {
						case sym.INOUT:p.direction = p.INOUT;break;
						case sym.INPUT:p.direction = p.INPUT;break;
						case sym.OUTPUT:p.direction = p.OUTPUT;break;
						default:error(sym.OUTPUT);break;
					}
					rp();break; // new direction
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void contents() throws Exception {				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.INSTANCE:instance();break; // define new instance
				case sym.NET:net();break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void instance() throws Exception {
		Instance i = new Instance();
		curPath.inst = i;
		i.name = getID();
		curPath.cell.inst.addElement(i);
				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {				
				case sym.VIEWREF:viewRef();i.cell = refPath.cell;break;
				case sym.PROPERTY:property(i);break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */
	private void property(Instance i) throws Exception {
		String name = getID();
		if(name.equalsIgnoreCase("LIBVER")) {
			lp();Symbol s=next();skipBlock(s);rp(); // skip library version
		  return;			
		}
		
		lp();Symbol s=next();skipBlock(s);rp();		// unknown property, skip
	}
		
	/** dissolves token with same name using LL(1) structure */	
	private void net() throws Exception {
		ENet n = new ENet();
		curPath.net = n;
		n.name = getID();
		curPath.cell.nets.addElement(n);

		boolean empty = checkEmpty();
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.JOINED:joined();break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void joined() throws Exception {
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {				
				case sym.PORTREF:
					portRef();
					curPath.net.inst.addElement(refPath.inst);
					curPath.net.ports.addElement(refPath.port);
					curPath.net.ports_i.addElement(new Integer(refPath.port_i));					
					break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
		
	/** dissolves token with same name using LL(1) structure */	
	private void design() throws Exception {
		getID();		
				
		boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.CELLREF:
					cellRef();
					curPath.edif.design=refPath.cell;break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
	
	/** utility for dissolving references */	
	private void dissolveRef(int level) throws Exception {
  	boolean empty = checkEmpty();		
		while(!empty) {
			Symbol s = next();
			switch(s.s) {
				case sym.LIBRARYREF:if(level>=1) libraryRef(); else skipBlock(s);break;// search for ref
				case sym.CELLREF:if(level>=2) cellRef(); else skipBlock(s);break;
				case sym.VIEWREF:if(level>=3) viewRef(); else skipBlock(s);break; 
				case sym.INSTANCEREF:if(level>=4) instanceRef(); else skipBlock(s);break;								
				case sym.PORTREF:if (level>=5) portRef(); else skipBlock(s);break;
				case sym.NET:if(level>=4) netRef(); else skipBlock(s);break;
				default:skipBlock(s);break;	// unsupported block found
			}
			empty = checkEmpty();
		}
	}
		
	/** dissolves token with same name using LL(1) structure */	
	private void libraryRef() throws Exception {
		String ref = getID();		
		rp();								  // remove belonging RPAREN
		Vector refs = curPath.edif.libraries;
				
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((Library)(refs.elementAt(i))).name)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			refPath.lib = (Library)(refs.elementAt(found));
		} else error("Invalid reference '"+ref+"'.");
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void cellRef() throws Exception {
		String ref = getID();
		refPath.lib = null;
		dissolveRef(1);
		if(refPath.lib == null) refPath.lib = curPath.lib;
		Vector refs = refPath.lib.cells;		
		
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((Cell)(refs.elementAt(i))).name)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			refPath.cell = (Cell)(refs.elementAt(found));
		} else error("Invalid reference '"+ref+"'.");
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void viewRef() throws Exception {
		String ref = getID();
		refPath.lib = null;
		dissolveRef(2);
		if(refPath.lib == null) refPath.lib = curPath.lib;
		Vector refs = refPath.lib.cells;
				
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((Cell)(refs.elementAt(i))).view)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			// default view already
		} else error("Invalid reference '"+ref+"'.");
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void portRef() throws Exception {
		String ref = getID();
		int member = arrayElt;
		refPath.inst = null;
		dissolveRef(4);
		Vector refs;
		if(refPath.inst == null) refs = curPath.cell.ports;
		else refs = refPath.inst.cell.ports;		
	
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((Port)(refs.elementAt(i))).name)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			refPath.port = (Port)(refs.elementAt(found));
			refPath.port_i = member;											// set index
			if((member>=refPath.port.width)||(member<0))	// check index
				error("Invalid index.");
		} else error("Invalid reference '"+ref+"'.");
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void instanceRef() throws Exception {
		String ref = getID();
		refPath.cell = null;
		dissolveRef(3);		
		if(refPath.cell == null) refPath.cell = curPath.cell;
		Vector refs = refPath.cell.inst;		
		
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((Instance)(refs.elementAt(i))).name)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			refPath.inst = (Instance)(refs.elementAt(found));
		} else error("Invalid reference '"+ref+"'.");
	}
	
	/** dissolves token with same name using LL(1) structure */	
	private void netRef() throws Exception {
		String ref = getID();
		refPath.cell = null;
		dissolveRef(3);
		if(refPath.cell == null) refPath.cell = curPath.cell;
		Vector refs = refPath.cell.nets;
	
		int found = -1;
		for(int i = 0;(found < 0)&&(i < refs.size()); i++) {
			if(ref.equalsIgnoreCase(((ENet)(refs.elementAt(i))).name)) found = i;
		}
		if(found >=0 ) {	// ref ok?
			refPath.net = (ENet)(refs.elementAt(found));
		} else error("Invalid reference '"+ref+"'.");
	}
}
