/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores;

import java.awt.*;
import java.awt.event.*;
import org.opencores.Conf;
import org.opencores.util.BitStreamWriter;
import org.opencores.structure.*;
import org.opencores.mapping.*;
import org.opencores.placement.*;
import org.opencores.graphics.*;
import org.opencores.routing.*;
import java.io.*;

/** Main class, that should be called from command line
  * @see main */
public final class Main extends Frame {
	/** Graph	we are working on. */
	private static Graph graph;
	/** Main instance. */
	private static Main display = new Main();
	
	/** main function that should be called from command line
	  * @param arg array of parameters
	  * @exception Exception is thrown if unexpected state is encountered */	  
	public static void main(String arg[]) throws Exception {
		if(Conf.setParameters(arg)) { // further processing needed?
			if(Conf.outFile == null) Conf.outFile = Conf.inFile + ".bin";
			switch(Conf.fileType(Conf.inFile)) {
				case Conf.FT_EDIF: // parse EDIF file
					System.out.println("Reading EDIF file "+Conf.inFile);
					graph = org.opencores.edifp.Parser.main(Conf.inFile);					
					break;
				case Conf.FT_VERILOG: // parse Verilog file
					System.out.println("Reading Verilog-HDL file "+Conf.inFile);
					org.opencores.verilogp.Parser p;
					p = new org.opencores.verilogp.Parser(Conf.inFile);
					graph = p.getGraph();
					System.out.println("Done.");
					//if(Conf.debug) Conf.log.println(graph.toString());
					break;
				default:
					throw new Error("Unknown file extension.");
			}
			/* check graph */
			if(Conf.debug) Conf.log.println("Number of repairs: "+ graph.repair(true));
			if(Conf.debug) Conf.log.println("Number of errors : "+ graph.check(true));			
			
			graph.setRandomPositions(Conf.X, Conf.Y);			
			graph.createFloatPositions();
			if(Conf.graphics) {			
				Main display = new Main();		// in graphics mode create display and show
				display.init();							  // progress
			} else {				
				Main display = new Main();		// in graphics mode create display and show
				display.processOption(0);
				display = null;
			}
			//if(Conf.debug) Conf.log.println(graph.toString());			
			if(!Conf.graphics) Conf.log.close();
		}		
	}
	
	/** Constructs new Main object */
	public Main() {		
	}

	/* ************* testing and drawing part ********* */
	/** Canvas like object. */
	private Draw draw;
	/** LineChart window. */
	private LineChart chart = new LineChart(300, 200);
	/** Blue data */
	private LineChartData edata = new LineChartData(Color.blue);
	/** Red data */
	private LineChartData tdata = new LineChartData(Color.red);
	/** Upper label, showing current status. */
	private Label label = new Label();
	/** Lower label, showing current object mouse is over. */
	private Label status = new Label();
	/** Listbox, on right */
	private List list = new List();
	/** @deprecated */
	
	/** Oprion names */
	private final static String OPTIONS[] = {
		"All","Random positions","Mapping",
		"Preplacement","Placement","Routing","Write bitstream"
	};
	
	/** Inits graphics mode */
	private void init() {
		draw = new Draw(graph);
		for(int i = 0; i < OPTIONS.length; i++) list.add(OPTIONS[i]);

		setLayout(null);		
		add(draw);
		add(status);
		add(label);		
		add(list);
		
		chart.addData(edata);
		chart.addData(tdata);				
		chart.setTitle(Conf.NAME+" chart");
		chart.setLocation(750,0);
		chart.setVisible(true);
		
		draw.setBounds(50,50,550,550);				
		list.setBounds(600,160,140,200);
		status.setBounds(30,600,700,20);
		label.setBounds(30,630,700,20);		
		list.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(ActionEvent a) {
				processOption(list.getSelectedIndex());
			}
		});
		addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent e) {
				Conf.log.close();
				System.exit(0);
			}
		});
		setBackground(Color.gray);
		setSize(750,700);		
		setVisible(true);
		setupDraw();
	}
	
	/** Processes option <i>option</i>.
	  * @param option option to process */
	private void processOption(int option) {
		switch(option) {
			case 0:processOption(2);
						 processOption(3);
						 processOption(4);
						 processOption(5);
						 processOption(6);
						 break;
			case 1:graph.setRandomPositions(Conf.X, Conf.Y);
						 graph.createFloatPositions();
						 break;			
			case 2:if(graph.mapped) {
						   System.out.println("Graph already mapped!");
						   if(draw!=null) label.setText("Graph already mapped!");
						   break;
						 }
					   System.out.println("Mapping...");
						 //Conf.log.println(graph.toString("before mapping"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));									 
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 /* insert LUTs before FFs */
						 FlipFlopMap.addLUTs(graph); 
						 //Conf.log.println(graph.toString("after LUTs added"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));									 
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 /* optimal LUT mapping */
						 FlowMap om = new FlowMap();
						 om.labeling(graph);
						 //Conf.log.println(graph.toString("after optimal - labelling"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));
						 Conf.log.println("Number of errors : "+ graph.check(true));
 			 			 Graph g = om.mapping(graph);
			       graph = g;
			       //Conf.log.println(graph.toString("after optimal - mapping"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 
			       GlobalMap gm = new GlobalMap(graph);
			       gm.mapping();
			       //Conf.log.println(graph.toString("after global mapping"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));
						 Conf.log.println("Number of errors : "+ graph.check(true));						 
						 
						 IOCMap.group(graph);
						 GPCMap.group(graph);
						 //Conf.log.println(graph.toString("after GPC groupping"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));									 
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 
						 //Global.mapping(graph); // collect floating LUTs
						 //Conf.log.println(graph.toString("after global net collecting"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));									 
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 /* update graph drawing parameters */
						 graph.name += ".mapped";
						 graph.mapped = true;
						 if(draw!=null) {
						   draw.setGraph(graph);
							 setupDraw();
							}
						 break;
			case 3:System.out.println("Preplacing...");
						 PrePlacement pp = new PrePlacement(graph);
						 pp.setGraph(graph);
						 pp.prePlacement();
						 System.out.println("Number of repairs: "+ graph.repair(true));									 
						 System.out.println("Number of errors : "+ graph.check(true));
						 graph.createFloatPositions();
						 graph.name += ".preplaced";						 
						 Conf.log.println("IOCs: "+100*pp.nIOC/Conf.NUM_IOC+"%");
						 Conf.log.println("GPCs: "+100*pp.nGPC/Conf.NUM_GPC+"%");
						 if(draw!=null) {
							 draw.setMode(draw.DM_GRID);
						   label.setText("IOCs: "+100*pp.nIOC/Conf.NUM_IOC+"%   "													    
												   + "GPCs: "+100*pp.nGPC/Conf.NUM_GPC+"%");
						 }
						 break;
			case 4:System.out.println("Placing...");
						 Conf.log.println(graph.toString("before annealing"));
						 Conf.log.println("Number of repairs: "+ graph.repair(true));									 
						 Conf.log.println("Number of errors : "+ graph.check(true));
						 Annealing a = new Annealing(graph);
						 if(draw!=null) {
							 edata.resetData();
						   tdata.resetData();
						 }
						 while(true) {
							 a.anneal();							 			 
							 if(a.doneCriteria()) break;
							 a.updateTemperature();
							 if(draw!=null) {								 
								 edata.addData(a.getWiringCost());	// energy
							   tdata.addData(a.getTemp());				// temperature
							   label.setText("energy "+a.getWiringCost()+" temperature "+a.getTemp());
							 }
						 }			
						  
						 graph.name += ".placed";
						 if(draw!=null) {
							 graph.createFloatPositions();
						   draw.setMode(draw.DM_GRID);
						   chart.repaint();
						 }
						 Statistics sc = new Statistics(graph);
						 for(int i = 0; i < sc.distCnt.length; i++)
						   Conf.log.println(i+"\t"+sc.distCnt[i]);
						 Conf.log.println();
						 break;
			case 5:System.out.println("Routing...");
						 graph.constructSegments();
						 graph.emtpySegments();
						 Wavefront wf = new Wavefront(graph);
						 boolean routed = wf.route();
						 if(draw!=null) label.setText("min free segments: "+wf.minFreeSegment()+" routed "+routed);
						 /*for(int i = 0; i < wf.statWireCnt.length; i++)
							 Conf.log.println(i+"\t"+wf.statLenCnt[i]+"\t"+wf.statHopCnt[i]+"\t"+wf.statWireCnt[i]);
						 Conf.log.println();						 */
						 Conf.log.println("Number of unrouted nets "+wf.unroutedNets.size()+"/"+graph.nets.size());
						 System.out.println("Number of unrouted nets "+wf.unroutedNets.size()+"/"+graph.nets.size());
						 for(int i = 0; i < wf.unroutedNets.size(); i++) {
							 Conf.log.print(((Net)wf.unroutedNets.elementAt(i)).name+", ");
							 System.out.print(((Net)wf.unroutedNets.elementAt(i)).name+", ");
						 }
						 System.out.println();
						 Conf.log.println();
						 if(draw!=null) draw.setMode(draw.DM_SEGMENT);
						 if(routed)graph.name += ".routed";
						 break;
			case 6:System.out.println("Writing bistream...");
						 try {
							 BitStreamWriter stream = new BitStreamWriter(new BufferedOutputStream(new FileOutputStream(Conf.outFile)));
							 graph.writeBitstream(stream, Conf.clear);
						   stream.close();
						 } catch(IOException e) {
							 e.printStackTrace();							 
							 Conf.log.close();
							 System.exit(0);
						 }
						 break;
		}		
		//Conf.log.println(graph.toString());
		System.out.println("Number of repairs: "+ graph.repair(true));									 
		System.out.println("Number of errors : "+ graph.check(true));		
		if(draw!=null) draw.repaint();
	}
	
	/** Inits main drawing frame. */
	private void setupDraw() {
		setTitle(Conf.NAME + " graphics view - " + graph.name);
		draw.addMouseMotionListener(new MouseMotionListener() {
			public void mouseDragged(MouseEvent e) {}
			public void mouseMoved(MouseEvent e) {				
				int nx = e.getX();
				int ny = e.getY();
				Object n = draw.pick(nx, ny);
				String s = "";
				if(n != null) {
					s = n.toString();
				}
				status.setText(s);
			}
		});
	}
}
