/* Copyright (C) 2001  Marko Mlinar
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */
package org.opencores;

import java.io.PrintWriter;
import java.io.FileWriter;
import java.io.BufferedWriter;

/** Class, where FPGA and algorithm tuning data is located */
public class Conf {
	/** maximum FPGA size */
	public final static int X = 24; // not counting ports
	/** maximum FPGA size */
	public final static int Y = 24; // not counting ports
	/** number of inputs per LUT */
	public final static int K = 5;

	/** number of GPCs in FPGA */
	public final static int NUM_GPC = X*Y;
	/** number of ports in FPGA */
	public final static int NUM_IOC = 2*(X + Y); // no ports at corners
	/** number of special resources in FPGA */
	public final static int NUM_SR = 6;
	
	/** minimum number of global net users should be &gt= 1*/
	public final static int GNET_MIN = 1;

  /** Direction */
  public final static int DIR_N = 0;
  /** Direction */
  public final static int DIR_S = 1;
  /** Direction */
  public final static int DIR_W = 2;
  /** Direction */
  public final static int DIR_E = 3;  
  
  /** Opposite directions */
  public final static int OPPOSITE_DIR[] = {
		DIR_S, DIR_N, DIR_E, DIR_W };  
  	 
	/** log output stream */
	public static PrintWriter log;			
	
	/** program name */
	public static final String NAME = "KRPAN";		 
	/** program version */
	public static final String VERSION = "0.2";
	/** input file name */
	public static String inFile;
	/** output file name */
	public static String outFile = null;
	/** log file name */
	public static String logFile = NAME + ".log";	 
	/** command line help */
	public static boolean help = false;
	/** display verbose info? */
  public static boolean verbose = false;				
	/** display debug info? */  	
  public static boolean debug = false;
  /** display graphically? */
  public static boolean graphics = false;
  /** source dir */
  public static String sdir = "";
  /** dest dir */
  public static String ddir = "";
  /** should be unused cells included in configuration */
  public static boolean clear = false;
  
  /** Bitstream GPC column header. */
  public static final int BSH_GPC = 0x00C0;
  /** Bitstream IOC column header. */
  public static final int BSH_IOC = 0x00C1;

  /** string representing line separator */
	public static final String NL = "\r\n";
 		
	/** two line program description as displayed at start */
	public static final String PROGRAM_DESC =
	"Program "+NAME+" v"+VERSION+", FPGA Mapping&Placement&Routing utility."+NL
 +"(C) 2001 Marko Mlinar."+NL
 +NAME+" comes with ABSOLUTELY NO WARRANTY; see license.txt for details.";
 
  /** parameter help text as displayed with option -h */ 
  public static final String PARAM_HELP =				
  "\n\tUsage: "+NAME+" input_file [output_file] [-option [-option ...]]"+NL
 +"\n Valid options:"+NL
 +" -h -? --help     this screen"+NL
 +" -v --verbose     verbose output to log file"+NL
 +" -d --debug		   verbose debug output to log file"+NL
 +" -g --graphics    display progress graphically"+NL
 +" -sd --sdir {dir} sets source dir (default \".\")"+NL
 +" -dd --ddir {dir} sets destination dir (default \".\")"+NL  
 +" -l  --log {file} sets different log file (default \""+logFile+"\")"+NL
 +NL
 +" -c --clear       include unused cells in configuration"+NL
 +NL
 +"Usage example:"+NL
 +NAME+" -sd examples -dd . -g test.v"+NL
 +"(generates bit stream specification file named 'test.v.bin', if successful)"+NL;

  /** all options listed here */
	public static final String OPTION_NAME[] = { 
		"-h", "-?", "--help",	"-v", "--verbose", "-d", "--debug", "-g", "--graphics",
		"-sd", "--sdir", "-dd","--ddir", "-l", "--log"};
	
	/** matching enum for OPTION_NAME @see OPTION_NAME */
	public static final int OPTION_ENUM[] = {
		1, 1, 1, 2, 2, 3, 3, 4, 4,
		5, 5, 6, 6,	7, 7, 8, 8 };
	
	/** current parameter index */
	private static int pIdx;
	
	/** Reads and processes parameters and options
		* @param param Command line parameters 
		* @return further processing needed */	
	public static boolean setParameters(String[] param) throws Exception {
		int nonOption = 0;
		System.out.println(PROGRAM_DESC);
		for(pIdx = 0; pIdx < param.length; pIdx++) {
			if(param[pIdx].charAt(0) == '-') {
				int found = -1;
				for(int i = 0; i < OPTION_NAME.length; i++)	// search for options
					if(OPTION_NAME[i].equalsIgnoreCase(param[pIdx])) {found = i;break;}

				if(found >= 0) setOption(param, OPTION_ENUM[found]);
				else throw new Error("Invalid option "+param[pIdx]);
			} else {
				switch(nonOption) {
					case 0: inFile = param[pIdx];break;	// input file parameter
					case 1: outFile = param[pIdx];break;// log file parameter
					default: throw new Error("Invalid parameter "+param[pIdx]);
				}
				nonOption++;
			}
		}
		if(nonOption == 0) {
			System.out.print(PARAM_HELP);
			return false;	// no further processing needed
		} else {
			log = new PrintWriter(
				new BufferedWriter(new FileWriter(ddir+logFile)), false);			
			return true;	// do P&R work
		}
	}

	/** function, that set command line option
	  * @param param all parameters
	  * @param option option as set in OPTION_ENUM and OPTION_NAME */
	private static void setOption(String param[], int option) {
		switch(option) {
			case 1: help = true; break;
			case 2: verbose = true; break;
			case 3: debug = true; break;
			case 4: graphics = true; break;
			case 5: if(++pIdx >= param.length) throw new Error("Directory expected");
							String dir = toDirectory(param[pIdx]);
							sdir = dir; break;
			case 6: if(++pIdx >= param.length) throw new Error("Directory expected");
							dir = toDirectory(param[pIdx]);
							ddir = dir; break;
			case 7: if(++pIdx >= param.length) throw new Error("File name expected");
							logFile = param[pIdx]; break;							
			case 8: clear = true; break;
			default: throw new Error("Invalid parameter.");
		}
	}
	
	/** Converts String into directory name */
	private static String toDirectory(String dir) {
		int l = dir.length();
		//dir.replace('\\','/');
		if(dir.charAt(l-1) == '/') return dir;
		else return dir+'/';
	}
	
	/** extension names */
	public final static String exts[]=
	{"edf","v"};
	/** extension types */
	public final static int FT_UNKNOWN = -1;
	public final static int FT_EDIF = 0;
	public final static int FT_VERILOG = 1;
	
	/** returns file type, based on extension */
	public static final int fileType(String filename) {	
		int found = -1;
		try {
			/* search for dot */
			for(int i = filename.length() - 1; i >= 0; i--) {
				if(filename.charAt(i) == '.') {found = i;break;}
			}
			if(found < 0) return FT_UNKNOWN;
			String ext = filename.substring(found+1);
			for(int i = 0; i < exts.length; i++)
				if(exts[i].equalsIgnoreCase(ext)) return i;
		} catch(Exception e) {
			return FT_UNKNOWN;
		}
		return FT_UNKNOWN;
	}
}
