/* dapkg_gen -- Distributed Arithmetic package generator
 * Copyright (C) 2000 Jamil Khatib
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */      

/* This program generates the DApkg.vhd file that is used to define 
 * the DA filter core and gives its parameters and the contents of the 
 * Distributed Arithmetic Look-up-table "DALUT" according to the DA algorithm
 */

/*
 *
 * To compile the program do: gcc -o DApkg_gen DApkg_gen.c
 *
 */
                                                                    
/*  INCLUDES */
#include <stdio.h>



/*  DEFINES  */

#define MAXTAPS 64		 		/* Maximum number of taps */
#define MAXWORDSIZE 64   		/* Maximum word size      */
#define MAXROMSIZE pow2(MAXTAPS)/* Maximum ROM size	      */



void tobinstr(unsigned int ,unsigned int ,char * );

unsigned int pow2(unsigned int );

void generate_LUT(unsigned int [] , unsigned int []  ,unsigned int );


/* Main function */
    
main(int argc,char **argv)
{
  
  unsigned int taps = 0;
  unsigned int word_size = 0;

  unsigned int constants[MAXTAPS];

  unsigned int constants_table[MAXROMSIZE];
  
  char binary_num[MAXWORDSIZE+3];

  unsigned int rom_size = 0;
  
  int i =0;
  
  FILE * f;
  
  printf("\n*********DApkg.vhd generator**********\n");


  printf("\nEnter word size ");
  scanf("%d",&word_size);
  
   if( word_size <= 0 )
    {
      printf("Negative numbers are not allowed\n");
      exit(1);
      
    }

  
  printf("\nEnter no. of filter taps ");
  scanf("%d",&taps);
  
  if( taps <= 0 )
    {
      printf("Negative numbers are not allowed\n");
      exit(1);
      
    }

  rom_size = pow2(taps); /* Calculate rom size */
  
  
      
   printf("\nEnter filter constants starting from tap[0]");
   
     for( i =0;i < taps; i++)
       {
         printf("\n[%d]:",i);
         scanf("%d",&constants[i]);

         
         if(constants[i] >= (pow2(word_size)))
           {

             printf("[Error]: large constant >= [%d]\n",pow2(word_size));
             exit(1);
                          
           }
                  
       }
   
  printf("\nFilter information\n");
  printf("\nFilter taps = [%d]",taps);
  printf("\nWord size = [%d]",word_size);
  printf("\nFilter constants\n");

  for( i =0; i< taps; i++)
  {
    printf("\n[%d]:[%d]",i,constants[i]);
    
  }

  printf("\n");

  printf("\nLUT contents\n");

 /* Calculate LUT contents */
  generate_LUT(constants, constants_table ,taps);
     
    
  f = fopen("DApkg.vhd","w+");
 
 fprintf(f,"-------------------------------------------------------------------------------
-- Title      :Distributed Arithmetic Package
-- Project    : Arithmetic blocks
-------------------------------------------------------------------------------
-- File        : DApkg.VHD
-- Author      : Jamil Khatib  
-- Organization: OpenIPCore Project
-- Created     : 2000/04/17
-- Last update : 2000/04/17
-- Platform    : 
-- Simulators  : Modelsim 5.3XE / Windows98
-- Synthesizers: Leonardo / WindowsNT
-- Target      : 
-- Dependency  : 
-------------------------------------------------------------------------------
-- Description: Distributed Arithmetic Package
-------------------------------------------------------------------------------
-- Copyright (c) 2000 Jamil Khatib
-- 
-- This VHDL design file is an open design; you can redistribute it and/or
-- modify it and/or implement it under the terms of the Openip General Public
-- License as it is going to be published by the OpenIPCore Organization and
-- any coming versions of this license.
-- You can check the draft license at
-- http://www.opencores.org/OIPC/license.html

-------------------------------------------------------------------------------
-- Revisions  :
-- Revision Number :   1
-- Version         :   0.1
-- Date            :   17th Apr 2000
-- Modifier        :   Jamil Khatib (khatib@ieee.org)
-- Desccription    :   Created
--
-------------------------------------------------------------------------------
-- Revisions  :
-- Revision Number :   2
-- Version         :   0.2
-- Date            :   30h Apr 2000
-- Modifier        :   Jamil Khatib (khatib@ieee.org)
-- Desccription    :   defining the constant CONSTANTS for DALUT contents
--					   which should be genrated automatically in later versions
--
-------------------------------------------------------------------------------

-- This file was generated by DApkg_gen binary tool

library ieee;
use ieee.std_logic_1164.all;

package DApkg is
  constant BUFFER_SIZE : integer := %d;  -- No. of inputs, No of filter taps,
                                        -- No. of constants , Buffer Size
  constant DATA_SIZE   : integer := %d;  -- Word size

  type TABLE is array ( 0 to (2**BUFFER_SIZE) -1) of std_logic_vector((DATA_SIZE -1 ) downto 0);
                                        -- Memory type
  constant CONSTANTS : TABLE := (\n\n",taps,word_size);

 
 /**************************************/
 
 /* write teh LUT contents into the file */
 for(i = 0; i < rom_size-1 ;i++)
 {

     tobinstr(constants_table[i],word_size,binary_num);
     printf("\nDALUT[%d] = [%d]   %sb",i,constants_table[i],binary_num);
     fprintf(f,"   %s,\n",binary_num);
     
 }

 
 tobinstr(constants_table[i],word_size,binary_num);
 printf("\nDALUT[%d] = [%d]   %sb",i,constants_table[i],binary_num);
 fprintf(f,"   %s\n",binary_num);
 
 /************************************/

 fprintf(f,"\n);\n-------------------------------------------------------------------------------
-- DA filter top core

  component DA
    generic (
      NOINPUTS  :     integer := BUFFER_SIZE;  -- number of inputs, no. of filter taps
      WORD_SIZE :     integer := DATA_SIZE;  -- word size
      CONTENTS  :     TABLE   := CONSTANTS);
    port (
      data      : in  std_logic_vector(WORD_SIZE -1 downto 0);  -- input data
      Clk       : in  std_logic;        -- system clock
      Rst_n     : in  std_logic;        -- System reset
      Result    : out std_logic_vector(WORD_SIZE -1 downto 0);  -- Result
      ValidOut  : out std_logic;        -- Output valid
      Overflow  : out std_logic);       -- Overflow signal
  end component;
-------------------------------------------------------------------------------
-- DALUT Distributed Arithmetic Lookup table

  component DALUT
    generic (
      ADDR_SIZE   : integer;                   --:= BUFFER_SIZE;
      -- LUT address size or number of input terms
      OUTPUT_SIZE : integer;  --:= DATA_SIZE;  -- Output size
      CONTENTS    : TABLE                      -- CONSTANTS
      -- These values should be generated automaticlly from the original
      -- constants of the filter

      );

    port (
      Address : in  std_logic_vector(ADDR_SIZE -1 downto 0);  -- DALUT Address
      Result  : out std_logic_vector(OUTPUT_SIZE -1 downto 0));  -- Result

  end component;

  ------------------------------------------------------------------------------

end DApkg;
package body DApkg is

end DApkg;");
 
  fclose(f);
  
  printf("\nDApkg.vhd is generated\n");

  
  

  return 0;
  
}

/******************************************************************
 * void tobinstr(unsigned int num,unsigned int size,char * str)   
 *
 * tobinary converts from number to binary representation string 
 *
 * Inputs:
 * num : the number to be converted
 * size: number of bits in the binary number
 * str: string buffer to store the converted number 
 ******************************************************************/

void tobinstr(unsigned int num,unsigned int size,char * str)
{
  unsigned int i = size;
  
  unsigned int tmp =0;
  
  str[0] = '"';
  str[size +1] = '"';
  str[size+2] = NULL;
  
  for (i=size; i > 0 ; i--)
    {
	        
	 tmp = num & 0x1;

         
     switch(tmp)
     {
       case 0:
           str[i] = '0';
           break;
         case 1:
             str[i] = '1';
             break;
         default :
             printf("\n!!!!!!!!! Illegal number conversion !!!!!!!!!!\n");
             exit(1);

	 }		


      num = num >> 1;
      
    }
  

  return ;
  
}


/******************************************************************
 * unsigned int pow2(unsigned int num)
 *
 * pow2  calculates 2^num 
 *
 * Inputs:
 * num : the power
 ******************************************************************/

unsigned int pow2(unsigned int num)
{
    unsigned int res = 1;
    int i=0;
    
    for(i = 0; i < num; i++)
    {
        res = 2 * res;
        
        
    }
    return res;
    
}


/******************************************************************
 * void generate_LUT(unsigned int  consts[], unsigned int  rom[],unsigned int num_const)
 *
 * Generates the LUT (rom) contents
 *
 * Inputs:
 * consts[] : Filter's constants
 * rom[]    : Array to store the LUT constants 
 * num_const: Number of filter's constants
 * 
 ******************************************************************/

void generate_LUT(unsigned int  consts[], unsigned int  rom[],unsigned int num_const)
{

    unsigned int i= 0;
    unsigned int j = 0;
    unsigned int max_addr = pow2(num_const);
    unsigned int flag = 0;
    unsigned index = 0;
    
    
    for (i= 0;i < max_addr; i++)/* i = rom address*/
    {
        index = i;
        rom[i] = 0;
        
        for (j = 0; j < num_const; j++)
        {
            flag = 0x1 & index;  /* Mask the first bit*/

            rom[i] = rom [i] + ( consts[j] * flag);
            
            index >>=1; /* get next bit in the address*/
            
        }

        
        
    }
    
    
    return ;
}
