// $Id: GetOpt.java,v 1.5 1997/10/14 15:37:56 xzhu Exp $
// GetOpt.java
//
// Get input options, port to java from GNU getopt.c, ... (glib2.9.5)
// 
// Author: Xiaokun Kelvin ZHU
// Address: kelvin@iname.com
// 1997, 09, 13
// Version 1.1
/*
   Declarations for getopt.
   Copyright (C) 1989,90,91,92,93,94,96,97 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/**
 GetOpt is java bean.

 properties:
   TYPE		NAME			(Read/Write)
   -------------------------------------------------
   String	shortOption;		W
   String[]	longOptionName;		W
   int[]	longOptionHasArg;	W
   int[]	longOptionValue;	R/W
   int		optionMode;		R/W
   int		optarg;			R
   int		optind;			R
   int		ordering;		R
   int		longind;		R
   String	inputArgs;		W
 
 Usage (only for short arguments):

 GetOpt g = new GetOpt();               // create a GetOpt `g' object with
 g.setInputArgs(args);                  // set input arguments
 g.setShortOption("a:fg::");            // options -a xxx -f -g[xxx]
					// GetOpt.EOF is end mark
 try {

 int c = g.getopt();                    // get next option 

 String value = g.getOptarg();		// get corresponding value

 int optind = g.getOptind();		// get current index in arguments

 }
 catch (GetOptException e)
 { System.out.println(e.getMessage()); } // when something go wrong,
 					 // GetOptException be thrown.

 See testGetOpt.java for mix long and short arguments.
 */
package zxk.util;

import java.io.Serializable;


public class GetOpt implements Serializable
{
  /* Describe how to deal with options that follow non-option ARGV-elements.

     If the caller did not specify anything,
     the default is REQUIRE_ORDER if the environment variable
     POSIXLY_CORRECT is defined, PERMUTE otherwise.

     REQUIRE_ORDER means don't recognize them as options;
     stop option processing when the first non-option is seen.
     This is what Unix does.
     This mode of operation is selected by either setting the environment
     variable POSIXLY_CORRECT, or using `+' as the first character
     of the list of option characters.
   
     PERMUTE is the default.  We permute the contents of ARGV as we scan,
     so that eventually all the non-options are at the end.  This allows options
     to be given in any order, even with programs that were not written to
     expect this.

     RETURN_IN_ORDER is an option available to programs that were written
     to expect options and other ARGV-elements in any order and that care about
     the ordering of the two.  We describe each non-option ARGV-element
     as if it were the argument of an option with character code 1.
     Using `-' as the first character of the list of option characters
     selects this mode of operation.
    
     The special argument `--' forces an end of option-scanning regardless
     of the value of `ordering'.  In the case of RETURN_IN_ORDER, only
     `--' can cause `getopt' to return -1 with `optind' != ARGC.  */
  private static final int REQUIRE_ORDER = 0;
  private static final int PERMUTE = 1;
  private static final int RETURN_IN_ORDER = 2;

  public static final int EOF = -1;

  private int ordering;

  public int getOrdering()
  {
    return ordering;
  }

  /** Value of POSIXLY_CORRECT environment variable. */
  private String posixly_correct;

  /** For communication from `getopt' to the caller.
   When `getopt' finds an option that takes an argument,
   the argument value is returned here.
   Also, when `ordering' is RETURN_IN_ORDER,
   each non-option ARGV-element is returned here.  */
  private String optarg = null;

  /** get property optarg.
   *  @return optarg.
   */
  public String getOptarg()
  {
    return optarg;
  }

  /** Index in ARGV of the next element to be scanned.
   This is used for communication to and from the caller
   and for communication between successive calls to `getopt'.

   On entry to `getopt', zero means this is the first call; initialize.

   When `getopt' returns -1, this is the index of the first of the
   non-option elements that the caller should itself scan.

   Otherwise, `optind' communicates from one call to the next
   how much of ARGV has been scanned so far.  
      1003.2 says this must be 1 before any call. 
      Formerly, initialization of getopt depended on optind==0, which
      causes problems with re-calling getopt as programs generally don't
      know that. 
      Java args[] starts from 0.
      */
  private int optind;

  /** get property optind.
   *  @return optind.
   */
  public int getOptind()
  {
    return optind;
  }

  /** Callers store false here to inhibit the error message `getopt' prints
   for unrecognized options.  */
  // private boolean opterr = true;

  /** Set to an option character which was unrecognized.  */
  // private int optopt = '?';

  /** long option index */
  private int longind;

  /** get property longind.
   *  @return longind.
   */
  public int getLongind()
  {
    return longind;
  }

  /** The next char to be scanned in the option-element
   in which the last option character we returned was found.
   This allows us to pick up the scan where we left off.

   If this is zero, or a null string, it means resume the scan
   by advancing to the next ARGV-element.  */
  private String nextchar = null;

  private boolean firstInitial;

  /** Create a GetOpt object. */
  public GetOpt()
  {
    firstInitial = true;
    shortOption = null;
    longOptionName = null;
    longOptionHasArg = null;
    longOptionValue = null;
    optionMode = SHORT_OPT;	
    optarg = null;
    optind = 0;
    longind = 0;
    inputArgs = null;
  }

  private String[] inputArgs;

  /** set property inputArgs.
   *  @param args new input arguments.
   */
  public void setInputArgs(String[] args)
  {
    inputArgs = args;
  }

  private String shortOption;

  /** set property shortOption.
   *  @param option new short options.
   */
  public void setShortOption(String option)
  {
    shortOption = option;
  }

  /** Describe the long-named options requested by the application.
   The LONG_OPTIONS argument to getopt_long or getopt_long_only is a vector
   of `struct option' terminated by an element containing a name which is
   zero.

   The field `has_arg' is:
   no_argument		(or 0) if the option does not take an argument,
   required_argument	(or 1) if the option requires an argument,
   optional_argument 	(or 2) if the option takes an optional argument.

   If the field `flag' is not NULL, it points to a variable that is set
   to the value given in the field `val' when the option is found, but
   left unchanged if the option is not found.

   To have a long-named option do something other than set an `int' to
   a compiled-in constant, such as set a value from `optarg', set the
   option's `flag' field to zero and its `val' field to a nonzero
   value (the equivalent single-letter option character, if there is
   one).  For long options that have a zero `flag' field, `getopt'
   returns the contents of the `val' field.  */

  private String[] longOptionName;

  /** set property longOptionName.
   *  @param names new long option names.
   */
  public void setLongOptionName(String[] names)
  {
    longOptionName = names;
  }

  public static final int NO_ARG = 0; //if the option does not take an argument
  public static final int REQ_ARG = 1; //if the option requires an argument
  public static final int OPT_ARG = 2; //if the option takes an optional argument.

  private int[] longOptionHasArg;

  /** set property longOptionHasArg.
   *  @param hasArg new long option hasArg.
   */
  public void setLongOptionHasArg(int[] hasArg)
  {
    longOptionHasArg = hasArg;
  }

  private int[] longOptionValue;

  /** set property longOptionValue.
   *  @param value new long option value.
   */
  public void setLongOptionValue(int[] value)
  {
    longOptionValue = value;
  }

  /** implement getopt_longOnly(). 
   Like getopt_long, but '-' as well as '--' can indicate a long option.
   If an option that starts with '-' (not '--') doesn't match a long option,
   but does match a short option, it is parsed as a short option
   instead.  */
  public static final int SHORT_OPT = 0;
  public static final int LONG_OPT = 1;
  public static final int LONG_ONLY_OPT = 3;

  private int optionMode;

  /** set property optionMode.
   *  @param mode new option mode (GetOpt.SHORT_OPT, GetOpt.LONG_OPT,
   *              GetOpt.LONG_ONLY_OPT).
   */
  public void setOptionMode(int mode)
  {
    optionMode = mode;
  }

  /** get property optionMode.
   *  @return option mode (GetOpt.NO_ARG, GetOpt.REQ_ARG,GetOpt.OPT_ARG).
   */
  public int getOptionMode()
  {
    return optionMode;
  }

  /** get property longOptionValue.
   *  @return long option value.
   */
  public int getLongOptionValue(int longIndex)
  {
    return longOptionValue[longIndex];
  }

  /** This version of `getopt' appears to the caller like standard Unix `getopt'
   but it behaves differently for the user, since it allows the user
   to intersperse the options with the other arguments.

   As `getopt' works, it permutes the elements of ARGV so that,
   when it is done, all the options precede everything else.  Thus
   all application programs are extended to handle flexible argument order.

   Setting the environment variable POSIXLY_CORRECT disables permutation.
   Then the behavior is completely standard.

   GNU application programs can use a third alternative mode in which
   they can distinguish the relative order of options and other arguments. 

   Scan elements of ARGV (whose length is ARGC) for option characters
   given in OPTSTRING.

   If an element of ARGV starts with '-', and is not exactly "-" or "--",
   then it is an option element.  The characters of this element
   (aside from the initial '-') are option characters.  If `getopt'
   is called repeatedly, it returns successively each of the option characters
   from each of the option elements.

   If `getopt' finds another option character, it returns that character,
   updating `optind' and `nextchar' so that the next call to `getopt' can
   resume the scan with the following option character or ARGV-element.

   If there are no more option characters, `getopt' returns -1.
   Then `optind' is the index in ARGV of the first ARGV-element
   that is not an option.  (The ARGV-elements have been permuted
   so that those that are not options now come last.)

   OPTSTRING is a string containing the legitimate option characters.
   If an option character is seen that is not listed in OPTSTRING,
   return '?' after printing an error message.  If you set `opterr' to
   zero, the error message is suppressed but we still return '?'.

   If a char in OPTSTRING is followed by a colon, that means it wants an arg,
   so the following text in the same ARGV-element, or the text of the following
   ARGV-element, is returned in `optarg'.  Two colons mean an option that
   wants an optional arg; if there is text in the current ARGV-element,
   it is returned in `optarg', otherwise `optarg' is set to zero.

   If OPTSTRING starts with `-' or `+', it requests different methods of
   handling the non-option ARGV-elements.
   See the comments about RETURN_IN_ORDER and REQUIRE_ORDER, above.

   Long-named options begin with `--' instead of `-'.
   Their names may be abbreviated as long as the abbreviation is unique
   or is an exact match for some defined option.  If they have an
   argument, it follows the option name in the same ARGV-element, separated
   from the option name by a `=', or else the in next ARGV-element.
   When `getopt' finds a long-named option, it returns 0 if that option's
   `flag' field is nonzero, the value of the option's `val' field
   if the `flag' field is zero.

   The elements of ARGV aren't really const, because we permute them.
   But we pretend they're const in the prototype to be compatible
   with other systems.

   LONGOPTS is a vector of `struct option' terminated by an
   element containing a name which is zero.

   LONGIND returns the index in LONGOPT of the long-named option found.
   It is only valid when a long-named option has been found by the most
   recent call.

   If LONG_ONLY is nonzero, '-' as well as '--' can introduce
   long-named options.  

   @return the candidate option in short option string, or the value in
           the long option value array. If no any exists, it retutn GetOpt.EOF
  */
  public int getopt() throws GetOptException
  {
    if ( firstInitial )
      getopt_initialize ();

    optarg = null;
    int i;
    boolean longOnly = false;
    if (optionMode == LONG_ONLY_OPT)
      longOnly = true;

    /* Test whether ARGV[optind] points to a non-option argument.
     Either it does not have option syntax, or there is an environment flag
     from the shell indicating it is not an option.  The later information
     is only used when the used in the GNU libc.  */

    if (nextchar == null || nextchar.length() == 0)
    {
      /* Advance to the next ARGV-element.  */

      /* Give FIRST_NONOPT & LAST_NONOPT rational values if OPTIND has been
	 moved back by the user (who may also have changed the arguments).  */
      if (last_nonopt > optind)
	last_nonopt = optind;
      if (first_nonopt > optind)
	first_nonopt = optind;

      if (ordering == PERMUTE)
	{
	  /* If we have just processed some options following some non-options,
	     exchange them so that the options come first.  */

	  if (first_nonopt != last_nonopt && last_nonopt != optind)
	    exchange ();
	  else if (last_nonopt != optind)
	    first_nonopt = optind;

	  /* Skip any additional non-options
	     and extend the range of non-options previously skipped.  */

	  while (optind < inputArgs.length && (inputArgs[optind].charAt(0) 
	    != '-' || inputArgs[optind].length() == 1))
	    optind++;
	  last_nonopt = optind;
	}

      /* The special ARGV-element `--' means premature end of options.
	 Skip it like a null option,
	 then exchange with previous non-options as if it were an option,
	 then skip everything else like a non-option.  */

      if (optind != inputArgs.length && inputArgs[optind].equals("--"))
	{
	  optind++;

	  if (first_nonopt != last_nonopt && last_nonopt != optind)
	    exchange ();
	  else if (first_nonopt == last_nonopt)
	    first_nonopt = optind;
	  last_nonopt = inputArgs.length;

	  optind = inputArgs.length;
	}

      /* If we have done all the ARGV-elements, stop the scan
	 and back over any non-options that we skipped and permuted.  */
      if (optind == inputArgs.length)
	{
	  /* Set the next-arg-index to point at the non-options
	     that we previously skipped, so the caller will digest them.  */
	  if (first_nonopt != last_nonopt)
	    optind = first_nonopt;
	  return GetOpt.EOF;
	}

      /* If we have come to a non-option and did not permute it,
	 either stop the scan or describe it to the caller and pass it by.  */

      if (inputArgs[optind].charAt(0) != '-' || inputArgs[optind].length() == 1)
	{
	  if (ordering == REQUIRE_ORDER)
	    return GetOpt.EOF;
	  optarg = inputArgs[optind++];
	  return 1;
	}

      /* We have found another option-ARGV-element.
	 Skip the initial punctuation.  */

      if (longOptionName != null && inputArgs[optind].charAt(1) == '-')
        nextchar = inputArgs[optind].substring(2);
      else
        nextchar = inputArgs[optind].substring(1);
    }
//System.out.println("zxk: " + nextchar);

    /* Decode the current option-ARGV-element.  */

    /* Check whether the ARGV-element is a long option.

     If longOnly and the ARGV-element has the form "-f", where f is
     a valid short option, don't consider it an abbreviated form of
     a long option that starts with f.  Otherwise there would be no
     way to give the -f short option.

     On the other hand, if there's a long option "fubar" and
     the ARGV-element is "-fu", do consider that an abbreviation of
     the long option, just like "--fu", and not "-f" with arg "u".

     This distinction seems to be the most useful approach.  */

    if (longOptionName != null && (inputArgs[optind].charAt(1) == '-' || 
      (longOnly && (inputArgs[optind].length() >1 || 
      my_index(shortOption, inputArgs[optind].charAt(1)) == null))))
    {
      String nameend;
      int p;
      int indfound = -1;
      boolean exact = false;
      boolean ambig = false;
      int option_index;
      int long_len;
      String newName;
      i = nextchar.indexOf('='); 
      if (i == -1)
      {
        nameend = new String("");
	newName = new String(nextchar);
      }
      else
      {
        nameend = nextchar.substring(i+1);
	newName = nextchar.substring(0,i);
      }

//System.out.println("zxk3: " + nameend + "," + newName);
      /* Test all long options for either exact match
	 or abbreviated matches.  */
      long_len = longOptionName.length;
      for (option_index=0; option_index<long_len; option_index++)
      {
        p = option_index;
	if (longOptionName[p].startsWith(newName))
	  {
	    if (i == longOptionName[p].length())
	      {
		/* Exact match found.  */
		indfound = option_index;
		exact = true;
		break;
	      }
	    else if (indfound == -1)
	      {
		/* First nonexact match found.  */
		indfound = option_index;
	      }
	    else
	      /* Second or later nonexact match found.  */
	      ambig = true;
	  }
      }

      if (ambig && !exact)
	{
	  nextchar = null;
	  optind++;
	  throw new GetOptException("option " + inputArgs[optind] + 
	    " is ambiguous");
	}

      if (indfound != -1)
	{
	  option_index = indfound;
	  optind++;
	  if (nameend.length() != 0)
	    {
	      /* Don't test has_arg with >, because some C compilers don't
		 allow it to be used on enums.  */
	      if (longOptionHasArg[indfound] != GetOpt.NO_ARG)
		optarg = nameend;
	      else
		{
		  nextchar = null;
	          String s =new Character((char)inputArgs[optind-1].charAt(0)).toString();
		  if (inputArgs[optind - 1].charAt(1) == '-')
		    /* --option */
		    throw new GetOptException("option `--" + 
		      longOptionName[indfound] + "' doesn't allow an argument");
		   else
		    /* +option or -option */
		    throw new GetOptException("option `" + s + 
		      longOptionName[indfound] + "' doesn't allow an argument");
		}
	    }
	  else if (longOptionHasArg[indfound] == GetOpt.REQ_ARG)
	    {
	      if (optind < inputArgs.length)
		optarg = inputArgs[optind++];
	      else
		{
		  nextchar = null;
		  throw new GetOptException("option `" + inputArgs[optind-1] + "' requires an argument");
		}
	    }
	  nextchar = null;
	  longind = indfound;
	  // return 0; in C code, if flag != null
	  return longOptionValue[indfound];
	}

      /* Can't find it as a long option.  If this is not getopt_longOnly,
	 or the option starts with '--' or is not a valid short
	 option, then it's an error.
	 Otherwise interpret it as a short option.  */
      if (!longOnly || inputArgs[optind].charAt(1) == '-'
	  || my_index (shortOption, nextchar.charAt(0)) == null)
	{
	  nextchar = null;
	  optind++;
	  String s =new Character((char)inputArgs[optind].charAt(0)).toString();
	  if (inputArgs[optind].charAt(1) == '-')
		/* --option */
            throw new GetOptException("unrecognized option `--" + nextchar + "'");
	  else
		/* +option or -option */
	    throw new GetOptException("unrecognized option `" +
		 s + nextchar + "'");
	}
    }

    /* Look at and handle the next short option-character.  */

    int c = nextchar.charAt(0);
    nextchar = nextchar.substring(1);
    String temp = my_index (shortOption, c);

//System.out.println("zxk2: " + c + " " + temp);
    /* Increment `optind' when we start to process its last character.  */
    if (nextchar.length() == 0)
      ++optind;

    if (temp == null || c == ':')
      {
	String s = new Character((char)c).toString();
	if (posixly_correct != null)
	      /* 1003.2 specifies the format of this message.  */
	  throw new GetOptException("illegal option -- " + s);
	else
	  throw new GetOptException("invalid option -- " + s);
      }
    /* Convenience. Treat POSIX -W foo same as long option --foo */
    if (temp.length() > 1 && temp.charAt(0) == 'W' && temp.charAt(1) == ';')
      {
	String nameend;
	int p;
	int indfound = 0;
	boolean exact = false;
	boolean ambig = false;
	int option_index;

	/* This is an option that requires an argument.  */
	if (nextchar.length() != 0)
	  {
	    optarg = nextchar;
	    /* If we end this ARGV-element by taking the rest as an arg,
	       we must advance to the next element now.  */
	    optind++;
	  }
	else if (optind == inputArgs.length)
	  {
	    String s = new Character((char)c).toString();
		/* 1003.2 specifies the format of this message.  */
            throw new GetOptException("option requires an argument -- " + s);
	  }
	else
	  /* We already incremented `optind' once;
	     increment it again when taking next ARGV-elt as argument.  */
	  optarg = inputArgs[optind++];

	/* optarg is now the argument, see if it's in the
	   table of longopts.  */

        String newName;
        i = nextchar.indexOf('='); 
        if (i == -1)
        {
          nameend = new String("");
    	  newName = new String("");
        }
        else
        {
          nameend = nextchar.substring(i+1);
	  newName = nextchar.substring(0,i);
        }

	/* Test all long options for either exact match
	   or abbreviated matches.  */
        int long_len = longOptionName.length;
        for (option_index=0; option_index<long_len; option_index++)
        {
          p = option_index;
	  if (longOptionName[p].startsWith(newName))
	    {
	      if (i == longOptionName[p].length())
		{
		  /* Exact match found.  */
		  indfound = p;
		  exact = true;
		  break;
		}
	      else if (p == -1)
		{
		  /* First nonexact match found.  */
		  indfound = p;
		}
	      else
		/* Second or later nonexact match found.  */
		ambig = true;
	    }
	}
	if (ambig && !exact)
	  {
	    nextchar = null;
	    optind++;
	    throw new GetOptException("option `-W " + inputArgs[optind] + "' is ambiguous");
	  }
	if (indfound != -1)
	  {
	    option_index = indfound;
	    if (nameend.length() != 0)
	      {
		/* Don't test has_arg with >, because some C compilers don't
		   allow it to be used on enums.  */
		if (longOptionHasArg[indfound] != GetOpt.NO_ARG)
		  optarg = nameend;
		else
		  {
		    nextchar = null;
		    throw new GetOptException("option `-W " + 
		      longOptionName[indfound] + "' doesn't allow an argument");
		  }
	      }
	    else if (longOptionHasArg[indfound] == GetOpt.REQ_ARG)
	      {
		if (optind < inputArgs.length)
		  optarg = inputArgs[optind++];
		else
		  {
		    nextchar = null;
		    throw new GetOptException("option `" + inputArgs[optind-1] + "' requires an argument");
		  }
	      }
	    nextchar = null;
	    longind = indfound;
		// return 0; in C code if flag != NULL
	    return longOptionValue[indfound];
	  }
	  nextchar = null;
	  return 'W';	/* Let the application handle it.   */
      }
    if (temp.length() > 1 && temp.charAt(1) == ':')
      {
	if (temp.length() > 2 && temp.charAt(2) == ':')
	  {
	    /* This is an option that accepts an argument optionally.  */
	    if (nextchar.length() != 0)
	      {
		optarg = nextchar;
		optind++;
	      }
	    else
	      optarg = null;
	    nextchar = null;
	  }
	else
	  {
	    /* This is an option that requires an argument.  */
	    if (nextchar.length() != 0)
	      {
		optarg = nextchar;
		/* If we end this ARGV-element by taking the rest as an arg,
		   we must advance to the next element now.  */
		optind++;
	      }
	    else if (optind == inputArgs.length)
	    {
	       String s = new Character((char)c).toString();
		    /* 1003.2 specifies the format of this message.  */
	       throw new GetOptException("option requires an argument -- " + s);
	    }
	    else
	      /* We already incremented `optind' once;
		 increment it again when taking next ARGV-elt as argument.  */
	      optarg = inputArgs[optind++];
	    nextchar = null;
	  }
      }
    return c;
  }

  /** Handle permutation of arguments. 
   Describe the part of ARGV that contains non-options that have
   been skipped.  `first_nonopt' is the index in ARGV of the first of them;
   `last_nonopt' is the index after the last of them.  */
  private int first_nonopt;
  private int last_nonopt;

  /** Exchange two adjacent subsequences of ARGV.
   One subsequence is elements [first_nonopt,last_nonopt)
   which contains all the non-options that have been skipped so far.
   The other is elements [last_nonopt,optind), which contains all
   the options processed since those non-options were skipped.

   `first_nonopt' and `last_nonopt' are relocated so that they describe
   the new indices of the non-options in ARGV after they are moved.  
   */
  private void exchange ()
  {
    int i;
    int bottom = first_nonopt;
    int middle = last_nonopt;
    String tem;
    int top = optind;

    /* Exchange the shorter segment with the far end of the longer segment.
     That puts the shorter segment into the right place.
     It leaves the longer segment in the right place overall,
     but it consists of two parts that need to be swapped next.  */

    while (top > middle && middle > bottom)
    {
      if (top - middle > middle - bottom)
	{
	  /* Bottom segment is the short one.  */
	  int len = middle - bottom;

	  /* Swap it with the top part of the top segment.  */
	  for (i = 0; i < len; i++)
	    {
	      tem = inputArgs[bottom + i];
	      inputArgs[bottom + i] = inputArgs[top - (middle - bottom) + i];
	      inputArgs[top - (middle - bottom) + i] = tem;
	    }
	  /* Exclude the moved bottom segment from further swapping.  */
	  top -= len;
	}
      else
	{
	  /* Top segment is the short one.  */
	  int len = top - middle;

	  /* Swap it with the bottom part of the bottom segment.  */
	  for (i = 0; i < len; i++)
	    {
	      tem = inputArgs[bottom + i];
	      inputArgs[bottom + i] = inputArgs[middle + i];
	      inputArgs[middle + i] = tem;
	    }
	  /* Exclude the moved top segment from further swapping.  */
	  bottom += len;
	}
    }

    /* Update records for the slots the non-options now occupy.  */
    first_nonopt += (optind - last_nonopt);
    last_nonopt = optind;
  }

  /** Initialize the internal data when the first call is made.  */
  private void getopt_initialize ()
    throws GetOptException
  {
    if (inputArgs == null)
      throw new GetOptException("No inputArgs");

    switch(optionMode)
    {
      case SHORT_OPT:
        if (shortOption == null)
          throw new GetOptException("No shortOption");
        break;
      case LONG_ONLY_OPT:
        if (longOptionName == null)
          throw new GetOptException("No longOptionName");
        if (longOptionHasArg == null)
          throw new GetOptException("No longOptionHasArg");
        if (longOptionValue == null)
          throw new GetOptException("No longOptionValue");
	if (longOptionName.length != longOptionHasArg.length ||
            longOptionName.length != longOptionValue.length)
          throw new GetOptException("Not the same array length among longOptionName, longOptionHasArg, longOptionValue");
        break;
      case LONG_OPT:
      default:
        if (shortOption == null)
          throw new GetOptException("No shortOption");
        if (longOptionName == null)
          throw new GetOptException("No longOptionName");
        if (longOptionHasArg == null)
          throw new GetOptException("No longOptionHasArg");
        if (longOptionValue == null)
          throw new GetOptException("No longOptionValue");
	if (longOptionName.length != longOptionHasArg.length ||
            longOptionName.length != longOptionValue.length)
          throw new GetOptException("Not the same array length among longOptionName, longOptionHasArg, longOptionValue");
    }

    firstInitial = false;

    /* Start processing options with ARGV-element 1 (since ARGV-element 0
     is the program name); the sequence of previously skipped
     non-option ARGV-elements is empty.  
     In Java, the first one is 0.
     */
    first_nonopt = last_nonopt = optind = longind = 0;

    nextchar = null;

    posixly_correct = System.getProperty ("POSIXLY_CORRECT");

    /* Determine how to handle the ordering of options and nonoptions.  */
    if (shortOption.charAt(0) == '-')
    {
      ordering = RETURN_IN_ORDER;
      shortOption = shortOption.substring(1);
    }
    else if (shortOption.charAt(0) == '+')
    {
      ordering = REQUIRE_ORDER;
      shortOption = shortOption.substring(1);
    }
    else if (posixly_correct != null)
      ordering = REQUIRE_ORDER;
    else
      ordering = PERMUTE;

    return;
  }

  private String my_index(String str, int chr)
  {
    int i = str.indexOf(chr);
    if (i == -1)
      return null;
    else
      return str.substring(i);
  }

  public static void main(String[] args)
  {
    int c;
    int digit_optind = 0;
    GetOpt g = new GetOpt();
    g.setShortOption("abc:d::0123456789");
    g.setInputArgs(args);
    g.setOptionMode(GetOpt.SHORT_OPT);
    int optind = 0;

    try
    {
      while ( (c = g.getopt ()) != GetOpt.EOF)
      {
        optind = g.getOptind();
        int this_option_optind = optind;

        switch (c)
	{
	  case '0':
	  case '1':
	  case '2':
	  case '3':
	  case '4':
	  case '5':
	  case '6':
	  case '7':
	  case '8':
	  case '9':
	    if (digit_optind != 0 && digit_optind != this_option_optind)
	      System.out.println("digits occur in two different argv-elements.");
	    digit_optind = this_option_optind;
	    String s = new Character((char)c).toString();
	    System.out.println("option " + s);
	    break;

	  case 'a':
	    System.out.println("option a");
	    break;

	  case 'b':
	    System.out.println("option b");
	    break;

	  case 'c':
	    System.out.println("option c with value `" + g.getOptarg() + "'");
	    break;

	  case 'd':
	    System.out.println("option d with value `" + g.getOptarg() + "'");
	    break;

	  default:
	    System.out.println("?? getopt returned character code " + c + " ??");
	}
      }
    }
    catch (GetOptException e)
    {
      System.out.println(e.getMessage());
    }

  optind = g.getOptind();
  if (optind < args.length)
    {
      System.out.println("non-option ARGV-elements: ");
      while (optind < args.length)
	System.out.print(args[optind++] + " ");
      System.out.println("");
    }
  if ( args.length == 0)
    System.out.println("Usage: GetOpt -a -b -c xxx -d [xxx] -0123456789");
  System.exit(0);
  }
}
