/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.webservice.server;

// $Id$

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;

import javax.management.MBeanServer;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.jboss.axis.AxisEngine;
import org.jboss.axis.AxisFault;
import org.jboss.axis.ConfigurationException;
import org.jboss.axis.MessageContext;
import org.jboss.axis.description.OperationDesc;
import org.jboss.axis.description.ServiceDesc;
import org.jboss.axis.server.AxisServer;
import org.jboss.axis.transport.http.AxisServlet;
import org.jboss.axis.utils.XMLUtils;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.webservice.AxisServiceMBean;
import org.jboss.webservice.PortComponentInfo;
import org.jboss.webservice.ServiceDeployer;
import org.w3c.dom.Document;

/**
 * The servlet that that is associated with context /ws4ee
 * 
 * It manages the 'Version' service
 *
 * @author Thomas.Diesler@jboss.org
 * @since 09-Feb-2005
 */
public abstract class AbstractServlet extends AxisServlet
{
   // provide logging
   protected final Logger log = Logger.getLogger(AbstractServlet.class);

   /**
    * Get the transport URL
    * <p/>
    * For JSE service endpoints
    * e.g. http://localhost:8080/whatever-url-mapping
    */
   protected String getTransportURL(HttpServletRequest req, String serviceName)
   {
      return req.getRequestURI();
   }

   protected void setupHTMLResponseHeader(HttpServletResponse res, PrintWriter writer)
   {
      writer.println("<head>");
      writer.println("<meta http-equiv='Content-Type content='text/html; charset=iso-8859-1'>");
      writer.println("<title>JBossWS</title>");
      writer.println("<link rel='stylesheet' href='/ws4ee/styles.css'>");
      writer.println("</head>");
   }

   /**
    * This method lists the available services; it is called when there is
    * nothing to execute on a GET
    */
   protected void reportAvailableServices(HttpServletResponse res, PrintWriter writer, HttpServletRequest req)
           throws ConfigurationException, AxisFault
   {
      AxisEngine engine = getEngine();

      res.setContentType("text/html");
      setupHTMLResponseHeader(res, writer);
      writer.println("<h2>And now... Some Services</h2>");
      writer.println("<ul>");

      // Get the JBossWS AxisService
      AxisServiceMBean axisService = null;
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         axisService = (AxisServiceMBean)MBeanProxyExt.create(AxisServiceMBean.class, AxisServiceMBean.OBJECT_NAME, server);
      }
      catch (Exception e)
      {
         throw new IllegalStateException("Cannot obtain: " + AxisServiceMBean.OBJECT_NAME + ": " + e.getMessage());
      }

      Iterator it = engine.getConfig().getDeployedServices();
      while (it.hasNext())
      {
         ServiceDesc sd = (ServiceDesc)it.next();
         String sdName = sd.getName();

         String serviceName = getServiceName(req);
         if (serviceName == null || serviceName.equals(sdName))
         {
            PortComponentInfo pcInfo = axisService.getPortComponentInfo(sdName);
            String serviceURL = null;
            if (pcInfo != null)
            {
               serviceURL = pcInfo.getServiceEndpointURL();
            }
            else
            {
               serviceURL = getWebappBase(req) + "/services/" + sdName;
            }

            StringBuffer sb = new StringBuffer("<li>" + sdName + "<a href='" + serviceURL + "?wsdl'>&nbsp;<i>(wsdl)</i></a></li>");

            ArrayList operations = sd.getOperations();
            if (!operations.isEmpty())
            {
               sb.append("<ul>");
               for (Iterator itOp = operations.iterator(); itOp.hasNext();)
               {
                  OperationDesc desc = (OperationDesc)itOp.next();
                  sb.append("<li>" + desc.getName() + "</li>");
               }
               sb.append("</ul>");
            }
            writer.println(sb.toString());
         }
      }
      writer.println("</ul>");
   }

   /**
    * Handle a wsdl request
    */
   protected void processWsdlRequest(MessageContext msgContext, HttpServletResponse res, PrintWriter writer)
           throws AxisFault
   {
      String serviceURL = (String)msgContext.getProperty(MessageContext.TRANS_URL);

      AxisEngine engine = getEngine();
      engine.generateWSDL(msgContext);

      Document doc = (Document)msgContext.getProperty("WSDL");
      if (doc == null)
         throw new AxisFault("Cannot get wsdl document for service: " + serviceURL);

      res.setContentType("text/xml");
      XMLUtils.DocumentToWriter(doc, writer);
   }

   /**
    * Get an AxisServer with the EngineConfiguration from the JMX AxisService
    */
   public AxisServer getEngine() throws AxisFault
   {
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         axisServer = (AxisServer)server.getAttribute(AxisServiceMBean.OBJECT_NAME, "AxisServer");
      }
      catch (Exception e)
      {
         log.warn("Cannot access AxisService, using default server config");
         axisServer = super.getEngine();
      }

      return axisServer;
   }
   
   /**
    * Get the service name as it is known to Axis
    * For WS4EE service endpoints it is obtained from the generated init parameter in web.xml
    */
   protected String getServiceName(HttpServletRequest req)
   {
      String serviceName = getServletConfig().getInitParameter(ServiceDeployer.INIT_PARAM_SERVICE_ENDPOINT_ID);
      return serviceName;
   }
}
