/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.webservice.encoding.ser;

// $Id$

import org.jboss.axis.description.AttributeDesc;
import org.jboss.axis.description.ElementDesc;
import org.jboss.axis.description.FieldDesc;
import org.jboss.axis.description.TypeDesc;
import org.jboss.axis.utils.BeanPropertyDescriptor;
import org.jboss.logging.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.webservice.deployment.BeanXMLMetaData;
import org.jboss.webservice.deployment.MetaDataRegistry;
import org.jboss.webservice.deployment.BeanXMLMetaData.ElementMetaData;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

/**
 * An Axis TypeDesc that can get additional information from the ws4ee-deployment.xml
 * <p/>
 * Type <typeMapping> element has the same basic structure as defined by the wsdd schema.
 * <p/>
 * A additionally you can have a <typeDesc> child element which more information on every field
 * <p/>
 * Initially the TypeDesc is initalized through introspection from a java type, all fields will be encoded
 * xsd style in a random order.
 * <p/>
 * Here you can specify the name and encoding style for every element.
 * <p/>
 * The <elementOrder> is optional, but when present must contain all fields from the java type. This is
 * usefull when the schema complexType specifies <sequence> instead of <all>.
 * <p/>
 * Here is an example:
 * <p/>
 * <typeMapping
 *    qname='ns1:SequenceStruct2' xmlns:ns1='http://MarshallTest.org/xsd'
 *    type='java:org.jboss.ws4ee.test.marshalltest.rpcenc.SequenceStruct2'
 *    serializer='org.jboss.webservice.encoding.ser.MetaDataBeanSerializerFactory'
 *    deserializer='org.jboss.webservice.encoding.ser.MetaDataBeanDeserializerFactory'
 *    encodingStyle='http://schemas.xmlsoap.org/soap/encoding/'>
 *    <typeDesc>
 *       <elementDesc fieldName="varBase64Binary" xmlName="varBase64Binary" xmlType="xsd:base64Binary"/>
 *       <elementDesc fieldName="varHexBinary" xmlName="varHexBinary" xmlType="xsd:hexBinary"/>
 *       <elementDesc fieldName="varSoapString" xmlName="varSoapString" xmlType="soapenc:string"/>
 *       <elementDesc fieldName="varSoapBoolean" xmlName="varSoapBoolean" xmlType="soapenc:boolean"/>
 *       <elementDesc fieldName="varSoapFloat" xmlName="varSoapFloat" xmlType="soapenc:float"/>
 *       <elementDesc fieldName="varSoapDouble" xmlName="varSoapDouble" xmlType="soapenc:double"/>
 *       <elementDesc fieldName="varSoapDecimal" xmlName="varSoapDecimal" xmlType="soapenc:decimal"/>
 *       <elementDesc fieldName="varSoapInt" xmlName="varSoapInt" xmlType="soapenc:int"/>
 *       <elementDesc fieldName="varSoapShort" xmlName="varSoapShort" xmlType="soapenc:short"/>
 *       <elementDesc fieldName="varSoapByte" xmlName="varSoapByte" xmlType="soapenc:byte"/>
 *       <elementDesc fieldName="varSoapBase64" xmlName="varSoapBase64" xmlType="soapenc:base64"/>
 *       <elementDesc fieldName="varDateTimeArray" xmlName="varDateTimeArray" itemXmlType="xsd:dateTime"/>
 *       <elementOrder>
 *          <element name="varString"/>
 *          <element name="varInteger"/>
 *          <element name="varInt"/>
 *          <element name="varLong"/>
 *          <element name="varShort"/>
 *          <element name="varDecimal"/>
 *          <element name="varFloat"/>
 *          <element name="varDouble"/>
 *          <element name="varBoolean"/>
 *          <element name="varByte"/>
 *          <element name="varQName"/>
 *          <element name="varDateTime"/>
 *          <element name="varSoapString"/>
 *          <element name="varSoapBoolean"/>
 *          <element name="varSoapFloat"/>
 *          <element name="varSoapDouble"/>
 *          <element name="varSoapDecimal"/>
 *          <element name="varSoapInt"/>
 *          <element name="varSoapShort"/>
 *          <element name="varSoapByte"/>
 *          <element name="varBase64Binary"/>
 *          <element name="varHexBinary"/>
 *          <element name="varSoapBase64"/>
 *          <element name="varSequenceStruct"/>
 *          <element name="varDateTimeArray"/>
 *       </elementOrder>
 *    </typeDesc>
 * </typeMapping>
 *
 * @author thomas.diesler@jboss.org
 * @since 18-June-2004
 */
public class MetaDataTypeDesc extends TypeDesc
{
   // provide logging
   private final Logger log = Logger.getLogger(MetaDataTypeDesc.class);

   public MetaDataTypeDesc(Class javaClass, QName xmlType, BeanXMLMetaData metaData)
   {
      super(javaClass);

      if (javaClass == null)
         throw new IllegalArgumentException("javaClass cannot be null");
      if (xmlType == null)
         throw new IllegalArgumentException("xmlType cannot be null");

      // On the client side this should never be null because the ServiceImpl installs the typemappings.
      if (metaData == null)
         metaData = getServerSideMetaData(xmlType);

      try
      {
         if (metaData != null)
         {
            setXmlType(xmlType);
            setupElementDescriptions(metaData);
            setupElementOrder(xmlType, metaData);
         }
      }
      catch (RuntimeException e)
      {
         log.error("Instanciation error", e);
         throw e;
      }
   }

   /**
    * The server side installs type mappings through the WSDD deployment, which constructs
    * the MetaDataBean[Serializer|Deserializer]Factory without metaData.
    *
    * This hack calls back to the AxisServer to obtain the missing metaData from a registry.
    * Ideally we would like to register the serializer/deserializer factories with axis complete
    * with meta data, but I could not figure out how to get to the TypeMappingRegistry at deploy time.
    */
   private BeanXMLMetaData getServerSideMetaData(QName xmlType)
   {
      BeanXMLMetaData metaData = null;
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         ObjectName objectName = new ObjectName("jboss.ws4ee:service=AxisService");
         MetaDataRegistry registry = (MetaDataRegistry)server.getAttribute(objectName, "MetaDataRegistry");
         metaData = registry.getTypeMappingMetaData(xmlType);
      }
      catch (Exception e)
      {
         log.warn("Cannot obtain metaData from AxisService: " + e);
      }
      return metaData;
   }

   private void setupElementDescriptions(BeanXMLMetaData metaData)
   {
      // Setup element descriptions
      Iterator itElDesc = metaData.getElementMetaData();
      while (itElDesc.hasNext())
      {
         BeanXMLMetaData.ElementMetaData elMetaData = (BeanXMLMetaData.ElementMetaData)itElDesc.next();
         String fieldName = elMetaData.getFieldName();
         QName xmlName = elMetaData.getXmlName();
         QName xmlType = elMetaData.getXmlType();
         boolean asAttribute = elMetaData.isAsAttribute();
         boolean asContent = elMetaData.isAsContent();

         FieldDesc fieldDesc = null;
         if (asAttribute)
            fieldDesc = new AttributeDesc();
         else
         {
            fieldDesc = new ElementDesc();
            ((ElementDesc)fieldDesc).setAsContent(asContent);
            ((ElementDesc)fieldDesc).setItemXmlType(elMetaData.getItemXmlType());

            Integer minOccurs = elMetaData.getMinOccurs();
            if (minOccurs != null)
            {
               ((ElementDesc)fieldDesc).setMinOccurs(minOccurs.intValue());
            }
         }

         fieldDesc.setFieldName(fieldName);
         fieldDesc.setXmlName(xmlName);
         fieldDesc.setXmlType(xmlType);
         addFieldDesc(fieldDesc);
      }
   }

   private void setupElementOrder(QName xmlType, BeanXMLMetaData metaData)
   {
      log.debug("setupElementOrder: " + xmlType);
      
      // Setup element order
      Iterator itElementOrder = metaData.getElementOrder().iterator();
      if (itElementOrder.hasNext())
      {
         Map bpdMap = getPropertyDescriptorMap();

         // Remove the properties that are not in the meta data
         ArrayList keys = new ArrayList(bpdMap.keySet());
         ArrayList removedKeys = new ArrayList();
         Iterator it = keys.iterator();
         while (it.hasNext())
         {
            String propName = (String)it.next();
            ElementMetaData elMetaData = metaData.getElementMetaDataByFieldName(propName);
            if ("class".equals(propName) == false && elMetaData == null)
            {
               removedKeys.add(propName);
               bpdMap.remove(propName);
            }
         }
         keys = new ArrayList(bpdMap.keySet());

         if (removedKeys.size() > 0)
         {
            log.warn("Remove unmapped properties: " + removedKeys + " from " + xmlType);
         }
         
         BeanPropertyDescriptor[] bpdArr = new BeanPropertyDescriptor[bpdMap.size()];

         int index = 0;
         if (bpdMap.size() > 0)
         {
            // The first one is fixed
            BeanPropertyDescriptor bpd = (BeanPropertyDescriptor)bpdMap.get("class");
            if (bpd != null)
            {
               bpdArr[index++] = bpd;
               keys.remove("class");
            }

            // copy the BeanPropertyDescriptors according to the meta data elementOrder
            while (itElementOrder.hasNext())
            {
               String fieldName = (String)itElementOrder.next();

               bpd = (BeanPropertyDescriptor)bpdMap.get(fieldName);
               if (bpd != null)
               {
                  bpdArr[index++] = bpd;
                  keys.remove(fieldName);
               }
            }

            // copy the remaining BeanPropertyDescriptors
            it = keys.iterator();
            while (it.hasNext())
            {
               String name = (String)it.next();
               bpd = (BeanPropertyDescriptor)bpdMap.get(name);
               if (bpd != null)
                  bpdArr[index++] = bpd;
            }

            setPropertyDescriptors(bpdArr);
         }
      }
   }
}
