/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.tc5.session;

/**
 * Binds the session id of a session that is currently being manipulated
 * locally to a ThreadLocal so CacheListener can ignore events generated by the
 * activity.
 * 
 * @author Brian Stansberry
 * @version $Id$
 */
class LocalSessionActivity
{
   private static ThreadLocal activeSessions = new ThreadLocal();
   
   private String sessionId;
   private int level = 0;
   
   /**
    * Marks the current thread as actively processing the given session.
    * If the thread has already been so marked, increases a counter
    * so a subsequent call to finishLocalActivity does not remove
    * the association (allowing nested calls).
    * 
    * @param   sessionId. Can be <code>null</code>.
    */
   public static void startLocalActivity(String sessionId)
   {
      LocalSessionActivity current = (LocalSessionActivity) activeSessions.get();
      if (current == null || !(current.sessionId.equals(sessionId)))
      {
         current = new LocalSessionActivity(sessionId);
         activeSessions.set(current);
      }
      current.level++;
   }
   
   /**
    * Marks the completion of activity on a given session.  Should be called
    * once for each invocation of {@link #startLocalActivity(String)}.
    *
    */
   public static void finishLocalActivity()
   {
      LocalSessionActivity current = (LocalSessionActivity) activeSessions.get();
      if (current != null)
      {
         if (--current.level == 0)
            activeSessions.set(null);
      }
   }
   
   public static boolean isLocallyActive(String sessionId)
   {
      boolean result = false;
      LocalSessionActivity active = (LocalSessionActivity) activeSessions.get();
      if (active != null)
         result = active.sessionId.equals(sessionId);
      
      return result;
   }
   
   private LocalSessionActivity(String sessionId) 
   {
      this.sessionId = sessionId;
   }
}
